"""
Scrapfly worker module for distributed scraping.

Provides worker classes for managing Scrapfly client sessions
and tracking agent processing progress.
"""

import uuid
from dataclasses import dataclass, field
from datetime import datetime
from typing import Optional, List

from scrapfly import ScrapflyClient
from loguru import logger

from ..utils.config import settings


@dataclass
class AgentProgress:
    """
    Track progress for an agent's scraping.

    Attributes:
        url: Agent's URL
        rental_urls: List of rental property URLs
        resale_urls: List of resale property URLs
        rental_processed: Whether rental properties have been processed
        resale_processed: Whether resale properties have been processed
        last_updated: Last update timestamp
        content_cached: Whether content has been cached
        has_rental: Whether agent has rental properties
        has_resale: Whether agent has resale properties
    """
    url: str
    rental_urls: List[str] = None
    resale_urls: List[str] = None
    rental_processed: bool = False
    resale_processed: bool = False
    last_updated: datetime = None
    content_cached: bool = False
    has_rental: Optional[bool] = None
    has_resale: Optional[bool] = None

    def __post_init__(self):
        if self.rental_urls is None:
            self.rental_urls = []
        if self.resale_urls is None:
            self.resale_urls = []

    def is_complete(self) -> bool:
        """Check if agent processing is complete."""
        return self.rental_processed and self.resale_processed

    def update_timestamp(self) -> None:
        """Update the last_updated timestamp."""
        self.last_updated = datetime.now()


@dataclass
class ScrapflyWorker:
    """
    Represents a Scrapfly client worker with its API key and current task.

    Attributes:
        key: API key for this worker
        client: Scrapfly client instance
        current_agent: URL of agent currently being processed
        is_busy: Whether worker is currently processing
        session_id: Current session ID
    """
    key: str
    client: ScrapflyClient
    current_agent: Optional[str] = None
    is_busy: bool = False
    session_id: Optional[str] = None

    async def initialize_session(self, target_url: str) -> None:
        """
        Initialize worker's session for a target URL.

        Args:
            target_url: URL that will be scraped
        """
        self.session_id = str(uuid.uuid4())
        logger.info(f"Worker {self.key[-6:]} initialized with session {self.session_id}")

    def start_task(self, agent_url: str) -> None:
        """
        Mark worker as busy with a specific agent.

        Args:
            agent_url: URL of the agent being processed
        """
        self.current_agent = agent_url
        self.is_busy = True

    def complete_task(self) -> None:
        """Mark worker as available."""
        self.current_agent = None
        self.is_busy = False

    @property
    def worker_id(self) -> str:
        """Get a short identifier for this worker."""
        return self.key[-6:]


def create_workers_from_env(max_workers: Optional[int] = None) -> List[ScrapflyWorker]:
    """
    Create workers from environment variables.

    Looks for SCRAPFLY_KEY_1 through SCRAPFLY_KEY_{max_workers}
    in the environment.

    Args:
        max_workers: Maximum number of workers to create.
                    If None, uses settings.max_workers from environment.

    Returns:
        List of initialized ScrapflyWorker instances
    """
    import os

    # Use provided value or fall back to settings
    worker_limit = max_workers if max_workers is not None else settings.max_workers

    workers = []
    for i in range(1, worker_limit + 1):
        key = os.getenv(f"SCRAPFLY_KEY_{i}")
        if key:
            worker = ScrapflyWorker(
                key=key,
                client=ScrapflyClient(key=key)
            )
            workers.append(worker)

    logger.info(f"Created {len(workers)} Scrapfly workers (max configured: {worker_limit})")
    return workers
