"""
Shared session manager for Scrapfly clients.

Key optimization: Instead of 15 clients each fetching cookies independently
(375 credits every 4 hours with ASP), we share a single session across all clients.
"""

import asyncio
import json
from datetime import datetime, timedelta, timezone
from pathlib import Path
from typing import Dict, Optional
from loguru import logger

from ..utils.paths import get_output_dir


class SharedSessionManager:
    """
    Manages shared session state across all Scrapfly clients.

    Previously: 15 clients x 25 ASP credits = 375 credits every 4 hours for cookies
    Now: 1 shared session = 1-25 credits every 4 hours (depending on tier)

    Savings: 93-99%
    """

    def __init__(
        self,
        session_file: Optional[Path] = None,
        refresh_interval_hours: float = 4.0,
    ):
        self.session_file = session_file or (get_output_dir() / "session_state.json")
        self.refresh_interval = timedelta(hours=refresh_interval_hours)

        self._cookies: Dict[str, str] = {}
        self._session_id: Optional[str] = None
        self._last_refresh: Optional[datetime] = None
        self._lock = asyncio.Lock()

        # Load existing session if available
        self._load_session()

    def _load_session(self):
        """Load session state from disk."""
        if self.session_file.exists():
            try:
                with open(self.session_file, "r") as f:
                    data = json.load(f)

                self._cookies = data.get("cookies", {})
                self._session_id = data.get("session_id")

                last_refresh_str = data.get("last_refresh")
                if last_refresh_str:
                    self._last_refresh = datetime.fromisoformat(last_refresh_str)

                logger.info(
                    f"Loaded session state: {len(self._cookies)} cookies, "
                    f"session_id={self._session_id}"
                )
            except Exception as e:
                logger.warning(f"Failed to load session state: {e}")

    def _save_session(self):
        """Save session state to disk."""
        self.session_file.parent.mkdir(parents=True, exist_ok=True)

        data = {
            "cookies": self._cookies,
            "session_id": self._session_id,
            "last_refresh": (
                self._last_refresh.isoformat() if self._last_refresh else None
            ),
        }

        with open(self.session_file, "w") as f:
            json.dump(data, f, indent=2)

        logger.debug("Saved session state to disk")

    @property
    def needs_refresh(self) -> bool:
        """Check if session needs to be refreshed."""
        if not self._last_refresh:
            return True

        age = datetime.now(timezone.utc) - self._last_refresh
        return age >= self.refresh_interval

    @property
    def cookies(self) -> Dict[str, str]:
        """Get current cookies."""
        return self._cookies.copy()

    @property
    def session_id(self) -> Optional[str]:
        """Get current session ID."""
        return self._session_id

    async def update(
        self,
        cookies: Dict[str, str],
        session_id: Optional[str] = None,
    ):
        """
        Update session state.

        Args:
            cookies: New cookies to store
            session_id: Optional new session ID
        """
        async with self._lock:
            self._cookies.update(cookies)
            if session_id:
                self._session_id = session_id
            self._last_refresh = datetime.now(timezone.utc)
            self._save_session()

            logger.info(f"Updated session: {len(self._cookies)} cookies")

    async def clear(self):
        """Clear session state."""
        async with self._lock:
            self._cookies = {}
            self._session_id = None
            self._last_refresh = None
            if self.session_file.exists():
                self.session_file.unlink()
            logger.info("Cleared session state")

    def get_age_seconds(self) -> Optional[float]:
        """Get session age in seconds."""
        if not self._last_refresh:
            return None
        age = datetime.now(timezone.utc) - self._last_refresh
        return age.total_seconds()

    def get_status(self) -> dict:
        """Get session status for debugging."""
        return {
            "has_cookies": bool(self._cookies),
            "cookie_count": len(self._cookies),
            "session_id": self._session_id,
            "last_refresh": (
                self._last_refresh.isoformat() if self._last_refresh else None
            ),
            "age_seconds": self.get_age_seconds(),
            "needs_refresh": self.needs_refresh,
        }
