"""Preset management for Idealista scraper configurations.

Provides functions to save, load, list, and delete scraper configuration presets.
"""
from __future__ import annotations

import json
from pathlib import Path
from typing import Optional

from loguru import logger

from idealista_scraper.cli.config import ScraperConfig

# Directory for storing presets
PRESETS_DIR = Path.home() / ".idealista-scraper"
PRESETS_FILE = PRESETS_DIR / "presets.json"


def _ensure_presets_dir() -> None:
    """Ensure the presets directory exists."""
    if not PRESETS_DIR.exists():
        PRESETS_DIR.mkdir(parents=True, exist_ok=True)
        logger.debug(f"Created presets directory: {PRESETS_DIR}")


def _load_all_presets() -> dict[str, dict]:
    """Load all presets from disk.

    Returns:
        Dictionary mapping preset names to configuration dictionaries
    """
    if not PRESETS_FILE.exists():
        return {}

    try:
        with open(PRESETS_FILE, "r", encoding="utf-8") as f:
            data = json.load(f)
            return data if isinstance(data, dict) else {}
    except (json.JSONDecodeError, OSError) as e:
        logger.error(f"Failed to load presets file: {e}")
        return {}


def _save_all_presets(presets: dict[str, dict]) -> None:
    """Save all presets to disk.

    Args:
        presets: Dictionary mapping preset names to configuration dictionaries
    """
    _ensure_presets_dir()

    try:
        with open(PRESETS_FILE, "w", encoding="utf-8") as f:
            json.dump(presets, f, indent=2, ensure_ascii=False)
        logger.debug(f"Saved {len(presets)} preset(s) to {PRESETS_FILE}")
    except OSError as e:
        logger.error(f"Failed to save presets file: {e}")
        raise


def save_preset(name: str, config: ScraperConfig) -> None:
    """Save a configuration preset.

    Args:
        name: Name for the preset
        config: ScraperConfig instance to save

    Raises:
        OSError: If unable to write to presets file
    """
    if not name or not name.strip():
        raise ValueError("Preset name cannot be empty")

    presets = _load_all_presets()
    presets[name] = config.to_dict()
    _save_all_presets(presets)

    logger.info(f"Saved preset '{name}'")


def load_preset(name: str) -> Optional[ScraperConfig]:
    """Load a configuration preset by name.

    Args:
        name: Name of the preset to load

    Returns:
        ScraperConfig instance if found, None otherwise
    """
    presets = _load_all_presets()

    if name not in presets:
        logger.warning(f"Preset '{name}' not found")
        return None

    try:
        config = ScraperConfig.from_dict(presets[name])
        logger.debug(f"Loaded preset '{name}'")
        return config
    except (KeyError, ValueError) as e:
        logger.error(f"Failed to parse preset '{name}': {e}")
        return None


def list_presets() -> list[str]:
    """Get a list of all saved preset names.

    Returns:
        List of preset names, sorted alphabetically
    """
    presets = _load_all_presets()
    return sorted(presets.keys())


def delete_preset(name: str) -> bool:
    """Delete a preset by name.

    Args:
        name: Name of the preset to delete

    Returns:
        True if preset was deleted, False if it didn't exist
    """
    presets = _load_all_presets()

    if name not in presets:
        logger.warning(f"Preset '{name}' not found, cannot delete")
        return False

    del presets[name]
    _save_all_presets(presets)

    logger.info(f"Deleted preset '{name}'")
    return True


def get_preset_info(name: str) -> Optional[dict]:
    """Get preset information without loading it as a ScraperConfig.

    Args:
        name: Name of the preset

    Returns:
        Dictionary with preset data if found, None otherwise
    """
    presets = _load_all_presets()
    return presets.get(name)
