"""Interactive CLI wizard for configuring Idealista scraper runs.

Provides a user-friendly interactive interface for building scraper configurations.
"""
from __future__ import annotations

from typing import Optional

from loguru import logger
from rich.console import Console
from rich.panel import Panel
from rich.prompt import Confirm, IntPrompt, Prompt

from idealista_scraper.cli.config import PropertyType, RegionSelection, ScraperConfig
from idealista_scraper.cli.presets import save_preset
from idealista_scraper.scraping.regions import list_regions


class InteractiveWizard:
    """Interactive wizard for building scraper configurations.

    Guides users through selecting country, regions, property type, and other
    configuration options with a friendly CLI interface.
    """

    def __init__(self, console: Optional[Console] = None) -> None:
        """Initialize the interactive wizard.

        Args:
            console: Rich Console instance. Creates one if not provided.
        """
        self.console = console or Console()

    def _show_welcome(self) -> None:
        """Display welcome banner."""
        self.console.print(
            Panel(
                "[bold cyan]Idealista Scraper Configuration Wizard[/bold cyan]\n\n"
                "This wizard will help you configure your scraping session.\n"
                "You'll be able to save your configuration as a preset for future use.",
                border_style="cyan",
                padding=(1, 2),
            )
        )
        self.console.print()

    def _select_country(self) -> str:
        """Select target country.

        Returns:
            Country name (spain or portugal)
        """
        self.console.print("[bold]Step 1: Select Country[/bold]\n")
        self.console.print("  1. Spain")
        self.console.print("  2. Portugal")
        self.console.print()

        while True:
            choice = Prompt.ask(
                "[cyan]Select country[/cyan]",
                choices=["1", "2"],
                default="1",
            )

            if choice == "1":
                return "spain"
            elif choice == "2":
                return "portugal"

    async def _select_regions(self, country: str) -> RegionSelection:
        """Select regions to scrape.

        Args:
            country: Country name

        Returns:
            RegionSelection with selected regions
        """
        self.console.print("\n[bold]Step 2: Select Regions[/bold]\n")
        self.console.print(f"[cyan]Fetching available regions for {country.title()}...[/cyan]\n")

        try:
            regions = await list_regions(country_name=country)

            if not regions:
                self.console.print("[yellow]No regions found, using all regions[/yellow]")
                return RegionSelection(all_regions=True)

            self.console.print(f"[green]Found {len(regions)} regions[/green]\n")
            self.console.print("  0. [bold]All regions[/bold]")

            for idx, region in enumerate(regions, start=1):
                count_str = f" ({region.property_count:,} listings)" if region.property_count else ""
                self.console.print(f"  {idx}. {region.name}{count_str}")

            self.console.print()

            # Get selections
            selections = Prompt.ask(
                "[cyan]Select regions (comma-separated numbers, or 0 for all)[/cyan]",
                default="0",
            )

            # Parse selections
            selected_indices = [s.strip() for s in selections.split(",")]

            # Check for "all regions" option
            if "0" in selected_indices:
                return RegionSelection(all_regions=True)

            # Build region selection
            codes = []
            names = []

            for idx_str in selected_indices:
                try:
                    idx = int(idx_str)
                    if 1 <= idx <= len(regions):
                        region = regions[idx - 1]
                        codes.append(region.code or region.slug)
                        names.append(region.name)
                    else:
                        self.console.print(f"[yellow]Skipping invalid index: {idx}[/yellow]")
                except ValueError:
                    self.console.print(f"[yellow]Skipping invalid input: {idx_str}[/yellow]")

            if not codes:
                self.console.print("[yellow]No valid regions selected, using all regions[/yellow]")
                return RegionSelection(all_regions=True)

            self.console.print(f"\n[green]Selected {len(names)} region(s): {', '.join(names)}[/green]")

            return RegionSelection(codes=codes, names=names, all_regions=False)

        except Exception as e:
            logger.error(f"Failed to fetch regions: {e}")
            self.console.print(f"[yellow]Could not fetch regions: {e}[/yellow]")
            self.console.print("[yellow]Defaulting to all regions[/yellow]")
            return RegionSelection(all_regions=True)

    def _select_property_type(self) -> PropertyType:
        """Select property type.

        Returns:
            PropertyType enum value
        """
        self.console.print("\n[bold]Step 3: Select Property Type[/bold]\n")
        self.console.print("  1. Rental properties")
        self.console.print("  2. Properties for sale")
        self.console.print()

        choice = Prompt.ask(
            "[cyan]Select property type[/cyan]",
            choices=["1", "2"],
            default="1",
        )

        return PropertyType.rental if choice == "1" else PropertyType.sale

    def _get_pages_per_region(self) -> int:
        """Get number of pages to scrape per region. 0 = all pages.

        Returns:
            Number of pages (0 means scrape all available pages)
        """
        self.console.print("\n[bold]Step 4: Pages Per Region[/bold]\n")
        self.console.print("Enter 0 to scrape all available pages.\n")

        pages = IntPrompt.ask(
            "[cyan]How many pages to scrape per region? (0 = all)[/cyan]",
            default=0,
        )

        return max(0, pages)  # 0 means all pages

    def _select_pipeline_steps(self) -> list[str]:
        """Select pipeline steps to execute.

        Returns:
            List of selected step names
        """
        self.console.print("\n[bold]Step 5: Select Pipeline Steps[/bold]\n")

        available_steps = [
            ("discover_agents", "Discover and scrape agent data"),
            ("scrape", "Scrape property listings"),
            ("transform", "Transform data to MongoDB format"),
            ("clean", "Clean and validate data"),
        ]

        self.console.print("Available pipeline steps:\n")
        for idx, (step_name, description) in enumerate(available_steps, start=1):
            self.console.print(f"  {idx}. {step_name} - {description}")

        self.console.print("\n  0. [bold]All steps[/bold]")
        self.console.print()

        selections = Prompt.ask(
            "[cyan]Select steps (comma-separated numbers, or 0 for all)[/cyan]",
            default="0",
        )

        # Parse selections
        selected_indices = [s.strip() for s in selections.split(",")]

        # Check for "all steps" option
        if "0" in selected_indices:
            return [step_name for step_name, _ in available_steps]

        # Build step list
        steps = []
        for idx_str in selected_indices:
            try:
                idx = int(idx_str)
                if 1 <= idx <= len(available_steps):
                    steps.append(available_steps[idx - 1][0])
                else:
                    self.console.print(f"[yellow]Skipping invalid index: {idx}[/yellow]")
            except ValueError:
                self.console.print(f"[yellow]Skipping invalid input: {idx_str}[/yellow]")

        if not steps:
            self.console.print("[yellow]No steps selected, using all steps[/yellow]")
            return [step_name for step_name, _ in available_steps]

        self.console.print(f"\n[green]Selected {len(steps)} step(s): {', '.join(steps)}[/green]")

        return steps

    def _show_summary(self, config: ScraperConfig) -> None:
        """Display configuration summary.

        Args:
            config: ScraperConfig to display
        """
        region_info = "All regions" if config.regions and config.regions.all_regions else (
            f"{len(config.regions.names)} region(s): {', '.join(config.regions.names)}"
            if config.regions and config.regions.names
            else "All regions"
        )

        summary_text = (
            f"[bold]Country:[/bold] {config.country.title()}\n"
            f"[bold]Regions:[/bold] {region_info}\n"
            f"[bold]Property Type:[/bold] {config.property_type.value.title()}\n"
            f"[bold]Pages per Region:[/bold] {config.pages_per_region}\n"
            f"[bold]Max Concurrent Requests:[/bold] {config.max_concurrent}\n"
            f"[bold]Pipeline Steps:[/bold] {', '.join(config.pipeline_steps)}\n"
        )

        if config.s3_bucket:
            summary_text += f"[bold]S3 Bucket:[/bold] {config.s3_bucket}\n"

        self.console.print()
        self.console.print(
            Panel(
                summary_text,
                title="Configuration Summary",
                border_style="cyan",
                padding=(1, 2),
            )
        )

    async def run(self) -> ScraperConfig:
        """Run the interactive wizard.

        Returns:
            Configured ScraperConfig instance

        Raises:
            KeyboardInterrupt: If user cancels the wizard
        """
        try:
            # Welcome
            self._show_welcome()

            # Step 1: Country
            country = self._select_country()

            # Step 2: Regions
            regions = await self._select_regions(country)

            # Step 3: Property Type
            property_type = self._select_property_type()

            # Step 4: Pages
            pages_per_region = self._get_pages_per_region()

            # Step 5: Pipeline Steps
            pipeline_steps = self._select_pipeline_steps()

            # Optional: S3 bucket
            self.console.print("\n[bold]Step 6: S3 Upload (Optional)[/bold]\n")
            upload_to_s3 = Confirm.ask(
                "[cyan]Do you want to upload images to S3?[/cyan]",
                default=False,
            )

            s3_bucket = None
            if upload_to_s3:
                s3_bucket = Prompt.ask(
                    "[cyan]Enter S3 bucket name[/cyan]",
                    default="",
                )
                if not s3_bucket:
                    s3_bucket = None

            # Build configuration
            config = ScraperConfig(
                country=country,
                regions=regions,
                property_type=property_type,
                pages_per_region=pages_per_region,
                max_concurrent=50,
                pipeline_steps=pipeline_steps,
                s3_bucket=s3_bucket,
            )

            # Show summary
            self._show_summary(config)

            # Confirm
            self.console.print()
            confirmed = Confirm.ask(
                "[cyan]Does this configuration look correct?[/cyan]",
                default=True,
            )

            if not confirmed:
                self.console.print("\n[yellow]Configuration cancelled[/yellow]")
                raise KeyboardInterrupt("User cancelled configuration")

            # Always ask to save as preset
            self.console.print()
            save_as_preset = Confirm.ask(
                "[cyan]Would you like to save this configuration as a preset?[/cyan]",
                default=True,
            )

            if save_as_preset:
                preset_name = Prompt.ask(
                    "[cyan]Enter a name for this preset[/cyan]",
                    default="default",
                )

                try:
                    save_preset(preset_name, config)
                    self.console.print(f"\n[green]Saved configuration as preset '{preset_name}'[/green]")
                except Exception as e:
                    self.console.print(f"\n[yellow]Failed to save preset: {e}[/yellow]")

            self.console.print("\n[green]Configuration complete![/green]\n")

            return config

        except KeyboardInterrupt:
            self.console.print("\n\n[yellow]Configuration cancelled by user[/yellow]")
            raise
