"""Configuration dataclasses for the Idealista scraper CLI.

Provides structured configuration for scraper runs with serialization support.
"""
from __future__ import annotations

from dataclasses import dataclass, field
from enum import Enum
from typing import Optional


class PropertyType(str, Enum):
    """Property listing types."""

    rental = "rental"
    sale = "sale"


@dataclass
class RegionSelection:
    """Represents selected regions for scraping."""

    codes: list[str] = field(default_factory=list)
    names: list[str] = field(default_factory=list)
    all_regions: bool = False

    def to_dict(self) -> dict:
        """Convert to dictionary for serialization."""
        return {
            "codes": self.codes,
            "names": self.names,
            "all_regions": self.all_regions,
        }

    @classmethod
    def from_dict(cls, data: dict) -> RegionSelection:
        """Create from dictionary."""
        return cls(
            codes=data.get("codes", []),
            names=data.get("names", []),
            all_regions=data.get("all_regions", False),
        )


@dataclass
class ScraperConfig:
    """Configuration for a scraper run.

    Attributes:
        country: Target country (spain, portugal)
        regions: Selected regions for scraping
        property_type: Type of properties to scrape
        pages_per_region: Number of pages to scrape per region
        max_concurrent: Maximum concurrent requests
        pipeline_steps: List of pipeline steps to execute
        s3_bucket: Optional S3 bucket for image uploads
    """

    country: str = "spain"
    regions: Optional[RegionSelection] = None
    property_type: PropertyType = PropertyType.rental
    pages_per_region: int = 10
    max_concurrent: int = 50
    pipeline_steps: list[str] = field(
        default_factory=lambda: ["discover_agents", "scrape", "transform", "clean"]
    )
    s3_bucket: Optional[str] = None

    def __post_init__(self) -> None:
        """Initialize default regions if not provided."""
        if self.regions is None:
            self.regions = RegionSelection()

        # Ensure property_type is a PropertyType enum
        if isinstance(self.property_type, str):
            self.property_type = PropertyType(self.property_type)

    def to_dict(self) -> dict:
        """Convert to dictionary for serialization.

        Returns:
            Dictionary representation of the config
        """
        return {
            "country": self.country,
            "regions": self.regions.to_dict() if self.regions else None,
            "property_type": self.property_type.value,
            "pages_per_region": self.pages_per_region,
            "max_concurrent": self.max_concurrent,
            "pipeline_steps": self.pipeline_steps,
            "s3_bucket": self.s3_bucket,
        }

    @classmethod
    def from_dict(cls, data: dict) -> ScraperConfig:
        """Create config from dictionary.

        Args:
            data: Dictionary containing config data

        Returns:
            ScraperConfig instance
        """
        regions_data = data.get("regions")
        regions = RegionSelection.from_dict(regions_data) if regions_data else None

        return cls(
            country=data.get("country", "spain"),
            regions=regions,
            property_type=PropertyType(data.get("property_type", "rental")),
            pages_per_region=data.get("pages_per_region", 10),
            max_concurrent=data.get("max_concurrent", 50),
            pipeline_steps=data.get("pipeline_steps", ["discover_agents", "scrape", "transform", "clean"]),
            s3_bucket=data.get("s3_bucket"),
        )
