"""
Centralized cache manager with LRU in-memory cache and disk persistence.

Features:
- LRU (Least Recently Used) eviction policy
- Configurable cache size
- Hash-based disk persistence for O(1) lookups
- Thread-safe operations
- TTL (Time-To-Live) support
"""

import hashlib
import json
import logging
import os
import pickle
import threading
import time
from collections import OrderedDict
from pathlib import Path
from typing import Any, Optional, TypeVar, Generic

logger = logging.getLogger(__name__)

T = TypeVar('T')


class CacheEntry(Generic[T]):
    """Represents a cache entry with value and metadata."""

    __slots__ = ('value', 'created_at', 'ttl', 'access_count')

    def __init__(self, value: T, ttl: Optional[float] = None):
        self.value = value
        self.created_at = time.time()
        self.ttl = ttl
        self.access_count = 0

    def is_expired(self) -> bool:
        """Check if the entry has expired based on TTL."""
        if self.ttl is None:
            return False
        return (time.time() - self.created_at) > self.ttl

    def touch(self) -> None:
        """Update access count."""
        self.access_count += 1


class CacheManager:
    """
    LRU cache manager with disk persistence.

    Uses OrderedDict for O(1) LRU operations and hash-based
    file naming for O(1) disk lookups.

    Args:
        max_size: Maximum number of items in memory cache (default: 1000)
        cache_dir: Directory for disk persistence (default: .cache)
        enable_disk: Enable disk persistence (default: True)
        default_ttl: Default TTL in seconds (default: None, no expiration)

    Example:
        cache = CacheManager(max_size=500)
        cache.set("my_key", {"data": "value"})
        result = cache.get("my_key")
        if cache.has("my_key"):
            print("Key exists!")
    """

    def __init__(
        self,
        max_size: int = 1000,
        cache_dir: Optional[str] = None,
        enable_disk: bool = True,
        default_ttl: Optional[float] = None
    ):
        self._max_size = max_size
        self._cache: OrderedDict[str, CacheEntry] = OrderedDict()
        self._lock = threading.RLock()
        self._enable_disk = enable_disk
        self._default_ttl = default_ttl
        self._stats = {
            'hits': 0,
            'misses': 0,
            'evictions': 0,
            'disk_reads': 0,
            'disk_writes': 0
        }

        # Set up cache directory
        if cache_dir is None:
            base_dir = Path(__file__).parent.parent.parent
            self._cache_dir = base_dir / '.cache'
        else:
            self._cache_dir = Path(cache_dir)

        if self._enable_disk:
            self._cache_dir.mkdir(parents=True, exist_ok=True)
            logger.info(f"Cache directory initialized: {self._cache_dir}")

    def _generate_hash(self, key: str) -> str:
        """Generate a hash for the given key for O(1) disk lookups."""
        return hashlib.sha256(key.encode('utf-8')).hexdigest()[:16]

    def _get_disk_path(self, key: str) -> Path:
        """Get the disk path for a given key using hash-based naming."""
        key_hash = self._generate_hash(key)
        # Use first 2 chars as subdirectory for better file distribution
        subdir = key_hash[:2]
        return self._cache_dir / subdir / f"{key_hash}.cache"

    def _save_to_disk(self, key: str, entry: CacheEntry) -> bool:
        """Persist cache entry to disk."""
        if not self._enable_disk:
            return False

        try:
            disk_path = self._get_disk_path(key)
            disk_path.parent.mkdir(parents=True, exist_ok=True)

            data = {
                'key': key,
                'value': entry.value,
                'created_at': entry.created_at,
                'ttl': entry.ttl
            }

            with open(disk_path, 'wb') as f:
                pickle.dump(data, f)

            self._stats['disk_writes'] += 1
            return True
        except Exception as e:
            logger.warning(f"Failed to save to disk: {e}")
            return False

    def _load_from_disk(self, key: str) -> Optional[CacheEntry]:
        """Load cache entry from disk."""
        if not self._enable_disk:
            return None

        try:
            disk_path = self._get_disk_path(key)
            if not disk_path.exists():
                return None

            with open(disk_path, 'rb') as f:
                data = pickle.load(f)

            # Verify key matches (hash collision check)
            if data.get('key') != key:
                return None

            entry = CacheEntry(data['value'], data.get('ttl'))
            entry.created_at = data.get('created_at', time.time())

            # Check if expired
            if entry.is_expired():
                disk_path.unlink(missing_ok=True)
                return None

            self._stats['disk_reads'] += 1
            return entry
        except Exception as e:
            logger.warning(f"Failed to load from disk: {e}")
            return None

    def _delete_from_disk(self, key: str) -> bool:
        """Delete cache entry from disk."""
        if not self._enable_disk:
            return False

        try:
            disk_path = self._get_disk_path(key)
            if disk_path.exists():
                disk_path.unlink()
                return True
            return False
        except Exception as e:
            logger.warning(f"Failed to delete from disk: {e}")
            return False

    def _evict_lru(self) -> None:
        """Evict the least recently used item from memory cache."""
        if len(self._cache) >= self._max_size:
            # Pop the first (oldest) item
            evicted_key, evicted_entry = self._cache.popitem(last=False)
            self._stats['evictions'] += 1
            # Keep on disk if enabled
            if self._enable_disk:
                self._save_to_disk(evicted_key, evicted_entry)
            logger.debug(f"Evicted LRU entry: {evicted_key[:20]}...")

    def get(self, key: str, default: Any = None) -> Any:
        """
        Get a value from the cache.

        Args:
            key: The cache key
            default: Default value if key not found

        Returns:
            The cached value or default
        """
        with self._lock:
            # Check memory cache first
            if key in self._cache:
                entry = self._cache[key]

                # Check if expired
                if entry.is_expired():
                    del self._cache[key]
                    self._delete_from_disk(key)
                    self._stats['misses'] += 1
                    return default

                # Move to end (most recently used)
                self._cache.move_to_end(key)
                entry.touch()
                self._stats['hits'] += 1
                return entry.value

            # Try disk cache
            entry = self._load_from_disk(key)
            if entry is not None:
                # Promote to memory cache
                self._evict_lru()
                self._cache[key] = entry
                self._stats['hits'] += 1
                return entry.value

            self._stats['misses'] += 1
            return default

    def set(
        self,
        key: str,
        value: Any,
        ttl: Optional[float] = None,
        persist: bool = True
    ) -> None:
        """
        Set a value in the cache.

        Args:
            key: The cache key
            value: The value to cache
            ttl: Optional TTL in seconds (overrides default)
            persist: Whether to persist to disk (default: True)
        """
        with self._lock:
            effective_ttl = ttl if ttl is not None else self._default_ttl
            entry = CacheEntry(value, effective_ttl)

            # Remove existing entry if present
            if key in self._cache:
                del self._cache[key]

            # Evict if necessary
            self._evict_lru()

            # Add to memory cache
            self._cache[key] = entry

            # Persist to disk if enabled
            if persist and self._enable_disk:
                self._save_to_disk(key, entry)

    def has(self, key: str) -> bool:
        """
        Check if a key exists in the cache (memory or disk).

        Args:
            key: The cache key

        Returns:
            True if key exists and is not expired
        """
        with self._lock:
            # Check memory first
            if key in self._cache:
                entry = self._cache[key]
                if not entry.is_expired():
                    return True
                # Expired, clean up
                del self._cache[key]
                self._delete_from_disk(key)
                return False

            # Check disk
            if self._enable_disk:
                entry = self._load_from_disk(key)
                return entry is not None

            return False

    def delete(self, key: str) -> bool:
        """
        Delete a key from the cache.

        Args:
            key: The cache key

        Returns:
            True if the key was deleted
        """
        with self._lock:
            deleted = False

            if key in self._cache:
                del self._cache[key]
                deleted = True

            if self._enable_disk:
                disk_deleted = self._delete_from_disk(key)
                deleted = deleted or disk_deleted

            return deleted

    def clear(self, memory_only: bool = False) -> None:
        """
        Clear the cache.

        Args:
            memory_only: If True, only clear memory cache (keep disk)
        """
        with self._lock:
            self._cache.clear()

            if not memory_only and self._enable_disk:
                import shutil
                try:
                    shutil.rmtree(self._cache_dir)
                    self._cache_dir.mkdir(parents=True, exist_ok=True)
                except Exception as e:
                    logger.warning(f"Failed to clear disk cache: {e}")

            logger.info("Cache cleared")

    def get_stats(self) -> dict:
        """Get cache statistics."""
        with self._lock:
            total_requests = self._stats['hits'] + self._stats['misses']
            hit_rate = (
                self._stats['hits'] / total_requests
                if total_requests > 0 else 0.0
            )

            return {
                **self._stats,
                'memory_size': len(self._cache),
                'max_size': self._max_size,
                'hit_rate': round(hit_rate, 4)
            }

    def __len__(self) -> int:
        """Return the number of items in memory cache."""
        return len(self._cache)

    def __contains__(self, key: str) -> bool:
        """Check if key exists in cache."""
        return self.has(key)


# Module-level singleton instance for shared caching
_default_cache: Optional[CacheManager] = None
_default_cache_lock = threading.Lock()


def get_default_cache(
    max_size: int = 1000,
    cache_dir: Optional[str] = None
) -> CacheManager:
    """
    Get or create the default cache manager singleton.

    Args:
        max_size: Maximum cache size (only used on first call)
        cache_dir: Cache directory (only used on first call)

    Returns:
        The default CacheManager instance
    """
    global _default_cache

    if _default_cache is None:
        with _default_cache_lock:
            if _default_cache is None:
                _default_cache = CacheManager(
                    max_size=max_size,
                    cache_dir=cache_dir
                )

    return _default_cache
