"""
Agent listings cache module.

Provides caching for agent page content and property listings
to avoid re-fetching during scraping operations.
"""

import json
import uuid
import aiofiles
from pathlib import Path
from datetime import datetime
from typing import Dict, Optional
from loguru import logger


class AgentListingsCache:
    """
    Manages caching for agent data and content.

    Provides separate caches for:
    - Agent page content (HTML)
    - Rental listings
    - Resale listings

    Example:
        >>> cache = AgentListingsCache(base_dir="output/cache")
        >>> await cache.cache_agent_content(agent_oid, url, html_content)
        >>> listings = await cache.get_cached_listings(agent_oid)
    """

    def __init__(self, base_dir: str = "output/cache"):
        """
        Initialize the listings cache.

        Args:
            base_dir: Base directory for cache files
        """
        self.base_dir = Path(base_dir)
        self.agents_dir = self.base_dir / "agents"
        self.listings_dir = self.base_dir / "listings"
        self.pages_dir = self.base_dir / "pages"

        # Create cache directories
        for directory in [self.agents_dir, self.listings_dir, self.pages_dir]:
            directory.mkdir(parents=True, exist_ok=True)

        logger.info("Initialized agent listings cache system")

    def get_agent_cache_dir(self, agent_oid: str) -> Path:
        """
        Get cache directory for specific agent.

        Args:
            agent_oid: Agent's ObjectId string

        Returns:
            Path to agent's cache directory
        """
        return self.agents_dir / agent_oid

    async def cache_agent_content(
        self,
        agent_oid: str,
        url: str,
        content: str,
        page_type: str = "main"
    ) -> None:
        """
        Cache agent page content.

        Args:
            agent_oid: Agent's ObjectId string
            url: URL that was fetched
            content: HTML content
            page_type: Type of page (e.g., 'main', 'rental', 'resale')
        """
        agent_dir = self.get_agent_cache_dir(agent_oid)
        agent_dir.mkdir(exist_ok=True)

        cache_file = agent_dir / f"{page_type}_{uuid.uuid4()}.html"
        cache_data = {
            'url': url,
            'content': content,
            'page_type': page_type,
            'timestamp': datetime.now().isoformat()
        }

        async with aiofiles.open(cache_file, 'w', encoding='utf-8') as f:
            await f.write(json.dumps(cache_data, ensure_ascii=False))

        logger.info(f"Cached {page_type} content for agent {agent_oid}")

    async def cache_agent_listings(self, agent_oid: str, listings_data: Dict) -> None:
        """
        Cache agent listings data with property type information.

        Args:
            agent_oid: Agent's ObjectId string
            listings_data: Dictionary with 'rental' and 'resale' listings
        """
        # Separate rental and resale listings
        rental_file = self.listings_dir / f"{agent_oid}_rental_listings.json"
        resale_file = self.listings_dir / f"{agent_oid}_resale_listings.json"

        # Save rental listings
        if listings_data.get("rental", {}).get("listings"):
            async with aiofiles.open(rental_file, 'w', encoding='utf-8') as f:
                await f.write(json.dumps({
                    'type': 'rental',
                    'data': listings_data["rental"],
                    'timestamp': datetime.now().isoformat()
                }, ensure_ascii=False))

        # Save resale listings
        if listings_data.get("resale", {}).get("listings"):
            async with aiofiles.open(resale_file, 'w', encoding='utf-8') as f:
                await f.write(json.dumps({
                    'type': 'resale',
                    'data': listings_data["resale"],
                    'timestamp': datetime.now().isoformat()
                }, ensure_ascii=False))

        logger.info(f"Cached listings data for agent {agent_oid}")

    async def get_cached_content(self, agent_oid: str, url: str) -> Optional[str]:
        """
        Get cached content for an agent URL.

        Args:
            agent_oid: Agent's ObjectId string
            url: URL to look up

        Returns:
            Cached HTML content or None if not found
        """
        agent_dir = self.get_agent_cache_dir(agent_oid)
        if not agent_dir.exists():
            return None

        for cache_file in agent_dir.glob("*.html"):
            try:
                async with aiofiles.open(cache_file, 'r', encoding='utf-8') as f:
                    data = json.loads(await f.read())
                    if data['url'] == url:
                        logger.info(f"Using cached content for {url}")
                        return data['content']
            except Exception as e:
                logger.error(f"Error reading cache file {cache_file}: {e}")
                continue
        return None

    async def get_cached_listings(self, agent_oid: str) -> Optional[Dict]:
        """
        Get cached listings for an agent.

        Args:
            agent_oid: Agent's ObjectId string

        Returns:
            Cached listings data or None if not found
        """
        listings_file = self.listings_dir / f"{agent_oid}_listings.json"
        if not listings_file.exists():
            return None

        try:
            async with aiofiles.open(listings_file, 'r', encoding='utf-8') as f:
                data = json.loads(await f.read())
                logger.info(f"Using cached listings for agent {agent_oid}")
                return data['data']
        except Exception as e:
            logger.error(f"Error reading listings cache for agent {agent_oid}: {e}")
            return None

    async def check_cache_status(self, agent_oid: str) -> Dict:
        """
        Check cache status for an agent.

        Args:
            agent_oid: Agent's ObjectId string

        Returns:
            Dictionary with cache status information
        """
        if not agent_oid:
            return {
                'has_listings': False,
                'rental_urls': [],
                'resale_urls': [],
                'needs_scraping': True,
                'has_rental': None,
                'has_resale': None
            }

        logger.info(f"Checking cache status for agent {agent_oid}")

        rental_cache = self.listings_dir / f"{agent_oid}_rental_listings.json"
        resale_cache = self.listings_dir / f"{agent_oid}_resale_listings.json"

        rental_urls = []
        resale_urls = []
        has_listings = False
        has_rental = None
        has_resale = None

        # Check rental cache
        if rental_cache.exists():
            try:
                async with aiofiles.open(rental_cache, 'r') as f:
                    rental_data = json.loads(await f.read())
                    rental_urls = rental_data.get('data', {}).get('listings', [])
                    has_rental = bool(rental_urls)
                    logger.info(f"Found cached rental data for {agent_oid}: {len(rental_urls)} properties")
                    has_listings = True
            except Exception as e:
                logger.error(f"Error reading rental cache for {agent_oid}: {e}")

        # Check resale cache
        if resale_cache.exists():
            try:
                async with aiofiles.open(resale_cache, 'r') as f:
                    resale_data = json.loads(await f.read())
                    resale_urls = resale_data.get('data', {}).get('listings', [])
                    has_resale = bool(resale_urls)
                    logger.info(f"Found cached resale data for {agent_oid}: {len(resale_urls)} properties")
                    has_listings = True
            except Exception as e:
                logger.error(f"Error reading resale cache for {agent_oid}: {e}")

        needs_scraping = not (rental_cache.exists() or resale_cache.exists())

        return {
            'has_listings': has_listings,
            'rental_urls': rental_urls,
            'resale_urls': resale_urls,
            'needs_scraping': needs_scraping,
            'has_rental': has_rental,
            'has_resale': has_resale
        }
