"""
Agent details cache module.

Provides simple file-based caching for agent details to avoid
re-fetching the same agent data from Idealista.
"""

import json
import time
from pathlib import Path
from typing import Dict, Optional

from ..utils.paths import get_cache_dir


class AgentDetailsCache:
    """
    Simple file-based cache for agent details to avoid re-fetching.

    Stores agent details in JSON files with TTL-based expiration.
    Each agent is stored in a separate file named by agent ID.

    Example:
        >>> cache = AgentDetailsCache(cache_dir=".cache/agents", ttl_hours=24)
        >>> cache.set("12345", {"phone": "123-456", "address": "Main St"})
        >>> details = cache.get("12345")  # Returns cached data or None if expired
    """

    def __init__(self, cache_dir: Path | str | None = None, ttl_hours: int = 24):
        """
        Initialize the agent cache.

        Args:
            cache_dir: Directory to store cache files
            ttl_hours: Time-to-live for cached data in hours
        """
        self.cache_dir = Path(cache_dir) if cache_dir else get_cache_dir() / "agents"
        self.cache_dir.mkdir(parents=True, exist_ok=True)
        self.ttl_seconds = ttl_hours * 3600

    def _get_cache_path(self, agent_id: str) -> Path:
        """
        Generate cache file path for an agent.

        Args:
            agent_id: Unique identifier for the agent

        Returns:
            Path to the cache file
        """
        return self.cache_dir / f"{agent_id}.json"

    def get(self, agent_id: str) -> Optional[Dict]:
        """
        Get cached agent details if fresh, else None.

        Args:
            agent_id: Unique identifier for the agent

        Returns:
            Cached agent details dictionary, or None if not cached or expired
        """
        cache_path = self._get_cache_path(agent_id)
        if not cache_path.exists():
            return None

        try:
            with open(cache_path, "r") as f:
                cached = json.load(f)

            # Check if cache is still valid
            cached_time = cached.get("_cached_at", 0)
            if time.time() - cached_time > self.ttl_seconds:
                cache_path.unlink()  # Delete stale cache
                return None

            # Remove metadata before returning
            result = {k: v for k, v in cached.items() if not k.startswith("_")}
            return result
        except (json.JSONDecodeError, IOError):
            return None

    def set(self, agent_id: str, details: Dict) -> None:
        """
        Cache agent details.

        Args:
            agent_id: Unique identifier for the agent
            details: Agent details dictionary to cache
        """
        cache_path = self._get_cache_path(agent_id)
        cached = {**details, "_cached_at": time.time()}
        try:
            with open(cache_path, "w") as f:
                json.dump(cached, f)
        except IOError as e:
            print(f"Warning: Failed to cache agent {agent_id}: {e}")

    def invalidate(self, agent_id: str) -> bool:
        """
        Invalidate (remove) cached data for an agent.

        Args:
            agent_id: Unique identifier for the agent

        Returns:
            True if cache was removed, False if not found
        """
        cache_path = self._get_cache_path(agent_id)
        if cache_path.exists():
            cache_path.unlink()
            return True
        return False

    def clear_all(self) -> int:
        """
        Clear all cached agent data.

        Returns:
            Number of cache files removed
        """
        count = 0
        for cache_file in self.cache_dir.glob("*.json"):
            cache_file.unlink()
            count += 1
        return count

    def get_stats(self) -> Dict:
        """
        Get cache statistics.

        Returns:
            Dictionary with cache statistics
        """
        cache_files = list(self.cache_dir.glob("*.json"))
        total_size = sum(f.stat().st_size for f in cache_files)
        return {
            "total_entries": len(cache_files),
            "total_size_bytes": total_size,
            "cache_dir": str(self.cache_dir),
            "ttl_hours": self.ttl_seconds / 3600
        }
