"""
Centralized caching layer for the idealistaScraper project.

This module provides:
- LRU in-memory caching with configurable size
- Disk persistence for cache with hash-based file naming
- URL deduplication tracking
- Cached configuration loading at module level

Usage:
    from src.cache import CacheManager, URLDeduplicator, config_loader

    # Cache manager
    cache = CacheManager(max_size=1000)
    cache.set("key", value)
    value = cache.get("key")

    # URL deduplication
    dedup = URLDeduplicator()
    if not dedup.is_scraped(url):
        # scrape url
        dedup.mark_scraped(url)

    # Config loading (module-level cached)
    patterns = config_loader.get_config('regex_patterns.json')
"""

from .manager import CacheManager, get_default_cache
from .dedup import URLDeduplicator, URLNormalizer, get_default_deduplicator
from .config_loader import (
    ConfigLoader,
    config_loader,
    get_regex_patterns,
    get_data_structure_config,
    get_settings_config,
    get_scrapfly_configs,
    get_views_config,
)
from .agent_cache import AgentDetailsCache
from .property_cache import PropertyProgressCache, PropertyContentCache
from .listing_cache import AgentListingsCache

__all__ = [
    # Cache manager
    'CacheManager',
    'get_default_cache',
    # URL deduplication
    'URLDeduplicator',
    'URLNormalizer',
    'get_default_deduplicator',
    # Config loading
    'ConfigLoader',
    'config_loader',
    'get_regex_patterns',
    'get_data_structure_config',
    'get_settings_config',
    'get_scrapfly_configs',
    'get_views_config',
    # Agent cache
    'AgentDetailsCache',
    # Property cache
    'PropertyProgressCache',
    'PropertyContentCache',
    # Listing cache
    'AgentListingsCache',
]
