"""

        This is SlideRunner - An Open Source Annotation Tool 
        for Digital Histology Slides.

         Marc Aubreville, Pattern Recognition Lab, 
         Friedrich-Alexander University Erlangen-Nuremberg 
         marc.aubreville@fau.de

        If you use this software in research, please citer our paper:
        M. Aubreville, C. Bertram, R. Klopfleisch and A. Maier:
        SlideRunner - A Tool for Massive Cell Annotations in Whole Slide Images. 
        In: Bildverarbeitung für die Medizin 2018. 
        Springer Vieweg, Berlin, Heidelberg, 2018. pp. 309-314.

        This file:
	   Visualize bounding boxes (object detection results)

"""

import SlideRunner.general.SlideRunnerPlugin as SlideRunnerPlugin
import queue
import os
from threading import Thread
from queue import Queue
import numpy as np
import openslide
import cv2
import datetime
from functools import partial
import time
import pickle
import torch
import torchvision.transforms as transforms
import SlideRunner_dataAccess.annotations as annotations
import sys

from PyQt5.QtCore import QSettings


settings = QSettings('TH Ingolstadt', 'SlideRunner Mitosis Detection Plugin')

import urllib.request

from tqdm import tqdm






class WSIImageGetter():
    def __init__(self, filename, width=128, height=128):
        self.slide = openslide.open_slide(filename)
        self.filename = filename
        self.width = width
        self.height = height

    def get_patch(self, x,y):
        return self._get_patch(x,y)

    def _get_patch(self, x,y):
        norm_mean = [0.7943, 0.6187, 0.7121]
        norm_std = [0.1720, 0.2366, 0.1834]
        return transforms.Normalize(norm_mean, norm_std)(Tensor(np.array(self.slide.read_region(location=(int(x-patchwidth/2), int(y-patchwidth/2)), size=(patchwidth,patchheight), level=0))[:,:,0:3]).permute(2,0,1)/255.)

#sys.path.insert(0, './SlideRunner/plugins')
from SlideRunner.plugins.Mitosis.model.RetinaNet import RetinaNet
from SlideRunner.plugins.Mitosis.lib.helper.fastai_helpers import *
from SlideRunner.plugins.Mitosis.lib.helper.object_detection_helper import *
from SlideRunner.plugins.Mitosis.lib.object_detection_helper import *
from SlideRunner.plugins.Mitosis.lib.helper.nms import non_max_suppression_by_distance

MODELIST = ['detection', 'MC heatmap', 'MC area proposal+detection', 'MC heatmap+detection']

def rescale_box(bboxes, size: Tensor):
    bboxes[:, :2] = bboxes[:, :2] - bboxes[:, 2:] / 2
    bboxes[:, :2] = (bboxes[:, :2] + 1) * size / 2
    bboxes[:, 2:] = bboxes[:, 2:] * size / 2
    bboxes = bboxes.long()
    return bboxes

class Plugin(SlideRunnerPlugin.SlideRunnerPlugin):
    version = 0.1
    shortName = 'NeuroPatho Mitosis Detection'
    inQueue = Queue()
    outQueue = Queue()
    initialOpacity=0.3
    updateTimer=0.1
    outputType = SlideRunnerPlugin.PluginOutputType.HEATMAP
    description = 'Perform mitosis detection'
    pluginType = SlideRunnerPlugin.PluginTypes.WHOLESLIDE_PLUGIN
    configurationList = list((
                            SlideRunnerPlugin.ComboboxPluginConfigurationEntry(uid='mode', name='Mode', options=MODELIST, selected_value=settings.value('defaultViewmode', 0)),
                            SlideRunnerPlugin.PluginConfigurationEntry(uid='thr', name='Detection threshold', initValue=0.55, minValue=0.3, maxValue=1.0),))
    
    COLORS = [[0,128,0,255],
              [128,0,0,255],
              [0,0,128,255],
              [128,128,0,255],
              [0,128,128,255],
              [128,128,128,255]]

    def __init__(self, statusQueue:Queue):
        self.statusQueue = statusQueue
        self.annotationLabels = {'Mitosis candidate' : SlideRunnerPlugin.PluginAnnotationLabel(0,'Mitosis candidate', [180,255,180,255]),
                                'Mitosis' : SlideRunnerPlugin.PluginAnnotationLabel(1,'Mitosis', [0,180,0,255]),
                                'ROI' : SlideRunnerPlugin.PluginAnnotationLabel(2,'ROI', [0,0,0,255]),}
        self.requestQueue = Queue()
        self.imageQueue = Queue()
        self.p = Thread(target=self.queueWorker, daemon=True)
        for t in range(4):# fork 4 children
            self.p2 = Thread(target=self.imageGetter, daemon=True)
            self.p2.start()
        self.p.start()
        self.overlay = None

        self.modelinitialized=False

    def updateProgressBar(self, total_size, count, block_size, retrieve_total_size):
        self.setProgressBar(int(100*count*block_size/total_size))

    def imageGetter(self):
        patchwidth=128
        patchheight=128
        while(True):
            (stage, data) = self.requestQueue.get()
            if (self.imageQueue.qsize()>100):
                self.requestQueue.put((stage,data)) # calculate once again
                time.sleep(0.1)
                continue

            if (stage==2): # second stage request
                [x1, y1, x2, y2, pred, score] = data
                x, y = int(0.5*(float(x1)+float(x2))),int(0.5*(float(y1)+float(y2)))
                norm_mean = [0.7943, 0.6187, 0.7121]
                norm_std = [0.1720, 0.2366, 0.1834]
                patch = transforms.Normalize(norm_mean, norm_std)(Tensor(np.array(self.slide.read_region(location=(int(x-patchwidth/2), int(y-patchwidth/2)), size=(patchwidth,patchheight), level=0))[:,:,0:3]).permute(2,0,1)/255.)
                patch = patch[None,:,:,:]
                self.imageQueue.put(([x1, y1, x2, y2, pred, score], patch)) # cave: different format, this is intended in order to not get confused results
            else:
                [x,y,ds] = data
                x_ds = int(np.floor(float(x)/ds))
                y_ds = int(np.floor(float(y)/ds))
                step_ds = int(np.ceil(float(self.shape)/ds))
                needCalculation = np.sum(self.activeMap[y_ds:y_ds+step_ds,x_ds:x_ds+step_ds])>0.9*step_ds*step_ds

                if not needCalculation:
                    self.imageQueue.put((False, x_ds, y_ds, x, y, None))
                    continue

                patch = np.array(self.slide.read_region(location=(int(x), int(y)),
                                                        level=self.level, size=(self.shape, self.shape)))[:, :, :3]

                patch = pil2tensor(patch / 255., np.float32)
                patch = transforms.Normalize(self.mean, self.std)(patch)
                self.imageQueue.put((True, x_ds, y_ds, x, y, patch))

    def download_url(self, url, output_path, total_size):
        urllib.request.urlretrieve(url, filename=output_path, reporthook=partial(self.updateProgressBar, total_size))


    def initModel(self):
        if (self.modelinitialized):
            return

        sys.path.append(os.path.dirname(os.path.realpath(__file__))+os.sep+'Mitosis')
        print('Path is:',sys.path)

        modelpath = os.path.dirname(os.path.realpath(__file__)) + os.sep + 'Mitosis/RetinaNet-TK-CB-SJ-1024-statedict.pth'
        if not (os.path.exists(modelpath)):
            self.setMessage('Retrieving first stage model ...')
            self.download_url('http://imi.thi.de/research/models/RetinaNet-TK-CB-SJ-1024-statedict.pth', output_path=modelpath, total_size=53604017)            

        self.setMessage('Loading models...')
        self.modelinitialized = True
        self.device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
        self.overlap = 0.9
        self.mean = [0.7568, 0.4490, 0.6464] # adjusted
        self.std = [0.1750, 0.1453, 0.1190]
        encoder = create_body(models.resnet18, True, -2)
        model = RetinaNet(encoder, n_classes=2, n_anchors=4, sizes=[64], chs=128, final_bias=-4., n_conv=3)
        if torch.cuda.is_available():
            model = model.cuda()


        if (torch.cuda.is_available()):
            model.load_state_dict(torch.load(modelpath))
        else:
            model.load_state_dict(torch.load(modelpath, map_location='cpu'))
        model.eval()
        self.model=model
        self.anchors = create_anchors(sizes=[(64,64)], ratios=[1], scales=[0.6, 0.7,0.8,0.9])


    def getAnnotationUpdatePolicy():
          # This is important to tell SlideRunner that he needs to update for every change in position (due to the heatmap)
          return SlideRunnerPlugin.AnnotationUpdatePolicy.UPDATE_ON_SCROLL_CHANGE

    def perform_inference(self, job, nms_thresh=0.4, batch_size=4, detect_thresh=0.52, firststage_cutoff=0.55):
        self.initModel()
        self.setMessage('Getting WSI overview')
        # Preprocess WSI
        self.slide = openslide.open_slide(job.slideFilename)

        downsamples_int = [int(x) for x in self.slide.level_downsamples]
        if 32 in downsamples_int:
            ds = 32
        elif 16 in downsamples_int:
            ds = 16
        else:
            self.showMessageBox('Error: Image does not contain downsampling of factor 16 or 32, which is required.')
            return

        level = np.where(np.abs(np.array(self.slide.level_downsamples)-ds)<0.1)[0][0]
        overview = self.slide.read_region(level=level, location=(0,0), size=self.slide.level_dimensions[level])

        # Convert to grayscale
        gray = cv2.cvtColor(np.array(overview)[:,:,0:3],cv2.COLOR_BGR2GRAY)

        # OTSU thresholding
        ret, thresh = cv2.threshold(gray,0,255,cv2.THRESH_BINARY_INV+cv2.THRESH_OTSU)

        # dilate
        dil = cv2.dilate(thresh, kernel = np.ones((7,7),np.uint8))

        # erode
        self.activeMap = cv2.erode(dil, kernel = np.ones((7,7),np.uint8))

        self.down_factor=1
        self.overlap=0.9
        self.shape = 1024
        uid=100000
        self.annos_original=[]
        self.level=0
        x_steps = range(0, int(self.slide.level_dimensions[0][0]),
                        int(self.shape * self.down_factor * self.overlap))
        y_steps = range(0, int(self.slide.level_dimensions[0][1]),
                        int(self.shape * self.down_factor * self.overlap))
        patches = []
        x_coordinates = []
        y_coordinates = []
        patch_counter = 0
        pstep_old = 0
        activeMapCopy = np.ones_like(self.activeMap)
        class_pred_batch, bbox_pred_batch = [], []
        calculateOnlyStage1=False
        self.overlay = np.zeros(np.array(overview).shape, np.uint8)[:,:,0:3]
        self.overlay[:,:,0] = self.activeMap

        t0 = time.time()
        seconds_elapsed = 0
        stage1_results_file = job.slideFilename+'_detected_stage1_1024.npz'
        stage2_results_file = job.slideFilename+'_detected_stage2_1024.npz'
        step_ds = int(np.ceil(float(self.shape)/ds))
        if not os.path.exists(stage1_results_file):
            for y in y_steps:
                for x in x_steps:
                    self.requestQueue.put((1,(x,y,ds)))

            with torch.no_grad():
                for y in y_steps:
                    for x in x_steps:
                        if (seconds_elapsed!=int(time.time()-t0)):
                            seconds_elapsed = int(time.time()-t0)
                            self.setMessage('Scanning WSI for Mitotic Figure candidates (stage 1) (%s) ' % (datetime.timedelta(seconds=seconds_elapsed)) )

                        needCalculation, x_ds, y_ds, x,y, patch = self.imageQueue.get()
                        activeMapCopy[y_ds:y_ds+step_ds,x_ds:x_ds+step_ds]=255
                        if not needCalculation:
                            patch_counter += 1
                            continue

                        if (torch.cuda.is_available()):
                            patch = patch.to(self.device)

                        patches.append(patch[None, :, :, :])
                        x_coordinates.append(x)
                        y_coordinates.append(y)

                        if len(patches) == batch_size:
                            class_pred, bbox_pred, _ = self.model(torch.cat(patches).to(self.device))
                            class_pred_batch.extend (class_pred.cpu())
                            bbox_pred_batch.extend(bbox_pred.cpu())

                            patches = []

                        patch_counter += 1
                        
                        
                        pstep = int((patch_counter / (len(x_steps) * len(y_steps))) * 1000)
                        if (pstep != pstep_old):
                            self.setProgressBar(float(pstep)/10)
                            self.overlay[activeMapCopy==255,1] = 255.0
                            self.updateAnnotations()
                        pstep_old=pstep


            if len(patches) > 0:
                class_pred, bbox_pred, _ = self.model(torch.cat(patches).to(self.device))
                class_pred_batch.extend (class_pred.cpu())
                bbox_pred_batch.extend(bbox_pred.cpu())


            self.overlay = None
            counter = 0
            self.setMessage('Non-maxima suppression')
            for clas_pred, bbox_pred, x, y in zip(class_pred_batch, bbox_pred_batch, x_coordinates, y_coordinates):
                if (seconds_elapsed!=int(time.time()-t0)):
                    seconds_elapsed = int(time.time()-t0)
                    self.setMessage('Non-maxima suppression (%s)' % datetime.timedelta(seconds=seconds_elapsed) )


                modelOutput = process_output(clas_pred.cpu(), bbox_pred.cpu(),
                                                        self.anchors, detect_thresh=0.3)
                
                bbox_pred, scores, preds = [modelOutput[x] for x in ['bbox_pred', 'scores', 'preds']]
                if bbox_pred is not None:
                    to_keep = nms(bbox_pred, scores, nms_thresh)  # nms_thresh=
                    bbox_pred, preds, scores = bbox_pred[to_keep].cpu(), preds[to_keep].cpu(), scores[to_keep].cpu()

                    t_sz = torch.Tensor([self.shape, self.shape])[None].float()
                    bbox_pred = rescale_box(bbox_pred, t_sz)

                    for box, pred, score in zip(bbox_pred, preds, scores):
                        y_box, x_box = box[:2]
                        h, w = box[2:4]

                        x1 = int(x_box) * self.down_factor + x
                        y1 = int(y_box) * self.down_factor + y
                        x2 = x1 + int(w) * self.down_factor
                        y2 = y1 + int(h) * self.down_factor

                        self.annos_original.append([x1, y1, x2, y2, int(pred), float(score)])

                        uid += 1

                self.setProgressBar((counter / (len(y_coordinates))) * 100)
                counter += 1
            self.perform_nms(nms_thresh)
            # save intermediate results
            np.savez_compressed(stage1_results_file, annos_original=np.array(self.annos_original))
        else:
            # reconstruct first stage
            results_stage1 = np.load(stage1_results_file)
            self.annos_original = results_stage1['annos_original'].tolist()


        calculateOnlyStage1 = True
        print('Annos raw: ',len(self.annos_original))
        for [x1, y1, x2, y2, pred, score] in self.annos_original:
            
            if (calculateOnlyStage1):
                if (score > job.configuration['thr']):
                    myanno = annotations.rectangularAnnotation(uid=uid,
                                                            x1=x1, y1=y1,
                                                            x2=x2, y2=y2,
                                                            text="{:1.2}".format(score),
                                                            pluginAnnotationLabel=self.annotationLabels[
                                                                'Mitosis'])

                self.annos.append(myanno)
            uid += 1
        self.overlay = None
        stage2_threshold = job.configuration['thr']
 #       self.annos=[]
        # calculate second stage
        self.updateAnnotations()

        self.slideObj = openslide.open_slide(job.slideFilename)
        self.ds = 32

        settings.setValue('defaultViewmode', job.configuration['mode'])

        if (job.configuration['mode']>0):
            self.downsampledMap = np.zeros((int(self.slideObj.dimensions[1]/self.ds), int(self.slideObj.dimensions[0]/self.ds)))

        countmit=0
        for [x1, y1, x2, y2, pred, prob_mitosis] in self.annos_original:
            if (prob_mitosis>stage2_threshold) or True:
                if (float(prob_mitosis)>stage2_threshold):
                    countmit+=1
                if (job.configuration['mode']>0) and (float(prob_mitosis)>job.configuration['thr']):
                    self.downsampledMap[int(((y2+y1)/2)/self.ds),int(((x1+x2)/2)/self.ds)] += 1
                if (float(prob_mitosis)>job.configuration['thr']):
                    myanno = annotations.rectangularAnnotation(uid=uid,
                                                            x1=x1, y1=y1,
                                                            x2=x2, y2=y2,
                                                            text="{:1.2}".format(prob_mitosis),
                                                            # text="conf: {:1.2}".format(score),
                                                            minimumAnnotationLabelZoom = 4,
                                                            pluginAnnotationLabel=self.annotationLabels[
                                                                'Mitosis' if float(prob_mitosis)>stage2_threshold else 'Mitosis candidate'])
                if (job.configuration['mode'] in [0,2,3]):
                    self.annos.append(myanno)

        self.setMessage('Found %d mitotic figures .' % (countmit))
        if (job.configuration['mode']>0):
            A = 2.5 # mm^2
            W_hpf_microns = np.sqrt(A*4/3) * 1000 # in microns
            H_hpf_microns = np.sqrt(A*3/4) * 1000 # in microns

            micronsPerPixel = self.slideObj.properties[openslide.PROPERTY_NAME_MPP_X]

            W_hpf = int(W_hpf_microns / float(micronsPerPixel))
            H_hpf = int(H_hpf_microns / float(micronsPerPixel))

            W_x = int(W_hpf / self.ds)
            W_y = int(H_hpf / self.ds)
            kernel = np.ones((W_y,W_x),np.float32)
            self.mitoticCount = cv2.filter2D(self.downsampledMap, -1, kernel, borderType=cv2.BORDER_CONSTANT )


        if (job.configuration['mode'] in [2]):
            maxC = np.unravel_index(np.argmax(self.mitoticCount, axis=None), self.mitoticCount.shape)
            
            myanno = annotations.rectangularAnnotation(uid=uid,
                                                    x1=int(maxC[1]*self.ds-W_hpf/2), y1=int(maxC[0]*self.ds-H_hpf/2),
                                                    x2=int(maxC[1]*self.ds+W_hpf/2), y2=int(maxC[0]*self.ds+H_hpf/2),
                                                    text="10 HPF ROI",
                                                    # text="conf: {:1.2}".format(score),
                                                    pluginAnnotationLabel=self.annotationLabels[
                                                        'ROI' ])
            x1 = int(maxC[1]*self.ds-W_hpf/2)
            y1 = int(maxC[0]*self.ds-H_hpf/2)
            w_singleHPF = int(W_hpf/10)
            h_singleHPF = int(H_hpf/1)
            hpfcount=0
            for x in range(10):                                                        
                for y in range(1):                  
                    hpfcount+=1                                      
                    myanno = annotations.rectangularAnnotation(uid=uid,
                                                            x1=(x1+(x)*w_singleHPF), y1=(y1+(y)*h_singleHPF),
                                                            x2=(x1+(x+1)*w_singleHPF), y2=(y1+(y+1)*h_singleHPF),
                                                            text="HPF %d" % (hpfcount),
                                                            # text="conf: {:1.2}".format(score),
                                                            pluginAnnotationLabel=self.annotationLabels[
                                                                'ROI' ])
                    self.annos.append(myanno)

        self.updateAnnotations()

    def final_nms(self, nms_thresh):
        boxes = np.array(self.annos_2nd)
        if len(self.annos_2nd) > 0:
            ids = non_max_suppression_by_distance(boxes, boxes[:, 5], 25, True)

            self.annos_2nd = [self.annos_2nd[id] for id in ids]
#            self.annos = [self.annos[id] for id in ids]

#            self.updateAnnotations()

    def perform_nms(self, nms_thresh):
        boxes = np.array(self.annos_original)
        if len(self.annos_original) > 0:
            ids = non_max_suppression_by_distance(boxes, boxes[:, 5], 25, True)

            self.annos_original = [self.annos_original[id] for id in ids]
#            self.annos = [self.annos[id] for id in ids]

            self.updateAnnotations()

    def overlayHeatmap(self, numpyImage) -> np.ndarray:

        if self.overlay is not None and numpyImage is not None:
            temp_overlay = cv2.resize(self.overlay, numpyImage.shape[:2][::-1])
            return temp_overlay#cv2.addWeighted(numpyImage, 0.7, temp_overlay, 0.3, 0)
        else:
            return numpyImage

    def queueWorker(self):
        debugModule= False
        quitSignal = False
        oldFilename = ''
        oldArchive = ''
        oldSlide = ''
        oldMode=-1
        oldthr=0.0
        self.ds=32
        self.sendAnnotationLabelUpdate()
        while not quitSignal:
            job = SlideRunnerPlugin.pluginJob(self.inQueue.get())
            print(job)
            print('Config:',job.configuration)

            if (job.jobDescription == SlideRunnerPlugin.JobDescription.QUIT_PLUGIN_THREAD):
                # signal to exit this thread
                quitSignal=True
                continue
            

            slideChanged = oldSlide != job.slideFilename
            thrChanged = job.configuration['thr'] != oldthr
            modeChanged = job.configuration['mode'] != oldMode
            oldMode = job.configuration['mode']
            oldSlide = job.slideFilename
            oldthr = job.configuration['thr']


            [foo,self.ext] = os.path.splitext(oldArchive)
            self.ext = self.ext.upper()

            nms_thresh = 0.3

            if (slideChanged) or (modeChanged) or (thrChanged):
                self.annos = list()
                self.perform_inference(job)
            if (job.configuration['mode'] in [1,3]):
                coords_ds = np.int16(np.array(job.coordinates)/self.ds)

                centerImg = cv2.getRectSubPix(np.float32(self.mitoticCount[:,:,None]), patchSize=(coords_ds[2], coords_ds[3]), center=(coords_ds[0]+coords_ds[2]*0.5,coords_ds[1]+coords_ds[3]*0.5))
                
                resized = cv2.resize(centerImg, dsize=(job.currentImage.shape[1], job.currentImage.shape[0]))
                self.updateAnnotations()
                print('Sending overlay')
                self.returnImage(resized)
            elif (modeChanged):
                self.returnImage(None)

            if (modeChanged):
                self.triggerRefreshView()

            self.setProgressBar(-1)



    def getAnnotations(self):
        return self.annos


    def getAnnotationLabels(self):
            # sending default annotation labels
            return [self.annotationLabels[k] for k in self.annotationLabels.keys()]

        
