# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['typeddfs']

package_data = \
{'': ['*']}

install_requires = \
['natsort>=7', 'pandas>=1.2,<2.0']

extras_require = \
{'feather': ['pyarrow>=4.0,<5.0'],
 'fwf': ['tabulate>=0.8,<0.9', 'wcwidth>=0.2']}

setup_kwargs = {
    'name': 'typeddfs',
    'version': '0.7.0',
    'description': 'Pandas DataFrame subclasses that enforce structure and can self-organize.',
    'long_description': '# Typed DataFrames\n\n[![Version status](https://img.shields.io/pypi/status/typeddfs?label=status)](https://pypi.org/project/typeddfs)\n[![License](https://img.shields.io/badge/License-Apache%202.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)\n[![Python version compatibility](https://img.shields.io/pypi/pyversions/typeddfs?label=Python)](https://pypi.org/project/typeddfs)\n[![Version on Github](https://img.shields.io/github/v/release/dmyersturnbull/typed-dfs?include_prereleases&label=GitHub)](https://github.com/dmyersturnbull/typed-dfs/releases)\n[![Version on PyPi](https://img.shields.io/pypi/v/typeddfs?label=PyPi)](https://pypi.org/project/typeddfs)\n[![Build (Actions)](https://img.shields.io/github/workflow/status/dmyersturnbull/typed-dfs/Build%20&%20test?label=Tests)](https://github.com/dmyersturnbull/typed-dfs/actions)\n[![Coverage (coveralls)](https://coveralls.io/repos/github/dmyersturnbull/typed-dfs/badge.svg?branch=main&service=github)](https://coveralls.io/github/dmyersturnbull/typed-dfs?branch=main)\n[![Documentation status](https://readthedocs.org/projects/typed-dfs/badge)](https://typed-dfs.readthedocs.io/en/stable/)\n[![Maintainability](https://api.codeclimate.com/v1/badges/6b804351b6ba5e7694af/maintainability)](https://codeclimate.com/github/dmyersturnbull/typed-dfs/maintainability)\n[![Scrutinizer Code Quality](https://scrutinizer-ci.com/g/dmyersturnbull/typed-dfs/badges/quality-score.png?b=main)](https://scrutinizer-ci.com/g/dmyersturnbull/typed-dfs/?branch=main)\n[![Created with Tyrannosaurus](https://img.shields.io/badge/Created_with-Tyrannosaurus-0000ff.svg)](https://github.com/dmyersturnbull/tyrannosaurus)\n\n\nPandas DataFrame subclasses that enforce structure and self-organize.  \n*Because your functions can’t exactly accept **any**  DataFrame**.  \n`pip install typeddfs[feather,fwf]`\n\nStop passing `index_cols=` and `header=` to `to_csv` and `read_csv`.\nYour “typed” dataframes will remember how they’re supposed to be written and read.\nThat means columns are used for the index, string columns are always read as strings,\nand custom constraints are verified.\n\nNeed to read a tab-delimited file? `read_file("myfile.tab")`.\nFeather? Parquet? HDF5? .json.zip? Gzipped fixed-width?\nUse `read_file`. Write a file? Use `write_file`.\n\nSome useful extra functions, plus various Pandas issues fixed:\n- `read_csv`/`to_csv`,  `read_json`/`to_json`, etc., are inverses.\n  `read_file`/`write_file`, too.\n- In Pandas, you can write an empty DataFrame but not read it.\n  Typed-dfs will always read in what you wrote out.\n- No more empty `.feather`/`.snappy`/`.h5` files written on error.\n- You can write fixed-width as well as read.\n\n```python\n\nfrom typeddfs._entries import TypedDfs\n\nMyDfType = (\n  TypedDfs.typed("MyDfType")\n    .require("name", index=True)  # always keep in index\n    .require("value", dtype=float)  # require a column and type\n    .drop("_temp")  # auto-drop a column\n    .verify(lambda ddf: len(ddf) == 12)  # require exactly 12 rows\n).build()\n\ndf = MyDfType.read_file(input("filename? [.feather/.csv.gz/.tsv.xz/etc.]"))\ndf.sort_natural().write_file("myfile.feather")\n```\n\n### 🎨 More complex example\n\nFor a CSV like this:\n\n| key   | value  | note |\n| ----- | ------ | ---- |\n| abc   | 123    | ?    |\n\n```python\n\nfrom typeddfs._entries import TypedDfs\n\n# Build me a Key-Value-Note class!\nKeyValue = (\n  TypedDfs.typed("KeyValue")  # With enforced reqs / typing\n    .require("key", dtype=str, index=True)  # automagically add to index\n    .require("value")  # required\n    .reserve("note")  # permitted but not required\n    .strict()  # disallow other columns\n).build()\n\n# This will self-organize and use "key" as the index:\ndf = KeyValue.read_csv("example.csv")\n\n# For fun, let"s write it and read it back:\ndf.to_csv("remke.csv")\ndf = KeyValue.read_csv("remake.csv")\nprint(df.index_names(), df.column_names())  # ["key"], ["value", "note"]\n\n\n# And now, we can type a function to require a KeyValue,\n# and let it raise an `InvalidDfError` (here, a `MissingColumnError`):\ndef my_special_function(df: KeyValue) -> float:\n  return KeyValue(df)["value"].sum()\n```\n\nAll of the normal DataFrame methods are available.\nUse `.untyped()` or `.vanilla()` to make a detyped copy that doesn’t enforce requirements.\n\n### 🔌 Serialization support\n\nLike Pandas, TypedDfs can read and write to various formats.\nIt provides the methods `read_file` and `write_file`, which guess the format from the\nfilename extension. For example, `df.write_file("myfile.snappy)` writes Parquet files,\nand `df.write_file("myfile.tab.gz")` writes a gzipped, tab-delimited file.\nThe `read_file` method works the same way: `MyDf.read_file("myfile.feather")` will\nread an Apache Arrow Feather file, and `MyDf.read_file("myfile.json.gzip")`reads\na gzipped JSON file. You can pass keyword arguments to those functions.\n\nSerialization is provided through Pandas, and some formats require additional packages.\nPandas does not specify compatible versions, so typed-dfs specifies\n[extras](https://python-poetry.org/docs/pyproject/#extras) are provided in typed-dfs\nto ensure that those packages are installed with compatible versions.\n- To install with [Feather](https://arrow.apache.org/docs/python/feather.html) support,\n  use `pip install typeddfs[feather]`.\n- To install with support for all serialization formats,\n  use `pip install typeddfs[feather] fastparquet tables`.\n\nHowever, hdf5 and parquet have limited compatibility,\nrestricted to some platforms and Python versions.\nIn particular, neither is supported in Python 3.9 on Windows as of 2021-03-02.\n(See the [llvmlite issue](https://github.com/numba/llvmlite/issues/669)\nand [tables issue](https://github.com/PyTables/PyTables/issues/854).)\n\nFeather offers massively better performance over CSV, gzipped CSV, and HDF5\nin read speed, write speed, memory overhead, and compression ratios.\nParquet typically results in smaller file sizes than Feather at some cost in speed.\nFeather is the preferred format for most cases.\n\n**⚠ Note:** The `hdf5` and `parquet` extras are currently disabled.\n\n| format   | packages              | extra     | compatibility | performance  |\n| -------- | --------------------  | --------- | ------------- | ------------ |\n| pickle   | none                  | none      | ❗ ️           | −           |\n| csv      | none                  | none      | ✅             | −−          |\n| json     | none                  | none      | /️            | −−-         |\n| .npy †   | none                  | none      | †️            | +           |\n| .npz †   | none                  | none      | †️            | +           |\n| flexwf   | none                  | `fwf`     | ✅             | −−-         |\n| Feather  | `pyarrow`             | `feather` | ✅             | ++++        |\n| Parquet  | `pyarrow,fastparquet` | `parquet` | ❌             | +++         |\n| HDF5     | `tables`              | `hdf5`    | ❌             | −           |\n\n❗ == Pickle is explicitly not supported due to vulnerabilities and other issues.  \n/ == Mostly. JSON has inconsistent handling of `None`.  \n† == .npy and .npz only serialize numpy objects and therefore skip indices.  \nNote: `.flexwf` is fixed-width with optional delimiters; `.fwf` is not used\nto avoid a potential future conflict with `pd.DataFrame.to_fwf` (which does not exist yet).\n\n### 📝 Extra notes\n\nA small note of caution: [natsort](https://github.com/SethMMorton/natsort) is not pinned\nto a specific major version because it receives somewhat frequent major updates.\nThis means that the result of typed-df’s `sort_natural` could change.\nYou can pin natsort to a specific major version;\ne.g. `natsort = "^7"` with [Poetry](https://python-poetry.org/) or `natsort>=7,<8` with pip.\n\nFixed-width format is provided through Pandas `read_fwf` but can be written\nvia [tabulate](https://pypi.org/project/tabulate/).\n\n### 🍁 Contributing\n\nTyped-Dfs is licensed under the [Apache License, version 2.0](https://www.apache.org/licenses/LICENSE-2.0).\n[New issues](https://github.com/dmyersturnbull/typed-dfs/issues) and pull requests are welcome.\nPlease refer to the [contributing guide](https://github.com/dmyersturnbull/typed-dfs/blob/main/CONTRIBUTING.md).\nGenerated with [Tyrannosaurus](https://github.com/dmyersturnbull/tyrannosaurus).\n',
    'author': 'Douglas Myers-Turnbull',
    'author_email': None,
    'maintainer': 'dmyersturnbull',
    'maintainer_email': None,
    'url': 'https://github.com/dmyersturnbull/typed-dfs',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
