"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BastionHost = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The database is located in an isolated subnet, meaning that it is not accessible from the public internet. As such, to interact with the database directly, a user must tunnel through a bastion host.
 *
 * ### Configuring
 *
 * This codebase controls _who_ is allowed to connect to the bastion host. This requires two steps:
 *
 * 1. Adding the IP address from which you are connecting to the `ipv4Allowlist` array
 * 1. Creating a bastion host system user by adding the user's configuration inform to `userdata.yaml`
 *
 * #### Adding an IP address to the `ipv4Allowlist` array
 *
 * The `BastionHost` construct takes in an `ipv4Allowlist` array as an argument. Find your IP address (eg `curl api.ipify.org`) and add that to the array along with the trailing CIDR block (likely `/32` to indicate that you are adding a single IP address).
 *
 * #### Creating a user via `userdata.yaml`
 *
 * Add an entry to the `users` array with a username (likely matching your local systems username, which you can get by running the `whoami` command in your terminal) and a public key (likely your default public key, which you can get by running `cat ~/.ssh/id_*.pub` in your terminal).
 *
 * #### Tips & Tricks when using the Bastion Host
 *
 * **Connecting to RDS Instance via SSM**
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID \
 * --document-name AWS-StartPortForwardingSessionToRemoteHost \
 * --parameters '{
 * "host": [
 * "example-db.c5abcdefghij.us-west-2.rds.amazonaws.com"
 * ],
 * "portNumber": [
 * "5432"
 * ],
 * "localPortNumber": [
 * "9999"
 * ]
 * }' \
 * --profile $AWS_PROFILE
 * ```
 *
 * ```sh
 * psql -h localhost -p 9999 # continue adding username (-U) and db (-d) here...
 * ```
 *
 * Connect directly to Bastion Host:
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID --profile $AWS_PROFILE
 * ```
 *
 * **Setting up an SSH tunnel**
 *
 * In your `~/.ssh/config` file, add an entry like:
 *
 * ```
 * Host db-tunnel
 * Hostname {the-bastion-host-address}
 * LocalForward 54322 {the-db-hostname}:5432
 * ```
 *
 * Then a tunnel can be opened via:
 *
 * ```
 * ssh -N db-tunnel
 * ```
 *
 * And a connection to the DB can be made via:
 *
 * ```
 * psql -h 127.0.0.1 -p 5433 -U {username} -d {database}
 * ```
 *
 * **Handling `REMOTE HOST IDENTIFICATION HAS CHANGED!` error**
 *
 * If you've redeployed a bastion host that you've previously connected to, you may see an error like:
 *
 * ```
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * @    WARNING: REMOTE HOST IDENTIFICATION HAS CHANGED!     @
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * IT IS POSSIBLE THAT SOMEONE IS DOING SOMETHING NASTY!
 * Someone could be eavesdropping on you right now (man-in-the-middle attack)!
 * It is also possible that a host key has just been changed.
 * The fingerprint for the ECDSA key sent by the remote host is
 * SHA256:mPnxAOXTpb06PFgI1Qc8TMQ2e9b7goU8y2NdS5hzIr8.
 * Please contact your system administrator.
 * Add correct host key in /Users/username/.ssh/known_hosts to get rid of this message.
 * Offending ECDSA key in /Users/username/.ssh/known_hosts:28
 * ECDSA host key for ec2-12-34-56-789.us-west-2.compute.amazonaws.com has changed and you have requested strict checking.
 * Host key verification failed.
 * ```
 *
 * This is due to the server's fingerprint changing. We can scrub the fingerprint from our system with a command like:
 *
 * ```
 * ssh-keygen -R 12.34.56.789
 * ```
 *
 */
class BastionHost extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { stackName } = aws_cdk_lib_1.Stack.of(this);
        // Build ec2 instance
        this.instance = new aws_cdk_lib_1.aws_ec2.Instance(this, "bastion-host", {
            vpc: props.vpc,
            vpcSubnets: { subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC },
            instanceName: stackName,
            instanceType: aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE4_GRAVITON, aws_cdk_lib_1.aws_ec2.InstanceSize.NANO),
            machineImage: aws_cdk_lib_1.aws_ec2.MachineImage.latestAmazonLinux({
                generation: aws_cdk_lib_1.aws_ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                cpuType: aws_cdk_lib_1.aws_ec2.AmazonLinuxCpuType.ARM_64,
            }),
            userData: props.userData,
            userDataCausesReplacement: true,
        });
        // Assign elastic IP
        if (props.createElasticIp ?? true) {
            new aws_cdk_lib_1.aws_ec2.CfnEIP(this, "IP", {
                instanceId: this.instance.instanceId,
                tags: [{ key: "Name", value: stackName }],
            });
        }
        // Allow bastion host to connect to db
        this.instance.connections.allowTo(props.db.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connection from bastion host");
        // Allow IP access to bastion host
        for (const ipv4 of props.ipv4Allowlist) {
            this.instance.connections.allowFrom(aws_cdk_lib_1.aws_ec2.Peer.ipv4(ipv4), aws_cdk_lib_1.aws_ec2.Port.tcp(props.sshPort || 22), "SSH Access");
        }
        // Integrate with SSM
        this.instance.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: [
                "ssmmessages:*",
                "ssm:UpdateInstanceInformation",
                "ec2messages:*",
            ],
            resources: ["*"],
        }));
        new aws_cdk_lib_1.CfnOutput(this, "instance-id-output", {
            value: this.instance.instanceId,
            exportName: `${stackName}-instance-id`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-ip-output", {
            value: this.instance.instancePublicIp,
            exportName: `${stackName}-instance-public-ip`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-dns-name-output", {
            value: this.instance.instancePublicDnsName,
            exportName: `${stackName}-public-dns-name`,
        });
    }
}
exports.BastionHost = BastionHost;
_a = JSII_RTTI_SYMBOL_1;
BastionHost[_a] = { fqn: "cdk-pgstac.BastionHost", version: "2.3.1" };
//# sourceMappingURL=data:application/json;base64,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