from .base import EngineSpecificParser
from ..engine.base import SessionResult
import tempfile
import os
import shutil

class LumericalFSPFile(EngineSpecificParser, SessionResult):
  """
  Holds the file path and potentially the entire data of a Lumerical FSP file.
  Instances of this class are generated by LumericalSession objects and can be 
  found in the "sess_res" fields of simulation results. Right before running a 
  simulation, a LumericalSession loads the FSP file it's about to run into memory
  and later passes it, along with the file path to the post-simulation FSP file, 
  to a new LumericalFSPFile. This allows you to access the pre-simulation FSP file
  (with all the geometry) without saving any actual files, as the data is stored
  in the simulation result object, but prevents loading massive files into memory
  by keeping the post-simulation FSP file on disk.
  """
  def __init__(self, data, fpath, engine=None):
    """
    Parameters
    ----------
    data : bytes
      The entire data, as a bytestring, of the FSP file
    fpath : str
      The path to the FSP file
    engine : LumericalEngine
      A LumericalEngine instance to open simulation files with
    """
    super().__init__((data, fpath), engine=engine)
  
  def show(self, data_only=False):
    """Open the FSP file, by first trying to find the file pointed to by fpath from 
    the creation of this instance and then trying the one stored as a bytestring. 

    Parameters
    ----------
    data_only : bool
      If True, only try to open the file stored as a bytestring in memory
    """
    if not self.engine or self.engine.name != "lumerical":
      raise ValueError("Current engine must be 'lumerical' to open this file")

    if self.data[1] and not data_only:
      self.engine.open_fsp(self.data[1])
    elif self.data[0]:
      with tempfile.TemporaryDirectory() as d:
        fp = open(os.path.join(d, "simulation.fsp"), "wb")
        fp.write(self.data[0])
        fp.close()
        
        self.engine.open_fsp(fp.name)
    else:
      raise ValueError("No fsp data or file path found")
  
  def __repr__(self):
    return "LumericalFSPFile(%d, %s)" % (len(self.data[0]) if self.data[0] else 0, self.data[1])

  def save(self, fpath, move=False, data_only=False):
    """Save the data of this file to a new path. Defaults to saving the file stored in the file path
    that was passed on creation of this instance, and if that wasn't given then it saves the file
    passed as a bytestring instead.

    Parameters
    ----------
    move : bool
      If True and the file that we end up saving is on disk, then instead of copying it just move
      it to the new location
    data_only : bool
      If True, only try to save the file stored as a bytestring in memory
    """
    if self.data[1] and not data_only:
      if move:
        os.rename(self.data[1], fpath)
        self.data[1] = fpath
      else:
        shutil.copyfile(self.data[1], fpath)
    elif self.data[0]:
      with open(fpath, "wb") as fp:
        fp.write(self.data[0])
    else:
      raise ValueError("No fsp data or file path found")
  