# functions for interacting with cellpose processed data
import os

import numpy as np
from matplotlib import pyplot as plt

from imagingplus.main.subcore import ImagingMetadata

from imagingplus.main.core import SingleImage
from imagingplus.plotting._utils import _add_scalebar


class CellposeImg(SingleImage):
    def __init__(self, dataPath: str, cellpose_npy_path, imparams: ImagingMetadata, saveDir: str, expGroup: str = '',
                 **kwargs):
        super().__init__(dataPath=dataPath, imparams=imparams, **kwargs)
        if not os.path.exists(cellpose_npy_path):
            raise FileNotFoundError(f'cellpose npy file not found from path: {cellpose_npy_path}')
        self.cellpose_npy_path = cellpose_npy_path

        self.expGroup = expGroup
        self.saveDir = saveDir
        self.roi_areas: dict = {}  #: dict containing coordinates of the whole masks generated by cellpose for each ROI
        self.roi_outlines: dict = {}  #: dict containing coordinates of the outlines generated by cellpose for each ROI
        self.masks: np.ndarray = np.empty(shape=[imparams.frame_x,
                                                 imparams.frame_y])  #: numpy array of ROIs masks, ROI # is allocated at the appropriate coordinate

    @property
    def trial_id(self):
        """
        ID of processed image.

        :return: str;
        """
        return f"{self.expGroup} {self.imgID}"

    @property
    def roi_ids(self):
        """
        IDs of ROIs identified in cellpose processsed image.

        :return: list;
        """
        return [*self.roi_areas]

    @classmethod
    def load_cellpose(cls, dataPath: str, cellpose_npy_path, imparams: ImagingMetadata, saveDir: str, expGroup: str = ''):
        """
        Alternate constructor.
        Import and load array coords from cellpose generated ROI masks for red channel interneurons.

        :param dataPath: path to source image
        :param cellpose_npy_path: location of .npy output of previously completed cellpose run
        :param imparams: image collection parameters of image
        :param saveDir: path to save cellpose processing object
        :param expGroup: experimental group of image
        """

        cellpose = cls(dataPath=dataPath, cellpose_npy_path=cellpose_npy_path, imparams=imparams, saveDir=saveDir,
                       expGroup=expGroup)

        # find all paths with '_seg.npy'
        npy_path = cellpose.cellpose_npy_path

        dat = np.load(npy_path, allow_pickle=True).item()
        rois = np.unique(dat['masks'])[1:]
        outlines = {}
        roi_areas = {}
        for i in rois:
            roi_areas[i] = np.where(dat['masks'] == i)
            outlines[i] = np.where(dat['outlines'] == i)

        cellpose.roi_outlines = outlines
        cellpose.roi_areas = roi_areas
        cellpose.masks = dat['masks']


    def plot_img(self, roi_labels=True, **kwargs):
        """
        Plotting source image with cellpose ROIs

        :param roi_labels: if True, label ROIs with cellpose IDs
        :param kwargs:
            :trialobj: ImagingTrial or SingleImage; object associated with input image.
            :scalebar_um: int; size of scalebar to plot on image (in um); must provide trialobj parameter.

        """
        fig, ax = plt.subplots(figsize=(6, 6))
        ax.imshow(self.data, cmap='hot')

        if roi_labels:
            # add labels of ROI numbers:
            for roi in self.roi_ids:
                # roi = self.roi_ids[0]
                coord = (int(np.mean(self.roi_areas[roi][0])),
                         int(np.mean(self.roi_areas[roi][1])))
                ax.text(x=coord[1], y=coord[0], s=f'{roi}', fontweight='bold', va='top', color='green', size='x-small')

        ax.set_title(f'{self.trial_id}')

        if 'scalebar_um' in kwargs and 'trialobj' in kwargs:
            _add_scalebar(ax=ax, **kwargs)

        fig.show()

    def plot_rois(self, roi_labels=True, **kwargs):
        """
        Plot cellpose ROI masks.

        :param roi_labels: if True, label ROIs with cellpose IDs
        :param kwargs:
            :trialobj: ImagingTrial or SingleImage; object associated with input image.
            :scalebar_um: int; size of scalebar to plot on image (in um); must provide trialobj parameter.
        """
        fig, ax = plt.subplots(figsize=(6, 6))
        ax.imshow(self.data, cmap='hot')
        ax.imshow(self.masks, cmap='hot', vmin=0, vmax=1)

        if roi_labels:
            # add labels of ROI numbers:
            for roi in self.roi_ids:
                # roi = self.roi_ids[0]
                coord = (int(np.mean(self.roi_areas[roi][0])),
                         int(np.mean(self.roi_areas[roi][1])))
                ax.text(x=coord[1], y=coord[0], s=f'{roi}', fontweight='bold', va='top', color='green', size='x-small')

        ax.set_title(f'{self.trial_id}')
        if 'scalebar_um' in kwargs and 'trialobj' in kwargs:
            _add_scalebar(ax=ax, **kwargs)

        fig.show()
