# Copyright 2013, Michael Stahn
# Use of this source code is governed by a GPLv2-style license that can be
# found in the LICENSE file.
"""
Checksum logic vor various protocols.
"""
import array
import socket
import logging
import sys
import ctypes
from ctypes import util as ctypes_util
import os

from pypacker.structcbs import unpack_H_le, unpack_H

logger = logging.getLogger("pypacker")

# avoid references for performance reasons
array_array = array.array
ntohs = socket.ntohs
ENDIANNES_IS_BIG = True if sys.byteorder == "big" else False

# TCP (RFC 793) and UDP (RFC 768) checksum


def in_cksum_add(s, buf):
	"""return -- s + checksum value of buf. Adds 0x0 padding to buf if not multiple of 2."""
	buflen = len(buf)
	# logger.debug("buflen for checksum: %d" % n)
	#a = [unpack_H_le(buf[off: off+2])[0] for off in range(0, buflen & ~0x1, 2)] # seems to be slower
	#a = data = numpy_frombuffer(bts, dtype=numpy_int16) # seems to be slower
	a = array_array("H", buf[:buflen & ~0x1])

	if ENDIANNES_IS_BIG:
		a.byteswap()

	if buflen & 0x1 == 0x1:
		# not multiple of 2: add padding
		a.append(unpack_H_le(buf[-1:] + b"\x00")[0])

	return s + sum(a)


def in_cksum_done(s):
	"""Complete checksum building."""
	# Add carry to sum itself
	s = (s >> 16) + (s & 0xFFFF)
	s += (s >> 16)
	# Return complement of sums
	return ntohs((~s) & 0xFFFF)


try:
	#DIR_CHECKSUMPY = os.path.dirname(os.path.realpath(__file__))
	CHECKSUM_NATIVE_LIB_NAME = "checksum_native"
	print(ctypes_util.find_library(CHECKSUM_NATIVE_LIB_NAME))
	logger.debug("Trying to load c based checksum implementation %s" % CHECKSUM_NATIVE_LIB_NAME)
	#chksumlib = ctypes.cdll.LoadLibrary(DIR_CHECKSUMPY + "/checksum_native.so")
	chksumlib = ctypes.cdll.LoadLibrary(CHECKSUM_NATIVE_LIB_NAME)

	in_chksum_c = chksumlib.in_chksum
	in_chksum_c.restype = ctypes.c_uint32
	in_chksum_c.argtypes = ctypes.POINTER(ctypes.c_char), ctypes.c_uint32

	def in_cksum(bts):
		return in_chksum_c(bts, len(bts))
#except Exception as ex:
except:
	logger.debug("Using Python checksum implementation")

	# Python implementation
	def in_cksum(bts):
		"""Return computed Internet Protocol checksum."""
		return in_cksum_done(in_cksum_add(0, bts))


# CRC-32C Checksum
# http://tools.ietf.org/html/rfc3309

crc32c_table = (
	0x00000000, 0xF26B8303, 0xE13B70F7, 0x1350F3F4, 0xC79A971F,
	0x35F1141C, 0x26A1E7E8, 0xD4CA64EB, 0x8AD958CF, 0x78B2DBCC,
	0x6BE22838, 0x9989AB3B, 0x4D43CFD0, 0xBF284CD3, 0xAC78BF27,
	0x5E133C24, 0x105EC76F, 0xE235446C, 0xF165B798, 0x030E349B,
	0xD7C45070, 0x25AFD373, 0x36FF2087, 0xC494A384, 0x9A879FA0,
	0x68EC1CA3, 0x7BBCEF57, 0x89D76C54, 0x5D1D08BF, 0xAF768BBC,
	0xBC267848, 0x4E4DFB4B, 0x20BD8EDE, 0xD2D60DDD, 0xC186FE29,
	0x33ED7D2A, 0xE72719C1, 0x154C9AC2, 0x061C6936, 0xF477EA35,
	0xAA64D611, 0x580F5512, 0x4B5FA6E6, 0xB93425E5, 0x6DFE410E,
	0x9F95C20D, 0x8CC531F9, 0x7EAEB2FA, 0x30E349B1, 0xC288CAB2,
	0xD1D83946, 0x23B3BA45, 0xF779DEAE, 0x05125DAD, 0x1642AE59,
	0xE4292D5A, 0xBA3A117E, 0x4851927D, 0x5B016189, 0xA96AE28A,
	0x7DA08661, 0x8FCB0562, 0x9C9BF696, 0x6EF07595, 0x417B1DBC,
	0xB3109EBF, 0xA0406D4B, 0x522BEE48, 0x86E18AA3, 0x748A09A0,
	0x67DAFA54, 0x95B17957, 0xCBA24573, 0x39C9C670, 0x2A993584,
	0xD8F2B687, 0x0C38D26C, 0xFE53516F, 0xED03A29B, 0x1F682198,
	0x5125DAD3, 0xA34E59D0, 0xB01EAA24, 0x42752927, 0x96BF4DCC,
	0x64D4CECF, 0x77843D3B, 0x85EFBE38, 0xDBFC821C, 0x2997011F,
	0x3AC7F2EB, 0xC8AC71E8, 0x1C661503, 0xEE0D9600, 0xFD5D65F4,
	0x0F36E6F7, 0x61C69362, 0x93AD1061, 0x80FDE395, 0x72966096,
	0xA65C047D, 0x5437877E, 0x4767748A, 0xB50CF789, 0xEB1FCBAD,
	0x197448AE, 0x0A24BB5A, 0xF84F3859, 0x2C855CB2, 0xDEEEDFB1,
	0xCDBE2C45, 0x3FD5AF46, 0x7198540D, 0x83F3D70E, 0x90A324FA,
	0x62C8A7F9, 0xB602C312, 0x44694011, 0x5739B3E5, 0xA55230E6,
	0xFB410CC2, 0x092A8FC1, 0x1A7A7C35, 0xE811FF36, 0x3CDB9BDD,
	0xCEB018DE, 0xDDE0EB2A, 0x2F8B6829, 0x82F63B78, 0x709DB87B,
	0x63CD4B8F, 0x91A6C88C, 0x456CAC67, 0xB7072F64, 0xA457DC90,
	0x563C5F93, 0x082F63B7, 0xFA44E0B4, 0xE9141340, 0x1B7F9043,
	0xCFB5F4A8, 0x3DDE77AB, 0x2E8E845F, 0xDCE5075C, 0x92A8FC17,
	0x60C37F14, 0x73938CE0, 0x81F80FE3, 0x55326B08, 0xA759E80B,
	0xB4091BFF, 0x466298FC, 0x1871A4D8, 0xEA1A27DB, 0xF94AD42F,
	0x0B21572C, 0xDFEB33C7, 0x2D80B0C4, 0x3ED04330, 0xCCBBC033,
	0xA24BB5A6, 0x502036A5, 0x4370C551, 0xB11B4652, 0x65D122B9,
	0x97BAA1BA, 0x84EA524E, 0x7681D14D, 0x2892ED69, 0xDAF96E6A,
	0xC9A99D9E, 0x3BC21E9D, 0xEF087A76, 0x1D63F975, 0x0E330A81,
	0xFC588982, 0xB21572C9, 0x407EF1CA, 0x532E023E, 0xA145813D,
	0x758FE5D6, 0x87E466D5, 0x94B49521, 0x66DF1622, 0x38CC2A06,
	0xCAA7A905, 0xD9F75AF1, 0x2B9CD9F2, 0xFF56BD19, 0x0D3D3E1A,
	0x1E6DCDEE, 0xEC064EED, 0xC38D26C4, 0x31E6A5C7, 0x22B65633,
	0xD0DDD530, 0x0417B1DB, 0xF67C32D8, 0xE52CC12C, 0x1747422F,
	0x49547E0B, 0xBB3FFD08, 0xA86F0EFC, 0x5A048DFF, 0x8ECEE914,
	0x7CA56A17, 0x6FF599E3, 0x9D9E1AE0, 0xD3D3E1AB, 0x21B862A8,
	0x32E8915C, 0xC083125F, 0x144976B4, 0xE622F5B7, 0xF5720643,
	0x07198540, 0x590AB964, 0xAB613A67, 0xB831C993, 0x4A5A4A90,
	0x9E902E7B, 0x6CFBAD78, 0x7FAB5E8C, 0x8DC0DD8F, 0xE330A81A,
	0x115B2B19, 0x020BD8ED, 0xF0605BEE, 0x24AA3F05, 0xD6C1BC06,
	0xC5914FF2, 0x37FACCF1, 0x69E9F0D5, 0x9B8273D6, 0x88D28022,
	0x7AB90321, 0xAE7367CA, 0x5C18E4C9, 0x4F48173D, 0xBD23943E,
	0xF36E6F75, 0x0105EC76, 0x12551F82, 0xE03E9C81, 0x34F4F86A,
	0xC69F7B69, 0xD5CF889D, 0x27A40B9E, 0x79B737BA, 0x8BDCB4B9,
	0x988C474D, 0x6AE7C44E, 0xBE2DA0A5, 0x4C4623A6, 0x5F16D052,
	0xAD7D5351
)


def crc32_add(crc, buf):
	# buf = array.array("B", buf)
	i = 0
	while i < len(buf):
		# crc = (crc >> 8) ^ crc32c_table[(crc ^ b) & 0xFF]
		crc = (crc >> 8) ^ crc32c_table[(crc ^ buf[i]) & 0xFF]
		i += 1
	return crc


def crc32_done(crc):
	tmp = ~crc & 0xFFFFFFFF
	b0 = tmp & 0xFF
	b1 = (tmp >> 8) & 0xFF
	b2 = (tmp >> 16) & 0xFF
	b3 = (tmp >> 24) & 0xFF
	crc = (b0 << 24) | (b1 << 16) | (b2 << 8) | b3
	return crc


def crc32_cksum(buf):
	"""Return computed CRC-32c checksum."""
	return crc32_done(crc32_add(0xFFFFFFFF, buf))


def fletcher32(data_to_checksum, amount_words):
	# 1 word = 2 Bytes
	sum1 = 0xFFFF
	sum2 = 0xFFFF
	datapos = 0

	while amount_words > 0:
		tlen = 359 if amount_words > 359 else amount_words
		amount_words -= tlen

		while tlen > 0:
			# sum1 += unpack_word_be(data_to_checksum[datapos:datapos+2])[0]
			# print("%d" % sum1)
			sum1 += unpack_H(data_to_checksum[datapos: datapos + 2])[0]
			datapos += 2
			sum2 += sum1
			# print("%d" % sum1)
			# print("%d" % sum2)
			# print("--")
			tlen -= 1
		sum1 = (sum1 & 0xFFFF) + (sum1 >> 16)
		sum2 = (sum2 & 0xFFFF) + (sum2 >> 16)
	# Second reduction step to reduce sums to 16 bits
	sum1 = (sum1 & 0xFFFF) + (sum1 >> 16)
	sum2 = (sum2 & 0xFFFF) + (sum2 >> 16)
	return (sum2 << 16) | sum1


crc_table_btle = (
	0x000000, 0x01B4C0, 0x036980, 0x02DD40, 0x06D300, 0x0767C0, 0x05BA80, 0x040E40,
	0x0DA600, 0x0C12C0, 0x0ECF80, 0x0F7B40, 0x0B7500, 0x0AC1C0, 0x081C80, 0x09A840,
	0x1B4C00, 0x1AF8C0, 0x182580, 0x199140, 0x1D9F00, 0x1C2BC0, 0x1EF680, 0x1F4240,
	0x16EA00, 0x175EC0, 0x158380, 0x143740, 0x103900, 0x118DC0, 0x135080, 0x12E440,
	0x369800, 0x372CC0, 0x35F180, 0x344540, 0x304B00, 0x31FFC0, 0x332280, 0x329640,
	0x3B3E00, 0x3A8AC0, 0x385780, 0x39E340, 0x3DED00, 0x3C59C0, 0x3E8480, 0x3F3040,
	0x2DD400, 0x2C60C0, 0x2EBD80, 0x2F0940, 0x2B0700, 0x2AB3C0, 0x286E80, 0x29DA40,
	0x207200, 0x21C6C0, 0x231B80, 0x22AF40, 0x26A100, 0x2715C0, 0x25C880, 0x247C40,
	0x6D3000, 0x6C84C0, 0x6E5980, 0x6FED40, 0x6BE300, 0x6A57C0, 0x688A80, 0x693E40,
	0x609600, 0x6122C0, 0x63FF80, 0x624B40, 0x664500, 0x67F1C0, 0x652C80, 0x649840,
	0x767C00, 0x77C8C0, 0x751580, 0x74A140, 0x70AF00, 0x711BC0, 0x73C680, 0x727240,
	0x7BDA00, 0x7A6EC0, 0x78B380, 0x790740, 0x7D0900, 0x7CBDC0, 0x7E6080, 0x7FD440,
	0x5BA800, 0x5A1CC0, 0x58C180, 0x597540, 0x5D7B00, 0x5CCFC0, 0x5E1280, 0x5FA640,
	0x560E00, 0x57BAC0, 0x556780, 0x54D340, 0x50DD00, 0x5169C0, 0x53B480, 0x520040,
	0x40E400, 0x4150C0, 0x438D80, 0x423940, 0x463700, 0x4783C0, 0x455E80, 0x44EA40,
	0x4D4200, 0x4CF6C0, 0x4E2B80, 0x4F9F40, 0x4B9100, 0x4A25C0, 0x48F880, 0x494C40,
	0xDA6000, 0xDBD4C0, 0xD90980, 0xD8BD40, 0xDCB300, 0xDD07C0, 0xDFDA80, 0xDE6E40,
	0xD7C600, 0xD672C0, 0xD4AF80, 0xD51B40, 0xD11500, 0xD0A1C0, 0xD27C80, 0xD3C840,
	0xC12C00, 0xC098C0, 0xC24580, 0xC3F140, 0xC7FF00, 0xC64BC0, 0xC49680, 0xC52240,
	0xCC8A00, 0xCD3EC0, 0xCFE380, 0xCE5740, 0xCA5900, 0xCBEDC0, 0xC93080, 0xC88440,
	0xECF800, 0xED4CC0, 0xEF9180, 0xEE2540, 0xEA2B00, 0xEB9FC0, 0xE94280, 0xE8F640,
	0xE15E00, 0xE0EAC0, 0xE23780, 0xE38340, 0xE78D00, 0xE639C0, 0xE4E480, 0xE55040,
	0xF7B400, 0xF600C0, 0xF4DD80, 0xF56940, 0xF16700, 0xF0D3C0, 0xF20E80, 0xF3BA40,
	0xFA1200, 0xFBA6C0, 0xF97B80, 0xF8CF40, 0xFCC100, 0xFD75C0, 0xFFA880, 0xFE1C40,
	0xB75000, 0xB6E4C0, 0xB43980, 0xB58D40, 0xB18300, 0xB037C0, 0xB2EA80, 0xB35E40,
	0xBAF600, 0xBB42C0, 0xB99F80, 0xB82B40, 0xBC2500, 0xBD91C0, 0xBF4C80, 0xBEF840,
	0xAC1C00, 0xADA8C0, 0xAF7580, 0xAEC140, 0xAACF00, 0xAB7BC0, 0xA9A680, 0xA81240,
	0xA1BA00, 0xA00EC0, 0xA2D380, 0xA36740, 0xA76900, 0xA6DDC0, 0xA40080, 0xA5B440,
	0x81C800, 0x807CC0, 0x82A180, 0x831540, 0x871B00, 0x86AFC0, 0x847280, 0x85C640,
	0x8C6E00, 0x8DDAC0, 0x8F0780, 0x8EB340, 0x8ABD00, 0x8B09C0, 0x89D480, 0x886040,
	0x9A8400, 0x9B30C0, 0x99ED80, 0x985940, 0x9C5700, 0x9DE3C0, 0x9F3E80, 0x9E8A40,
	0x972200, 0x9696C0, 0x944B80, 0x95FF40, 0x91F100, 0x9045C0, 0x929880, 0x932C40
)

CRC_BTLE_INIT_ADV		= 0x555555
CRC_BTLE_INIT_ADV_REORDERED	= 0xAAAAAA


def crc_btle_check(buf, crc_init):
	"""
	AA + header + data + crc -> crc(header...data) -> compare crc_new <-> crc
	buf -- AA + header + data + crc
	return -- True if crc is correct, False otherwise
	"""
	crc_old = (buf[-1] & 0xFF) << 16 | (buf[-2] & 0xFF) << 8 | (buf[-3] & 0xFF)
	crc_new = crc_btle_update(buf[4:-3], crc_init)

	#logger.debug("crc old/new: %s == %s" % (crc_old, crc_new))
	return crc_old == crc_new


def crc_btle_update(data, crc_in):
	dlen = len(data)
	idx_data = 0

	for idx_data in range(dlen):
		idx_table = (crc_in ^ data[idx_data]) & 0xFF
		crc_in = (crc_table_btle[idx_table] ^ (crc_in >> 8)) & 0xFFFFFF
	return crc_in & 0xFFFFFF


def crc_btle_init_reorder(crc_init):
	# reverse bits: 0110001 -> 1000110
	return int("".join(reversed(bin(crc_init)[2:].rjust(3 * 8, "0"))), 2)
