# Copyright 2021 Patrick C. Tapping
#
# This program is free software: you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program.  If not, see <http:#www.gnu.org/licenses/>.

"""
The ``protocol`` package deals with the encoding and decoding of messages passed between the
host computer and the device, but does not deal with any of the aspects of the transport
layer (for example, the USB serial connection itself).

The majority of this module contains constant definitions autogenerated from the firmware's
``TRSInterface.h`` c header file.

These definitions can aid in writing more readable code, for example:

.. code-block:: python

    from trs_interface import TRSInterface
    from trs_interface.protocol import *

    trsi = TRSInterface()
    trsi.camera_trig_polarity = Polarity.HIGH
    # ...
    if msg.id == ID.GOT_UNKNOWN_MESSAGE:
        raise RuntimeError("Device received an unknown message.")


"""

from enum import IntEnum

HEADER_SIZE = 10
"""Size (in bytes) of a message header, or a basic (header only) message type."""

# MSB indicates extended message type, where data field encodes payload size
LONG_FORM = 0x8000
"""
The most significant bit of a message ID indicates whether the message is a "long form" message,
which contains additional "payload" data.
This is a bit mask which can be used to test for long form messages by way of a logical AND operation.
"""

class ID(IntEnum):
    """
    Enumeration of known message IDs.
    """
    GET_PING = 0x0000
    GOT_PING = 0x0001
    GOT_UNKNOWN_MSG = 0x0011
    GET_VERSION = 0x0020
    GOT_VERSION = 0x0021 | LONG_FORM
    STORE_SETTINGS = 0x0802
    GET_LASER_SYNC_POLARITY = 0x1000
    GOT_LASER_SYNC_POLARITY = 0x1001
    SET_LASER_SYNC_POLARITY = 0x1002
    GET_CHOPPER_SYNCIN_POLARITY = 0x1010
    GOT_CHOPPER_SYNCIN_POLARITY = 0x1011
    SET_CHOPPER_SYNCIN_POLARITY = 0x1012
    GET_CHOPPER_SYNCOUT_POLARITY = 0x1020
    GOT_CHOPPER_SYNCOUT_POLARITY = 0x1021
    SET_CHOPPER_SYNCOUT_POLARITY = 0x1022
    GET_DELAY_TRIG_POLARITY = 0x1030
    GOT_DELAY_TRIG_POLARITY = 0x1031
    SET_DELAY_TRIG_POLARITY = 0x1032
    GET_CAMERA_TRIG_POLARITY = 0x1050
    GOT_CAMERA_TRIG_POLARITY = 0x1051
    SET_CAMERA_TRIG_POLARITY = 0x1052
    GET_QUADRATURE_POLARITY = 0x1060
    GOT_QUADRATURE_POLARITY = 0x1061
    SET_QUADRATURE_POLARITY = 0x1062
    GET_QUADRATURE_DIRECTION = 0x1070
    GOT_QUADRATURE_DIRECTION = 0x1071
    SET_QUADRATURE_DIRECTION = 0x1072
    GET_CHOPPER_SYNC_DELAY = 0x1100
    GOT_CHOPPER_SYNC_DELAY = 0x1101
    SET_CHOPPER_SYNC_DELAY = 0x1102
    GET_CHOPPER_SYNC_DURATION = 0x1110
    GOT_CHOPPER_SYNC_DURATION = 0x1111
    SET_CHOPPER_SYNC_DURATION = 0x1112
    GET_CAMERA_SYNC_DELAY = 0x1120
    GOT_CAMERA_SYNC_DELAY = 0x1121
    SET_CAMERA_SYNC_DELAY = 0x1122
    GET_CAMERA_SYNC_DURATION = 0x1130
    GOT_CAMERA_SYNC_DURATION = 0x1131
    SET_CAMERA_SYNC_DURATION = 0x1132
    GET_CHOPPER_DIVIDER = 0x1200
    GOT_CHOPPER_DIVIDER = 0x1201
    SET_CHOPPER_DIVIDER = 0x1202
    GET_QUADRATURE_VALUE = 0x1210
    GOT_QUADRATURE_VALUE = 0x1211
    SET_QUADRATURE_VALUE = 0x1212
    TRIGGER = 0x2004
    ARM = 0x2008
    START = 0x2018
    STOP = 0x2019
    GOT_DATA = 0x4001 | LONG_FORM
    GET_LASER_SYNC_PERIOD = 0x4100
    GOT_LASER_SYNC_PERIOD = 0x4101

class Polarity(IntEnum):
    """
    Enumeration of signal or edge polarities.
    """
    LOW = 0
    HIGH = 1
    FALLING = 2
    RISING = 3


class Direction(IntEnum):
    """
    Enumeration of quadrature counting direction.
    """
    FORWARD = 0
    REVERSE = 1