"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssociationManager = void 0;
const cdk = require("aws-cdk-lib");
const aws_servicecatalog_1 = require("aws-cdk-lib/aws-servicecatalog");
const util_1 = require("./util");
const validation_1 = require("./validation");
class AssociationManager {
    static associateProductWithPortfolio(portfolio, product, options) {
        validation_1.InputValidator.validateLength(this.prettyPrintAssociation(portfolio, product), 'description', 0, 2000, options === null || options === void 0 ? void 0 : options.description);
        const associationKey = util_1.hashValues(portfolio.node.addr, product.node.addr, product.stack.node.addr);
        const constructId = `PortfolioProductAssociation${associationKey}`;
        const existingAssociation = portfolio.node.tryFindChild(constructId);
        const cfnAssociation = existingAssociation
            ? existingAssociation
            : new aws_servicecatalog_1.CfnPortfolioProductAssociation(portfolio, constructId, {
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
            });
        return {
            associationKey: associationKey,
            cfnPortfolioProductAssociation: cfnAssociation,
        };
    }
    static constrainTagUpdates(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `ResourceUpdateConstraint${association.associationKey}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnResourceUpdateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                tagUpdateOnProvisionedProduct: options.allow === false ? 'NOT_ALLOWED' : 'ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot have multiple tag update constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static notifyOnStackEvents(portfolio, product, topic, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchNotificationConstraint${util_1.hashValues(topic.node.addr, topic.stack.node.addr, association.associationKey)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnLaunchNotificationConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                notificationArns: [topic.topicArn],
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Topic ${topic.node.path} is already subscribed to association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static constrainCloudFormationParameters(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchTemplateConstraint${util_1.hashValues(association.associationKey, options.rule.ruleName)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnLaunchTemplateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                rules: this.formatTemplateRule(portfolio.stack, options.rule),
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Provisioning rule ${options.rule.ruleName} already configured on association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static setLaunchRole(portfolio, product, launchRole, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            roleArn: launchRole.roleArn,
        });
    }
    static setLocalLaunchRoleName(portfolio, product, launchRoleName, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            localRoleName: launchRoleName,
        });
    }
    static deployWithStackSets(portfolio, product, options) {
        var _a;
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a launch role has already been set.
        if (portfolio.node.tryFindChild(this.launchRoleConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot configure StackSet deployment when a launch role is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.stackSetConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnStackSetConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: (_a = options.description) !== null && _a !== void 0 ? _a : '',
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                accountList: options.accounts,
                regionList: options.regions,
                adminRole: options.adminRole.roleArn,
                executionRole: options.executionRoleName,
                stackInstanceControl: options.allowStackSetInstanceOperations ? 'ALLOWED' : 'NOT_ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot configure multiple StackSet deployment constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static associateTagOptions(resource, resourceId, tagOptions) {
        const resourceStack = cdk.Stack.of(resource);
        for (const [key, tagOptionsList] of Object.entries(tagOptions.tagOptionsMap)) {
            validation_1.InputValidator.validateLength(resource.node.addr, 'TagOption key', 1, 128, key);
            tagOptionsList.forEach((value) => {
                validation_1.InputValidator.validateLength(resource.node.addr, 'TagOption value', 1, 256, value);
                const tagOptionKey = util_1.hashValues(key, value, resourceStack.node.addr);
                const tagOptionConstructId = `TagOption${tagOptionKey}`;
                let cfnTagOption = resourceStack.node.tryFindChild(tagOptionConstructId);
                if (!cfnTagOption) {
                    cfnTagOption = new aws_servicecatalog_1.CfnTagOption(resourceStack, tagOptionConstructId, {
                        key: key,
                        value: value,
                        active: true,
                    });
                }
                const tagAssocationKey = util_1.hashValues(key, value, resource.node.addr);
                const tagAssocationConstructId = `TagOptionAssociation${tagAssocationKey}`;
                if (!resource.node.tryFindChild(tagAssocationConstructId)) {
                    new aws_servicecatalog_1.CfnTagOptionAssociation(resource, tagAssocationConstructId, {
                        resourceId: resourceId,
                        tagOptionId: cfnTagOption.ref,
                    });
                }
            });
        }
        ;
    }
    static setLaunchRoleConstraint(portfolio, product, options, roleOptions) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a stackset deployment constraint has already been configured.
        if (portfolio.node.tryFindChild(this.stackSetConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot set launch role when a StackSet rule is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.launchRoleConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnLaunchRoleConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                roleArn: roleOptions.roleArn,
                localRoleName: roleOptions.localRoleName,
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot set multiple launch roles for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static stackSetConstraintLogicalId(associationKey) {
        return `StackSetConstraint${associationKey}`;
    }
    static launchRoleConstraintLogicalId(associationKey) {
        return `LaunchRoleConstraint${associationKey}`;
    }
    static prettyPrintAssociation(portfolio, product) {
        return `- Portfolio: ${portfolio.node.path} | Product: ${product.node.path}`;
    }
    static formatTemplateRule(stack, rule) {
        return JSON.stringify({
            [rule.ruleName]: {
                Assertions: this.formatAssertions(stack, rule.assertions),
                RuleCondition: rule.condition ? stack.resolve(rule.condition) : undefined,
            },
        });
    }
    static formatAssertions(stack, assertions) {
        return assertions.reduce((formattedAssertions, assertion) => {
            formattedAssertions.push({
                Assert: stack.resolve(assertion.assert),
                AssertDescription: assertion.description,
            });
            return formattedAssertions;
        }, new Array());
    }
    ;
}
exports.AssociationManager = AssociationManager;
//# sourceMappingURL=data:application/json;base64,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