"""This is the utilities module of the expansion package. Contains miscellaneous utilities, i.e.
functions and classes that are not specific to the expansion package.
"""

__version__ = '1.0.1'
__author__ = 'Rajarshi Mandal'
__all__ = ['Timer',
           'stitch']

import glob
import os
import time

import cv2


class Timer:
    """Once instantiated, can be called to time a function.

    Args:
        func (callable): Function to be timed.

    Attributes:
        func (callable): Function to be timed.
        result (any): Result of function passed to timer.
    """
    def __init__(self, func):
        self.func = func
        self._result = None
        self._timed = False

    @property
    def result(self):
        """any: Result of function passed to Timer, only accesible once Timer instance has been
        called.

        Raises:
            AttributeError: If accessed before Timer instance has been called.
        """
        if self._timed:
            return self._result

        raise AttributeError('Timer.result has not been evaluated yet!')

    def __call__(self, *args, **kwargs):
        """Times a function with an arbitrary number of arguments.

        Args:
            *args: Positional arguments to be passed to the function.
            **kwargs: Keyword arguments to be passed to the function.

        Returns:
            float: Time elapsed, from before calling the function to after calling the function,
                in seconds.
        """
        old_time = time.time()
        self._result = self.func(*args, **kwargs)
        new_time = time.time()

        self._timed = True

        time_elapsed = new_time - old_time

        return time_elapsed

def stitch(directory, name, fps, dim, f_format, fourcc, frame_ff): # pylint: disable=too-many-arguments
    """Generates a video with a given fourcc Uses frames in a directory generated by the
    :class:`expansion.callbacks.Sample` callback used during
    :class:`expansion.ColoredPointHandler.simulate`. Frames must be numbered, e.g. '1.png'.

    Args:
        directory (str): Directory of the frames.
        name (str): Desired name of the video.
        fps (int): Desired frame rate of the video.
        dim (tuple(int)): A tuple consisting of two integers, representing the dimensions of each
            frame.
        f_format (str): Desired file format of the video, e.g. 'mp4'.
        fourcc (str): Desired fourcc of the video, e.g. 'h264'.
            You can visit https://www.fourcc.org/ for a detailed list of fourcc codes. Note: you
            must have the desired fourcc installed on your system to generate a video with that
            particular fourcc.
        frame_ff (str): File format of frames, e.g. 'png'.
    """
    # pylint: disable=no-member

    video = cv2.VideoWriter(f'{directory}{os.path.sep}{name}.{f_format}',
                            cv2.VideoWriter_fourcc(*fourcc), fps, dim)

    for image in sorted(glob.iglob('[0-9].{frame_ff}'),
                        key=lambda f_name: int(f_name[:-len(frame_ff)-1])):
        video.write(cv2.imread(image))

    video.release()

    # pylint: enable=no-member
