# Copyright 2014-2020 Scalyr Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Module responsible for parsing data from build_info file which is available with each agent package.

Keep in mind that that file is only available for package and not for dev installs.
"""

from __future__ import unicode_literals
from __future__ import absolute_import

if False:  # NOSONAR
    from typing import Dict

import os
import platform
from io import open

import six

from scalyr_agent.compat import subprocess_check_output

BASE_DIR = os.path.abspath(os.path.dirname(__file__))

IS_WINDOWS = platform.system() == "Windows"

# Directory structure for packages looks like this:
# /usr/share/scalyr-agent-2/py/scalyr_agent/build_info.py
# /usr/share/scalyr-agent-2/build_info
BUILD_INFO_PATH_LINUX = os.path.abspath(os.path.join(BASE_DIR, "../../build_info"))

# On Windows, it's stored under bin/ directory and there we also use binary so we can't work with
# relative paths
BUILD_INFO_PATH_WINDOWS = "C:\\Program Files (x86)\\Scalyr\\bin\\build_info"

GIT_GET_HEAD_REVISION_CMD = "git rev-parse HEAD"


def _get_build_info_linux():
    # type: () -> Dict[str, str]
    if not os.path.isfile(BUILD_INFO_PATH_LINUX):
        return {}

    with open(BUILD_INFO_PATH_LINUX, "r") as fp:
        content = fp.read()

    return _parse_build_info_content(content)


def _get_build_info_windows():
    # type: () -> Dict[str, str]
    if not os.path.isfile(BUILD_INFO_PATH_WINDOWS):
        return {}

    with open(BUILD_INFO_PATH_WINDOWS, "r") as fp:
        content = fp.read()

    return _parse_build_info_content(content)


def _parse_build_info_content(content):
    # type: (str) -> Dict[str, str]
    result = {}
    for line in content.strip().split("\n"):
        line = line.strip()

        split = line.split(":", 1)

        if len(split) != 2:
            continue

        key, value = split
        key = key.strip().lower().replace(" ", "_")
        value = value.strip()

        result[key] = value

    return result


def get_build_info():
    # type: () -> Dict[str, str]
    """
    Return sanitized dictionary populated with data from build_info file.
    """
    if IS_WINDOWS:
        return _get_build_info_windows()
    else:
        return _get_build_info_linux()


def get_build_revision_from_git():
    # type: () -> str
    """
    Return build revision from git ref log (if available).

    NOTE: This function is only used on dev (non-package) installs.
    """
    cmd = GIT_GET_HEAD_REVISION_CMD

    try:
        output = subprocess_check_output(cmd, shell=True)
    except Exception:
        return "unknown"

    return six.ensure_text(output).strip()


def get_build_revision():
    # type: () -> str
    """
    This function retrieves git commit which was used to build the agent package from the
    build_info file which is generated by build_package.py script.

    If we are running on a dev install, it retrieves the commit revision by querying git reflog
    instead.
    """
    # NOTE: We use lazy import to avoid import time side affects
    try:
        from __scalyr__ import DEV_INSTALL, INSTALL_TYPE
    except ImportError:
        from scalyr_agent.__scalyr__ import DEV_INSTALL, INSTALL_TYPE  # type: ignore

    if INSTALL_TYPE == DEV_INSTALL:
        if IS_WINDOWS and not os.path.isfile(BUILD_INFO_PATH_WINDOWS):
            return get_build_revision_from_git()
        elif not os.path.isfile(BUILD_INFO_PATH_LINUX):
            return get_build_revision_from_git()

    build_info = get_build_info()
    return build_info.get("latest_commit", "unknown")
