# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pelican', 'pelican.plugins.image_process']

package_data = \
{'': ['*'],
 'pelican.plugins.image_process': ['test_data/*',
                                   'test_data/exif/*',
                                   'test_data/results/blur/*',
                                   'test_data/results/contour/*',
                                   'test_data/results/crop/*',
                                   'test_data/results/detail/*',
                                   'test_data/results/edge_enhance/*',
                                   'test_data/results/edge_enhance_more/*',
                                   'test_data/results/emboss/*',
                                   'test_data/results/find_edges/*',
                                   'test_data/results/flip_horizontal/*',
                                   'test_data/results/flip_vertical/*',
                                   'test_data/results/grayscale/*',
                                   'test_data/results/resize/*',
                                   'test_data/results/rotate/*',
                                   'test_data/results/scale_in/*',
                                   'test_data/results/scale_out/*',
                                   'test_data/results/sharpen/*',
                                   'test_data/results/smooth/*',
                                   'test_data/results/smooth_more/*']}

install_requires = \
['Pillow>=8.0,<9.0',
 'beautifulsoup4>=4.9,<5.0',
 'lxml>=4.6,<5.0',
 'pelican>=3,<5',
 'six>=1.15.0,<2.0.0']

extras_require = \
{'markdown': ['markdown>=3.2,<4.0']}

setup_kwargs = {
    'name': 'pelican-image-process',
    'version': '2.1.2',
    'description': 'Pelican plugin that automates image processing.',
    'long_description': '# Image Process: A Plugin for Pelican\n\n[![Build Status](https://img.shields.io/github/workflow/status/pelican-plugins/image-process/build)](https://github.com/pelican-plugins/image-process/actions)\n[![PyPI Version](https://img.shields.io/pypi/v/pelican-image-process)](https://pypi.org/project/pelican-image-process/)\n![License](https://img.shields.io/badge/license-AGPL--3.0-blue)\n\n*Image Process* is a plugin for [Pelican](https://getpelican.com),\na static site generator written in Python.\n\n*Image Process* let you automate the processing of images based on their\nclass attribute. Use this plugin to minimize the overall page weight\nand to save you a trip to Gimp or Photoshop each time you include an\nimage in your post.\n\n*Image Process* also makes it easy to create responsive images using\nthe HTML5 `srcset` attribute and `<picture>` tag. It does this\nby generating multiple derivative images from one or more sources.\n\n*Image Process* will not overwrite your original images.\n\n## Installation\n\nThe easiest way to install *Image Process* is via Pip. This\nwill also install the required dependencies automatically.\n\n```sh\npython -m pip install pelican-image-process\n```\n\nYou will then need to configure your desired transformations (see *Usage*\nbelow) and add the appropriate class to images you want processed.\n\n## Usage\n\n*Image Process* scans your content for `<img>` tags with special\nclasses. It then maps the classes to a set of image processing\ninstructions, computes new images, and modifies HTML code according to\nthe instructions.\n\n### Define Transformations\n\nThe first step in using this module is to define some image\ntransformations in your Pelican configuration file. Transformations\nare defined in the `IMAGE_PROCESS` dictionary, mapping a\ntransformation name to a list of operations. There are three kinds of\ntransformations: image replacement, responsive image, and picture set.\n\n#### Image Replacement\n\nThe simplest image processing will replace the original image by a\nnew, transformed image computed from the original. You may use this,\nfor example, to ensure that all images are of the same size, or to\ncompute a thumbnail from a larger image:\n\n```python\nIMAGE_PROCESS = {\n    "article-image": ["scale_in 300 300 True"],\n    "thumb": ["crop 0 0 50% 50%", "scale_out 150 150 True", "crop 0 0 150 150"],\n}\n```\n\nThese transformations tell *Image Process* to transform the image\nreferred to by the `src` attribute of an `<img>` according to the\nlist of operations specified, and replace the `src` attribute with the\nURL of the transformed image.\n\nFor consistency with other types of transformations described\nbelow, there is an alternative syntax for the processing instructions:\n\n```python\nIMAGE_PROCESS = {\n    "thumb": {\n        "type": "image",\n        "ops": ["crop 0 0 50% 50%", "scale_out 150 150 True", "crop 0 0 150 150"],\n    },\n    "article-image": {\n        "type": "image",\n        "ops": ["scale_in 300 300 True"],\n    },\n}\n```\n\nTo apply image replacement to the images in your articles, you must add to them\nthe special class `image-process-<transform>`, in which `<transform>` is the ID\nof the transformation you wish to apply.\n\nLet\'s say you have defined the transformation described above. To get your\nimage processed, it needs to have the right CSS class:\n\n```html\n<img class="image-process-article-image" src="/images/pelican.jpg"/>\n```\n\nThis can be produced in Markdown with:\n\n```markdown\n![](/images/pelican.png){: .image-process-article-image}\n```\n\nIn reStructuredText, use the `:class:` attribute of the `image` or\nthe `figure` directive:\n\n```rst\n.. image:: /images/pelican.png\n   :class: image-process-article-image\n```\n\n```rst\n.. figure:: /images/pelican.png\n    :class: image-process-article-image\n```\n\n⚠️ **Warning:**\n\n> The reStructuredText reader will convert underscores (`_`) to\n> dashes (`-`) in class names. To make sure everything runs\n> smoothly, do not use underscores in your transformation names.\n\n\n#### Responsive Images\n\nYou can use *Image Process* to automatically generate a set of\nimages that will be selected for display by browsers according to the\nviewport width or according to the device resolution. To accomplish\nthis, *Image Process* will add a [`srcset` attribute](https://caniuse.com/srcset)\n(and maybe a `media` and a `sizes` attribute) to the `<img>` tag.\n\nHTML5 supports two types of responsive image sets. The first one is\ndevice-pixel-ratio-based, selecting higher resolution images for higher\nresolution devices; the second one is viewport-based, selecting\nimages according to the viewport size. You can read more about\n[HTML5 responsive images][] for a gentle introduction to the `srcset`\nand `<picture>` syntaxes.\n\nTo tell *Image Process* to generate a responsive image, add a\n`responsive-image` transformation to your your `IMAGE_PROCESS`\ndictionary, with the following syntax:\n\n```python\nIMAGE_PROCESS = {\n    "crisp": {\n        "type": "responsive-image",\n        "srcset": [\n            ("1x", ["scale_in 800 600 True"]),\n            ("2x", ["scale_in 1600 1200 True"]),\n            ("4x", ["scale_in 3200 2400 True"]),\n        ],\n        "default": "1x",\n    },\n    "large-photo": {\n        "type": "responsive-image",\n        "sizes": (\n            "(min-width: 1200px) 800px, "\n            "(min-width: 992px) 650px, "\n            "(min-width: 768px) 718px, "\n            "100vw"\n        ),\n        "srcset": [\n            ("600w", ["scale_in 600 450 True"]),\n            ("800w", ["scale_in 800 600 True"]),\n            ("1600w", ["scale_in 1600 1200 True"]),\n        ],\n        "default": "800w",\n    },\n}\n```\n\nThe `crisp` transformation is an example of a transformation\nenabling device-pixel-ratio-based selection. The `srcset` is a list\nof tuples, each tuple containing the image description (`"1x"`,\n`"2x"`, etc.) and the list of operations to generate the derivative\nimage from the original image (the original image is the value of the\n`src` attribute of the `<img>`). Image descriptions are hints\nabout the resolution of the associated image and must have the suffix\n`x`. The `default` setting specifies the image to use to replace the `src`\nattribute of the image.  This is the image that will be displayed by\nbrowsers that do not support the `srcset` syntax.\n\nThe `large-photo` transformation is an example of a transformation\nenabling viewport-based selection. The `sizes` contains a rule to\ncompute the width of the displayed image from the width of the\nviewport. Once the browser knows the image width, it will select an\nimage source from the `srcset`. The `srcset` is a list of tuple,\neach tuple containing the image description (`"600w"`, `"800w"`,\netc.) and the list of operations to generate the derivative image from\nthe original image (the original image is the value of the `src`\nattribute of the `<img>`). Image descriptions are hints about the\nwidth in pixels of the associated image and must have the suffix\n`w`. The `default` setting specifies the image to use to replace the `src`\nattribute of the image.  This is the image that will be displayed by\nbrowsers that do not support the `srcset` syntax.\n\nIn the two examples above, the `default` setting is a string referring to\none of the images in the `srcset`. However, the `default` value\ncould also be a list of operations to generate a different derivative\nimage.\n\nTo make the images in your article responsive, you must add to them the\nspecial class `image-process-<transform>`, in which `<transform>` is the ID of the\ntransformation you wish to apply, exactly like you would do for the\nimage replacement case, described above.\n\nSo, in HTML it should look like this:\n\n```html\n<img class="image-process-large-photo" src="/images/pelican.jpg"/>\n```\n\nWhich can be produced in Markdown with:\n\n```markdown\n![](/images/pelican.jpg){: .image-process-large-photo}\n```\n\nIn reStructuredText, use the `:class:` attribute of the `image` or\nthe `figure` directive:\n\n```rst\n.. image:: /images/pelican.jpg\n   :class: image-process-large-photo\n```\n\n```rst\n.. figure:: /images/pelican.jpg\n    :class: image-process-large-photo\n```\n\n#### Picture Set\n\n*Image Process* can be used to generate the images used by a\n`<picture>` tag. The `<picture>` syntax allows for more\nflexibility in providing a choice of image to the browser.\nAgain, you can read more about [HTML5 responsive images][] for a\ngentle introduction to the `srcset` and `<picture>` syntaxes.\n\nTo tell *Image Process* to generate the images for a `<picture>`,\nadd a `picture` entry to your `IMAGE_PROCESS` dictionary with the\nfollowing syntax:\n\n```python\nIMAGE_PROCESS = {\n    "example-pict": {\n        "type": "picture",\n        "sources": [\n            {\n                "name": "default",\n                "media": "(min-width: 640px)",\n                "srcset": [\n                    ("640w", ["scale_in 640 480 True"]),\n                    ("1024w", ["scale_in 1024 683 True"]),\n                    ("1600w", ["scale_in 1600 1200 True"]),\n                ],\n                "sizes": "100vw",\n            },\n            {\n                "name": "source-1",\n                "srcset": [\n                    ("1x", ["crop 100 100 200 200"]),\n                    ("2x", ["crop 100 100 300 300"]),\n                ]\n            },\n        ],\n        "default": ("default", "640w"),\n    },\n}\n```\n\nEach of the `sources` entries is very similar to the `responsive\nimage` describe above. Here, each source must have a `name`, which\nwill be used to find the URL of the original image for this source in\nyour article. The source may also have a `media`, which contains a\nrule used by the browser to select the active source. The `default`\nsetting specifies the image to use to replace the `src` attribute of\nthe `<img>` inside the `<picture>`.  This is the image that will be\ndisplayed by browsers that do not support the `<picture>` syntax. In\nthis example, it will use the image `640w` from the source `default`.\nA list of operations could have been specified instead of `640w`.\n\nTo generate a responsive `<picture>` for the images in your\narticles, you must add to your article a pseudo `<picture>` tag that\nlooks like this:\n\n```html\n<picture>\n    <source class="source-1" src="/images/pelican-closeup.jpg"></source>\n    <img class="image-process-example-pict" src="/images/pelican.jpg"/>\n</picture>\n```\n\nEach `<source>` tag maps a source name (the `class` attribute) to\na file (the `src` attribute). The `<img>` must have the special\nclass `image-process-` followed by the name of the transformation\nyou wish to apply. The file referenced by the `src` attribute of the\n`<img>` will be used as the special `default` source in your\ntransformation definition.\n\nYou can\'t produce this with pure Markdown and must instead resort to raw HTML.\n\nIn reStructuredText, however, you can also use the `figure` directive\nto generate a `<picture>`. The figure image file will be used as the\nspecial `default` source; other sources must be added in the legend\nsection of the `figure` as `image` directives. The figure class must\nbe `image-process-` followed by the name of the transformation you\nwish to apply, while the other images must have two classes:\n`image-process` and the name of the source they provide an image for:\n\n```rst\n.. figure:: /images/pelican.jpg\n   :class: image-process-example-pict\n\n    Test picture\n\n    .. image:: /images/pelican-closeup.jpg\n       :class: image-process source-1\n```\n\nThe images in the legend section that are used as source for the\n`<picture>` will be removed from the image legend, so that they do\nnot appear in your final article.\n\n### Transformations\n\nAvailable operations for transformations are:\n\n* `crop <top> <left> <right> <bottom>`:\n\n    Crop the image to the box (`<left>`, `<top>`)-(`<right>`, `<bottom>`). Values\n    can be absolute (a number) or relative to the size of the image (a\n    number followed by a percent sign `%`).\n\n* `flip_horizontal`:\n\n    Flip the image horizontally.\n\n* `flip_vertical`:\n\n    Flip the image vertically.\n\n* `grayscale`:\n\n    Convert the image to grayscale.\n\n* `resize <width> <height>`:\n\n    Resize the image. This operation does *not* preserve the image aspect\n    ratio. Values can be absolute (a number) or relative to the\n    size of the image (a number followed by a percent sign `%`).\n\n* `rotate <degrees>`:\n\n    Rotate the image.\n\n* `scale_in <width> <height> <upscale>`:\n\n    Resize the image. This operation preserves the image aspect ratio\n    and the resulting image will be no larger than `<width>` x\n    `<height>`. Values can be absolute (a number) or relative to the\n    size of the image (a number followed by a percent sign `%`).\n    If `<upscale>` is `False`, smaller images will not be enlarged.\n\n* `scale_out <width> <height> <upscale>`:\n\n    Resize the image. This operation preserves the image aspect ratio\n    and the resulting image will be no smaller than `<width>` x\n    `<height>`. Values can be absolute (a number) or relative to the\n    size of the image (a number followed by a percent sign `%`).\n    If `<upscale>` is `False`, smaller images will not be enlarged.\n\n* `blur`:\n\n    Apply the `pillow.ImageFilter.BLUR` filter to the image.\n\n* `contour`:\n\n    Apply the `pillow.ImageFilter.CONTOUR` filter to the image.\n\n* `detail`:\n\n    Apply the `pillow.ImageFilter.DETAIL` filter to the image.\n\n* `edge_enhance`:\n\n    Apply the `pillow.ImageFilter.EDGE_ENHANCE` filter to the image.\n\n* `edge_enhance_more`:\n\n    Apply the `pillow.ImageFilter.EDGE_ENHANCE_MORE` filter to the image.\n\n* `emboss`:\n\n    Apply the `pillow.ImageFilter.EMBOSS` filter to the image.\n\n* `find_edges`:\n\n    Apply the `pillow.ImageFilter.FIND_EDGES` filter to the image.\n\n* `smooth`:\n\n    Apply the `pillow.ImageFilter.SMOOTH filter` to the image.\n\n* `smooth_more`:\n\n    Apply the `pillow.ImageFilter.SMOOTH_MORE` filter to the image.\n\n* `sharpen`:\n\n    Apply the `pillow.ImageFilter.SHARPEN` filter to the image.\n\nYou can also define your own operations; the only requirement is that\nyour operation should be a callable object expecting a `pillow.Image` as\nits first parameter and it should return the transformed image:\n\n```python\ndef crop_face(image):\n    """Detect face in image and crop around it."""\n    # Fancy algorithm.\n    return image\n\nIMAGE_PROCESS = {\n    "face-thumbnail": [crop_face, "scale_out 150 150 True"]\n}\n```\n\n### Additional Settings\n\n#### Destination Directory\n\nBy default, the new images will be stored in a directory named\n`derivative/<TRANSFORMATION_NAME>` in the output folder at\nthe same location as the original image.\nFor example, if the original image is located in\nthe `content/images` folder, the computed images will be stored\nin `output/images/derivative/<TRANSFORMATION_NAME>`.\nAll the transformations are done in the output directory in order\nto avoid confusion with the source files or if we test multiple\ntransformations. You can replace `derivative` by something else using\nthe `IMAGE_PROCESS_DIR` setting in your Pelican settings file:\n\n```python\nIMAGE_PROCESS_DIR = "derivees"\n```\n\n#### Force Image Processing\n\nIf the transformed image already exists and is newer than the original\nimage, the plugin assumes that it should not re-compute it again. You\ncan force the plugin to re-compute all images by setting\n`IMAGE_PROCESS_FORCE` to `True` in your Pelican configuration file.\n\n```python\nIMAGE_PROCESS_FORCE = True\n```\n\n#### Selecting a HTML Parser\n\nYou may select the HTML parser which is used. The default is the built-in\n`html.parser` but you may also select `html5lib` or `lxml` by setting\n`IMAGE_PROCESS_PARSER` in your Pelican settings file. For example:\n\n```python\nIMAGE_PROCESS_PARSER = "html5lib"\n```\n\nFor details, refer to the [BeautifulSoup documentation on parsers][].\n\n#### File Encoding\n\nYou may select a different file encoding to be used by BeautifulSoup as it\nopens your files. The default is `utf-8`.\n\n```python\nIMAGE_PROCESS_ENCODING = "utf-8"\n```\n\n#### Copying EXIF Tags\n\nYou may ask _Image Process_ to copy the EXIF tags from your original image to\nthe transformed images. You must have [exiftool](https://exiftool.org/) installed.\n\n```python\nIMAGE_PROCESS_COPY_EXIF_TAGS = True\n```\n\n## Known Issues\n\n* Pillow, when resizing animated GIF files, [does not return an animated file](https://github.com/pelican-plugins/image-process/issues/11).\n\n## Contributing\n\nContributions are welcome and much appreciated. Every little bit helps. You can contribute by improving the documentation, adding missing features, and fixing bugs. You can also help out by reviewing and commenting on [existing issues][].\n\nTo start contributing to this plugin, review the [Contributing to Pelican][] documentation, beginning with the **Contributing Code** section.\n\n[existing issues]: https://github.com/pelican-plugins/image-process/issues\n[Contributing to Pelican]: https://docs.getpelican.com/en/latest/contribute.html\n\n## License\n\nThis project is licensed under the [AGPL-3.0 license](http://www.gnu.org/licenses/agpl-3.0.html).\n\n[Pelican image](https://web.archive.org/web/20090505115626/http://www.pdphoto.org/PictureDetail.php?mat=&pg=5726) in test data by Jon Sullivan. Published under a [Creative Commons Public Domain license](https://creativecommons.org/licenses/publicdomain/).\n\n\n[HTML5 responsive images]: https://www.smashingmagazine.com/2014/05/14/responsive-images-done-right-guide-picture-srcset/\n[BeautifulSoup documentation on parsers]: https://www.crummy.com/software/BeautifulSoup/bs4/doc/#installing-a-parser\n',
    'author': 'Pelican Dev Team',
    'author_email': 'authors@getpelican.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pelican-plugins/image-process',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.2,<4.0',
}


setup(**setup_kwargs)
