"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipelineMergeAction = void 0;
const aws_codepipeline_1 = require("@aws-cdk/aws-codepipeline");
const aws_codepipeline_actions_1 = require("@aws-cdk/aws-codepipeline-actions");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const merge_branches_function_1 = require("./merge-branches-function");
/**
 * Represents a reference to a CodePipelineMergeAction.
 */
class CodePipelineMergeAction extends aws_codepipeline_actions_1.Action {
    constructor(props) {
        super({
            ...props,
            category: aws_codepipeline_1.ActionCategory.INVOKE,
            provider: 'Lambda',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 0,
                minOutputs: 0,
                maxOutputs: 0,
            },
        });
        this.props = props;
    }
    bound(scope, _stage, options) {
        const { repository, sourceCommitSpecifier, destinationCommitSpecifier, crossAccountRole } = this.props;
        const mergeBranchesFunction = new merge_branches_function_1.MergeBranchesFunction(scope, 'MergeBranchesFunction', {
            repository,
            crossAccountRole,
        });
        // allow pipeline to list functions
        options.role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:ListFunctions'],
            resources: ['*'],
        }));
        // allow pipeline to invoke this lambda functionn
        options.role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [mergeBranchesFunction.functionArn],
        }));
        // allow lambda to put job results for this pipeline
        // CodePipeline requires this to be granted to '*'
        // (the Pipeline ARN will not be enough)
        mergeBranchesFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: ['codepipeline:PutJobSuccessResult', 'codepipeline:PutJobFailureResult'],
        }));
        return {
            configuration: {
                FunctionName: mergeBranchesFunction.functionName,
                UserParameters: core_1.Stack.of(scope).toJsonString({
                    repositoryName: repository.repositoryName,
                    sourceCommitSpecifier,
                    destinationCommitSpecifier,
                }),
            },
        };
    }
}
exports.CodePipelineMergeAction = CodePipelineMergeAction;
//# sourceMappingURL=data:application/json;base64,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