"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const codecommit = require("@aws-cdk/aws-codecommit");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const cloud9 = require("../lib");
let stack;
let vpc;
beforeEach(() => {
    stack = new cdk.Stack();
    vpc = new ec2.Vpc(stack, 'VPC');
});
test('create resource correctly with only vpc provide', () => {
    // WHEN
    new cloud9.Ec2Environment(stack, 'C9Env', { vpc });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Cloud9::EnvironmentEC2'));
});
test('create resource correctly with both vpc and subnetSelectio', () => {
    // WHEN
    new cloud9.Ec2Environment(stack, 'C9Env', {
        vpc,
        subnetSelection: {
            subnetType: ec2.SubnetType.PRIVATE,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Cloud9::EnvironmentEC2'));
});
test('import correctly from existing environment', () => {
    // WHEN
    const c9env = cloud9.Ec2Environment.fromEc2EnvironmentName(stack, 'ImportedEnv', 'existingEnvName');
    // THEN
    expect(c9env).toHaveProperty('ec2EnvironmentName');
});
test('create correctly with instanceType specified', () => {
    // WHEN
    new cloud9.Ec2Environment(stack, 'C9Env', {
        vpc,
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.LARGE),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Cloud9::EnvironmentEC2'));
});
test('throw error when subnetSelection not specified and the provided VPC has no public subnets', () => {
    // WHEN
    const privateOnlyVpc = new ec2.Vpc(stack, 'PrivateOnlyVpc', {
        maxAzs: 2,
        subnetConfiguration: [
            {
                subnetType: ec2.SubnetType.ISOLATED,
                name: 'IsolatedSubnet',
                cidrMask: 24,
            },
        ],
    });
    // THEN
    expect(() => {
        new cloud9.Ec2Environment(stack, 'C9Env', {
            vpc: privateOnlyVpc,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.LARGE),
        });
    }).toThrow(/no subnetSelection specified and no public subnet found in the vpc, please specify subnetSelection/);
});
test('can use CodeCommit repositories', () => {
    // WHEN
    const repo = codecommit.Repository.fromRepositoryName(stack, 'Repo', 'foo');
    new cloud9.Ec2Environment(stack, 'C9Env', {
        vpc,
        clonedRepositories: [
            cloud9.CloneRepository.fromCodeCommit(repo, '/src'),
        ],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cloud9::EnvironmentEC2', {
        InstanceType: 't2.micro',
        Repositories: [
            {
                PathComponent: '/src',
                RepositoryUrl: {
                    'Fn::Join': [
                        '',
                        [
                            'https://git-codecommit.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix',
                            },
                            '/v1/repos/foo',
                        ],
                    ],
                },
            },
        ],
    }));
});
//# sourceMappingURL=data:application/json;base64,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