"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
var KeyLength;
(function (KeyLength) {
    KeyLength[KeyLength["L2048"] = 2048] = "L2048";
    KeyLength[KeyLength["L4096"] = 4096] = "L4096";
})(KeyLength = exports.KeyLength || (exports.KeyLength = {}));
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
* An EC2 Key Pair
*/
class KeyPair extends cdk.Construct {
    /**
    * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
    */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
        * ARN of the private key in AWS Secrets Manager
        */
        this.arn = '';
        /**
        * Name of the Key Pair
        */
        this.name = '';
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncrypt(fn.role);
        }
        const tags = props.tags || {};
        tags.CreatedBy = ID;
        const key = new cfn.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                name: props.name,
                description: props.description || '',
                keyLength: props.keyLength || KeyLength.L2048,
                kms: ((_a = props.kms) === null || _a === void 0 ? void 0 : _a.keyArn) || 'alias/aws/secretsmanager',
                StackName: stack,
                tags: tags,
            }
        });
        this.arn = key.getAttString('PrivateKeyARN');
        this.name = key.getAttString('KeyPairName');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'ec2:CreateKeyPair',
                        'ec2:DeleteKeyPair',
                        'ec2:DescribeKeyPairs',
                        'secretsmanager:ListSecrets',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:TagResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:RequestTag/CreatedBy': ID,
                        }
                    },
                }),
                new iam.PolicyStatement({
                    actions: [
                        'secretsmanager:DeleteResourcePolicy',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:DescribeSecret',
                        'secretsmanager:GetResourcePolicy',
                        'secretsmanager:ListSecretVersionIds',
                        'secretsmanager:PutResourcePolicy',
                        'secretsmanager:PutSecretValue',
                        'secretsmanager:RestoreSecret',
                        'secretsmanager:UntagResource',
                        'secretsmanager:UpdateSecret',
                        'secretsmanager:UpdateSecretVersionStage',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:ResourceTag/CreatedBy': ID,
                        }
                    },
                }),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ]
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout)
        });
        return fn;
    }
    /**
    * Grants read access to the private key in the AWS Secrets Manager
    */
    grantRead(grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [this.arn],
            scope: this
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
//# sourceMappingURL=data:application/json;base64,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