import numpy as np
from scipy.signal import convolve


def parse_filter(filt, normalize=True):
    """Parse the name or array like, and return a column shaped filter (which is normalized by default)

    Used during pyramid construction.

    Parameters
    ----------
    filt : `str` or `array_like`.
        Name of the filter, as accepted by `named_filter`, or array to use as a filter. See that function for acceptable names.

    Returns
    -------
    filt : `array` or `dict`
        If `filt` was one of the steerable pyramids, then this will be a dictionary
        containing the various steerable pyramid filters. Else, it will be an array containing
        the specified filter.

    See also
    --------
    named_filter : function that converts `filter_name` str into an array or dict of arrays.
    """

    if isinstance(filt, str):
        filt = named_filter(filt)

    elif isinstance(filt, np.ndarray) or isinstance(filt, list) or isinstance(filt, tuple):
        filt = np.array(filt)
        if filt.ndim == 1:
            filt = np.reshape(filt, (filt.shape[0], 1))
        elif filt.ndim == 2 and filt.shape[0] == 1:
            filt = np.reshape(filt, (-1, 1))

    # TODO expand normalization options
    if normalize:
        filt = filt / filt.sum()

    return filt

def binomial_filter(order_plus_one):
    '''returns a vector of binomial coefficients of order (order_plus_one-1)'''
    if order_plus_one < 2:
        raise Exception("Error: order_plus_one argument must be at least 2")

    kernel = np.array([[0.5], [0.5]])
    for i in range(order_plus_one - 2):
        kernel = convolve(np.array([[0.5], [0.5]]), kernel)
    return kernel


def named_filter(name):
    '''Some standard 1D filter kernels.

    These are returned as column vectors (shape [N, 1]) and scaled such that their L2-norm is 1.0 (except for 'binomN')

    * `'binomN'` - binomial coefficient filter of order N-1
    * `'haar'` - Haar wavelet
    * `'qmf8'`, `'qmf12'`, `'qmf16'` - Symmetric Quadrature Mirror Filters [1]_
    * `'daub2'`, `'daub3'`, `'daub4'` - Daubechies wavelet [2]_
    * `'qmf5'`, `'qmf9'`, `'qmf13'`   - Symmetric Quadrature Mirror Filters [3]_, [4]_
    * `'spN_filters'` - steerable pyramid filters of order N (N must be one of {0, 1, 3, 5}) [5]_,
                        [6]_

    References
    ----------
    .. [1] J D Johnston, "A filter family designed for use in quadrature mirror filter banks",
       Proc. ICASSP, pp 291-294, 1980.
    .. [2] I Daubechies, "Orthonormal bases of compactly supported wavelets", Commun. Pure Appl.
       Math, vol. 42, pp 909-996, 1988.
    .. [3] E P Simoncelli,  "Orthogonal sub-band image transforms", PhD Thesis, MIT Dept. of Elec.
       Eng. and Comp. Sci. May 1988. Also available as: MIT Media Laboratory Vision and Modeling
       Technical Report #100.
    .. [4] E P Simoncelli and E H Adelson, "Subband image coding", Subband Transforms, chapter 4,
       ed. John W Woods, Kluwer Academic Publishers,  Norwell, MA, 1990, pp 143--192.
    .. [5] E P Simoncelli and W T Freeman, "The Steerable Pyramid: A Flexible Architecture for
       Multi-Scale Derivative Computation," Second Int'l Conf on Image Processing, Washington, DC,
       Oct 1995.
    .. [6] A Karasaridis and E P Simoncelli, "A Filter Design Technique for Steerable Pyramid
       Image Transforms", ICASSP, Atlanta, GA, May 1996.

    '''

    if name.startswith("binom"):
        kernel = np.sqrt(2) * binomial_filter(int(name[5:]))

    elif name.startswith('sp'):
        kernel = steerable_filters(name)

    elif name is "qmf5":
        kernel = np.array([[-0.076103], [0.3535534], [0.8593118], [0.3535534], [-0.076103]])
    elif name is "qmf9":
        kernel = np.array([[0.02807382], [-0.060944743], [-0.073386624], [0.41472545], [0.7973934],
                           [0.41472545], [-0.073386624], [-0.060944743], [0.02807382]])
    elif name is "qmf13":
        kernel = np.array([[-0.014556438], [0.021651438], [0.039045125], [-0.09800052],
                           [-0.057827797], [0.42995453], [0.7737113], [0.42995453], [-0.057827797],
                           [-0.09800052], [0.039045125], [0.021651438], [-0.014556438]])
    elif name is "qmf8":
        kernel = np.sqrt(2) * np.array([[0.00938715], [-0.07065183], [0.06942827], [0.4899808],
                                        [0.4899808], [0.06942827], [-0.07065183], [0.00938715]])
    elif name is "qmf12":
        kernel = np.array([[-0.003809699], [0.01885659], [-0.002710326], [-0.08469594],
                           [0.08846992], [0.4843894], [0.4843894], [0.08846992],
                           [-0.08469594], [-0.002710326], [0.01885659], [-0.003809699]])
        kernel *= np.sqrt(2)
    elif name is "qmf16":
        kernel = np.array([[0.001050167], [-0.005054526], [-0.002589756], [0.0276414],
                           [-0.009666376], [-0.09039223], [0.09779817], [0.4810284], [0.4810284],
                           [0.09779817], [-0.09039223], [-0.009666376], [0.0276414],
                           [-0.002589756], [-0.005054526], [0.001050167]])
        kernel *= np.sqrt(2)
    elif name is "haar":
        kernel = np.array([[1], [1]]) / np.sqrt(2)
    elif name is "daub2":
        kernel = np.array([[0.482962913145], [0.836516303738], [0.224143868042],
                           [-0.129409522551]])
    elif name is "daub3":
        kernel = np.array([[0.332670552950], [0.806891509311], [0.459877502118], [-0.135011020010],
                           [-0.085441273882], [0.035226291882]])
    elif name is "daub4":
        kernel = np.array([[0.230377813309], [0.714846570553], [0.630880767930],
                           [-0.027983769417], [-0.187034811719], [0.030841381836],
                           [0.032883011667], [-0.010597401785]])
    elif name is "gauss5":  # for backward-compatibility
        kernel = np.sqrt(2) * np.array([[0.0625], [0.25], [0.375], [0.25], [0.0625]])
    elif name is "gauss3":  # for backward-compatibility
        kernel = np.sqrt(2) * np.array([[0.25], [0.5], [0.25]])
    else:
        raise Exception("Error: Unknown filter name: %s" % (name))

    return kernel


def steerable_filters(filter_name):
    '''Steerable pyramid filters.

    Transform described in [1]_, filter kernel design described in [2]_.

    References
    ----------
    .. [1] E P Simoncelli and W T Freeman, "The Steerable Pyramid: A Flexible Architecture for
       Multi-Scale Derivative Computation," Second Int'l Conf on Image Processing, Washington, DC,
       Oct 1995.
    .. [2] A Karasaridis and E P Simoncelli, "A Filter Design Technique for Steerable Pyramid
       Image Transforms", ICASSP, Atlanta, GA, May 1996.
    '''
    if filter_name == 'sp0_filters':
        return _sp0_filters()
    elif filter_name == 'sp1_filters':
        return _sp1_filters()
    elif filter_name == 'sp3_filters':
        return _sp3_filters()
    elif filter_name == 'sp5_filters':
        return _sp5_filters()
    # elif os.path.isfile(filter_name):
    #     raise Exception("Filter files not supported yet")
    else:
        raise Exception("filter parameters value %s not supported" % (filter_name))


def _sp0_filters():
    filters = {}
    filters['harmonics'] = np.array([0])
    filters['lo0filt'] = (
        np.array([[-4.514000e-04, -1.137100e-04, -3.725800e-04, -3.743860e-03,
                   -3.725800e-04, -1.137100e-04, -4.514000e-04],
                  [-1.137100e-04, -6.119520e-03, -1.344160e-02, -7.563200e-03,
                   -1.344160e-02, -6.119520e-03, -1.137100e-04],
                  [-3.725800e-04, -1.344160e-02, 6.441488e-02, 1.524935e-01,
                   6.441488e-02, -1.344160e-02, -3.725800e-04],
                  [-3.743860e-03, -7.563200e-03, 1.524935e-01, 3.153017e-01,
                   1.524935e-01, -7.563200e-03, -3.743860e-03],
                  [-3.725800e-04, -1.344160e-02, 6.441488e-02, 1.524935e-01,
                   6.441488e-02, -1.344160e-02, -3.725800e-04],
                  [-1.137100e-04, -6.119520e-03, -1.344160e-02, -7.563200e-03,
                   -1.344160e-02, -6.119520e-03, -1.137100e-04],
                  [-4.514000e-04, -1.137100e-04, -3.725800e-04, -3.743860e-03,
                   -3.725800e-04, -1.137100e-04, -4.514000e-04]]))
    filters['lofilt'] = (
        np.array([[-2.257000e-04, -8.064400e-04, -5.686000e-05, 8.741400e-04,
                   -1.862800e-04, -1.031640e-03, -1.871920e-03, -1.031640e-03,
                   -1.862800e-04, 8.741400e-04, -5.686000e-05, -8.064400e-04,
                   -2.257000e-04],
                  [-8.064400e-04, 1.417620e-03, -1.903800e-04, -2.449060e-03,
                   -4.596420e-03, -7.006740e-03, -6.948900e-03, -7.006740e-03,
                   -4.596420e-03, -2.449060e-03, -1.903800e-04, 1.417620e-03,
                   -8.064400e-04],
                  [-5.686000e-05, -1.903800e-04, -3.059760e-03, -6.401000e-03,
                   -6.720800e-03, -5.236180e-03, -3.781600e-03, -5.236180e-03,
                   -6.720800e-03, -6.401000e-03, -3.059760e-03, -1.903800e-04,
                   -5.686000e-05],
                  [8.741400e-04, -2.449060e-03, -6.401000e-03, -5.260020e-03,
                   3.938620e-03, 1.722078e-02, 2.449600e-02, 1.722078e-02,
                   3.938620e-03, -5.260020e-03, -6.401000e-03, -2.449060e-03,
                   8.741400e-04],
                  [-1.862800e-04, -4.596420e-03, -6.720800e-03, 3.938620e-03,
                   3.220744e-02, 6.306262e-02, 7.624674e-02, 6.306262e-02,
                   3.220744e-02, 3.938620e-03, -6.720800e-03, -4.596420e-03,
                   -1.862800e-04],
                  [-1.031640e-03, -7.006740e-03, -5.236180e-03, 1.722078e-02,
                   6.306262e-02, 1.116388e-01, 1.348999e-01, 1.116388e-01,
                   6.306262e-02, 1.722078e-02, -5.236180e-03, -7.006740e-03,
                   -1.031640e-03],
                  [-1.871920e-03, -6.948900e-03, -3.781600e-03, 2.449600e-02,
                   7.624674e-02, 1.348999e-01, 1.576508e-01, 1.348999e-01,
                   7.624674e-02, 2.449600e-02, -3.781600e-03, -6.948900e-03,
                   -1.871920e-03],
                  [-1.031640e-03, -7.006740e-03, -5.236180e-03, 1.722078e-02,
                   6.306262e-02, 1.116388e-01, 1.348999e-01, 1.116388e-01,
                   6.306262e-02, 1.722078e-02, -5.236180e-03, -7.006740e-03,
                   -1.031640e-03],
                  [-1.862800e-04, -4.596420e-03, -6.720800e-03, 3.938620e-03,
                   3.220744e-02, 6.306262e-02, 7.624674e-02, 6.306262e-02,
                   3.220744e-02, 3.938620e-03, -6.720800e-03, -4.596420e-03,
                   -1.862800e-04],
                  [8.741400e-04, -2.449060e-03, -6.401000e-03, -5.260020e-03,
                   3.938620e-03, 1.722078e-02, 2.449600e-02, 1.722078e-02,
                   3.938620e-03, -5.260020e-03, -6.401000e-03, -2.449060e-03,
                   8.741400e-04],
                  [-5.686000e-05, -1.903800e-04, -3.059760e-03, -6.401000e-03,
                   -6.720800e-03, -5.236180e-03, -3.781600e-03, -5.236180e-03,
                   -6.720800e-03, -6.401000e-03, -3.059760e-03, -1.903800e-04,
                   -5.686000e-05],
                  [-8.064400e-04, 1.417620e-03, -1.903800e-04, -2.449060e-03,
                   -4.596420e-03, -7.006740e-03, -6.948900e-03, -7.006740e-03,
                   -4.596420e-03, -2.449060e-03, -1.903800e-04, 1.417620e-03,
                   -8.064400e-04],
                  [-2.257000e-04, -8.064400e-04, -5.686000e-05, 8.741400e-04,
                   -1.862800e-04, -1.031640e-03, -1.871920e-03, -1.031640e-03,
                   -1.862800e-04, 8.741400e-04, -5.686000e-05, -8.064400e-04,
                   -2.257000e-04]]))
    filters['mtx'] = np.array([1.000000])
    filters['hi0filt'] = (
        np.array([[5.997200e-04, -6.068000e-05, -3.324900e-04, -3.325600e-04,
                   -2.406600e-04, -3.325600e-04, -3.324900e-04, -6.068000e-05,
                   5.997200e-04],
                  [-6.068000e-05, 1.263100e-04, 4.927100e-04, 1.459700e-04,
                   -3.732100e-04, 1.459700e-04, 4.927100e-04, 1.263100e-04,
                   -6.068000e-05],
                  [-3.324900e-04, 4.927100e-04, -1.616650e-03, -1.437358e-02,
                   -2.420138e-02, -1.437358e-02, -1.616650e-03, 4.927100e-04,
                   -3.324900e-04],
                  [-3.325600e-04, 1.459700e-04, -1.437358e-02, -6.300923e-02,
                   -9.623594e-02, -6.300923e-02, -1.437358e-02, 1.459700e-04,
                   -3.325600e-04],
                  [-2.406600e-04, -3.732100e-04, -2.420138e-02, -9.623594e-02,
                   8.554893e-01, -9.623594e-02, -2.420138e-02, -3.732100e-04,
                   -2.406600e-04],
                  [-3.325600e-04, 1.459700e-04, -1.437358e-02, -6.300923e-02,
                   -9.623594e-02, -6.300923e-02, -1.437358e-02, 1.459700e-04,
                   -3.325600e-04],
                  [-3.324900e-04, 4.927100e-04, -1.616650e-03, -1.437358e-02,
                   -2.420138e-02, -1.437358e-02, -1.616650e-03, 4.927100e-04,
                   -3.324900e-04],
                  [-6.068000e-05, 1.263100e-04, 4.927100e-04, 1.459700e-04,
                   -3.732100e-04, 1.459700e-04, 4.927100e-04, 1.263100e-04,
                   -6.068000e-05],
                  [5.997200e-04, -6.068000e-05, -3.324900e-04, -3.325600e-04,
                   -2.406600e-04, -3.325600e-04, -3.324900e-04, -6.068000e-05,
                   5.997200e-04]]))
    filters['bfilts'] = (
        np.array([-9.066000e-05, -1.738640e-03, -4.942500e-03, -7.889390e-03,
                  -1.009473e-02, -7.889390e-03, -4.942500e-03, -1.738640e-03,
                  -9.066000e-05, -1.738640e-03, -4.625150e-03, -7.272540e-03,
                  -7.623410e-03, -9.091950e-03, -7.623410e-03, -7.272540e-03,
                  -4.625150e-03, -1.738640e-03, -4.942500e-03, -7.272540e-03,
                  -2.129540e-02, -2.435662e-02, -3.487008e-02, -2.435662e-02,
                  -2.129540e-02, -7.272540e-03, -4.942500e-03, -7.889390e-03,
                  -7.623410e-03, -2.435662e-02, -1.730466e-02, -3.158605e-02,
                  -1.730466e-02, -2.435662e-02, -7.623410e-03, -7.889390e-03,
                  -1.009473e-02, -9.091950e-03, -3.487008e-02, -3.158605e-02,
                  9.464195e-01, -3.158605e-02, -3.487008e-02, -9.091950e-03,
                  -1.009473e-02, -7.889390e-03, -7.623410e-03, -2.435662e-02,
                  -1.730466e-02, -3.158605e-02, -1.730466e-02, -2.435662e-02,
                  -7.623410e-03, -7.889390e-03, -4.942500e-03, -7.272540e-03,
                  -2.129540e-02, -2.435662e-02, -3.487008e-02, -2.435662e-02,
                  -2.129540e-02, -7.272540e-03, -4.942500e-03, -1.738640e-03,
                  -4.625150e-03, -7.272540e-03, -7.623410e-03, -9.091950e-03,
                  -7.623410e-03, -7.272540e-03, -4.625150e-03, -1.738640e-03,
                  -9.066000e-05, -1.738640e-03, -4.942500e-03, -7.889390e-03,
                  -1.009473e-02, -7.889390e-03, -4.942500e-03, -1.738640e-03,
                  -9.066000e-05]))
    filters['bfilts'] = filters['bfilts'].reshape(len(filters['bfilts']), 1)
    return filters


def _sp1_filters():
    filters = {}
    filters['harmonics'] = np.array([1])
    filters['mtx'] = np.eye(2)
    filters['lo0filt'] = (
        np.array([[-8.701000e-05, -1.354280e-03, -1.601260e-03, -5.033700e-04,
                   2.524010e-03, -5.033700e-04, -1.601260e-03, -1.354280e-03,
                   -8.701000e-05],
                  [-1.354280e-03, 2.921580e-03, 7.522720e-03, 8.224420e-03,
                   1.107620e-03, 8.224420e-03, 7.522720e-03, 2.921580e-03,
                   -1.354280e-03],
                  [-1.601260e-03, 7.522720e-03, -7.061290e-03, -3.769487e-02,
                   -3.297137e-02, -3.769487e-02, -7.061290e-03, 7.522720e-03,
                   -1.601260e-03],
                  [-5.033700e-04, 8.224420e-03, -3.769487e-02, 4.381320e-02,
                   1.811603e-01, 4.381320e-02, -3.769487e-02, 8.224420e-03,
                   -5.033700e-04],
                  [2.524010e-03, 1.107620e-03, -3.297137e-02, 1.811603e-01,
                   4.376250e-01, 1.811603e-01, -3.297137e-02, 1.107620e-03,
                   2.524010e-03],
                  [-5.033700e-04, 8.224420e-03, -3.769487e-02, 4.381320e-02,
                   1.811603e-01, 4.381320e-02, -3.769487e-02, 8.224420e-03,
                   -5.033700e-04],
                  [-1.601260e-03, 7.522720e-03, -7.061290e-03, -3.769487e-02,
                   -3.297137e-02, -3.769487e-02, -7.061290e-03, 7.522720e-03,
                   -1.601260e-03],
                  [-1.354280e-03, 2.921580e-03, 7.522720e-03, 8.224420e-03,
                   1.107620e-03, 8.224420e-03, 7.522720e-03, 2.921580e-03,
                   -1.354280e-03],
                  [-8.701000e-05, -1.354280e-03, -1.601260e-03, -5.033700e-04,
                   2.524010e-03, -5.033700e-04, -1.601260e-03, -1.354280e-03,
                   -8.701000e-05]]))
    filters['lofilt'] = (
        np.array([[-4.350000e-05, 1.207800e-04, -6.771400e-04, -1.243400e-04,
                   -8.006400e-04, -1.597040e-03, -2.516800e-04, -4.202000e-04,
                   1.262000e-03, -4.202000e-04, -2.516800e-04, -1.597040e-03,
                   -8.006400e-04, -1.243400e-04, -6.771400e-04, 1.207800e-04,
                   -4.350000e-05],
                  [1.207800e-04, 4.460600e-04, -5.814600e-04, 5.621600e-04,
                   -1.368800e-04, 2.325540e-03, 2.889860e-03, 4.287280e-03,
                   5.589400e-03, 4.287280e-03, 2.889860e-03, 2.325540e-03,
                   -1.368800e-04, 5.621600e-04, -5.814600e-04, 4.460600e-04,
                   1.207800e-04],
                  [-6.771400e-04, -5.814600e-04, 1.460780e-03, 2.160540e-03,
                   3.761360e-03, 3.080980e-03, 4.112200e-03, 2.221220e-03,
                   5.538200e-04, 2.221220e-03, 4.112200e-03, 3.080980e-03,
                   3.761360e-03, 2.160540e-03, 1.460780e-03, -5.814600e-04,
                   -6.771400e-04],
                  [-1.243400e-04, 5.621600e-04, 2.160540e-03, 3.175780e-03,
                   3.184680e-03, -1.777480e-03, -7.431700e-03, -9.056920e-03,
                   -9.637220e-03, -9.056920e-03, -7.431700e-03, -1.777480e-03,
                   3.184680e-03, 3.175780e-03, 2.160540e-03, 5.621600e-04,
                   -1.243400e-04],
                  [-8.006400e-04, -1.368800e-04, 3.761360e-03, 3.184680e-03,
                   -3.530640e-03, -1.260420e-02, -1.884744e-02, -1.750818e-02,
                   -1.648568e-02, -1.750818e-02, -1.884744e-02, -1.260420e-02,
                   -3.530640e-03, 3.184680e-03, 3.761360e-03, -1.368800e-04,
                   -8.006400e-04],
                  [-1.597040e-03, 2.325540e-03, 3.080980e-03, -1.777480e-03,
                   -1.260420e-02, -2.022938e-02, -1.109170e-02, 3.955660e-03,
                   1.438512e-02, 3.955660e-03, -1.109170e-02, -2.022938e-02,
                   -1.260420e-02, -1.777480e-03, 3.080980e-03, 2.325540e-03,
                   -1.597040e-03],
                  [-2.516800e-04, 2.889860e-03, 4.112200e-03, -7.431700e-03,
                   -1.884744e-02, -1.109170e-02, 2.190660e-02, 6.806584e-02,
                   9.058014e-02, 6.806584e-02, 2.190660e-02, -1.109170e-02,
                   -1.884744e-02, -7.431700e-03, 4.112200e-03, 2.889860e-03,
                   -2.516800e-04],
                  [-4.202000e-04, 4.287280e-03, 2.221220e-03, -9.056920e-03,
                   -1.750818e-02, 3.955660e-03, 6.806584e-02, 1.445500e-01,
                   1.773651e-01, 1.445500e-01, 6.806584e-02, 3.955660e-03,
                   -1.750818e-02, -9.056920e-03, 2.221220e-03, 4.287280e-03,
                   -4.202000e-04],
                  [1.262000e-03, 5.589400e-03, 5.538200e-04, -9.637220e-03,
                   -1.648568e-02, 1.438512e-02, 9.058014e-02, 1.773651e-01,
                   2.120374e-01, 1.773651e-01, 9.058014e-02, 1.438512e-02,
                   -1.648568e-02, -9.637220e-03, 5.538200e-04, 5.589400e-03,
                   1.262000e-03],
                  [-4.202000e-04, 4.287280e-03, 2.221220e-03, -9.056920e-03,
                   -1.750818e-02, 3.955660e-03, 6.806584e-02, 1.445500e-01,
                   1.773651e-01, 1.445500e-01, 6.806584e-02, 3.955660e-03,
                   -1.750818e-02, -9.056920e-03, 2.221220e-03, 4.287280e-03,
                   -4.202000e-04],
                  [-2.516800e-04, 2.889860e-03, 4.112200e-03, -7.431700e-03,
                   -1.884744e-02, -1.109170e-02, 2.190660e-02, 6.806584e-02,
                   9.058014e-02, 6.806584e-02, 2.190660e-02, -1.109170e-02,
                   -1.884744e-02, -7.431700e-03, 4.112200e-03, 2.889860e-03,
                   -2.516800e-04],
                  [-1.597040e-03, 2.325540e-03, 3.080980e-03, -1.777480e-03,
                   -1.260420e-02, -2.022938e-02, -1.109170e-02, 3.955660e-03,
                   1.438512e-02, 3.955660e-03, -1.109170e-02, -2.022938e-02,
                   -1.260420e-02, -1.777480e-03, 3.080980e-03, 2.325540e-03,
                   -1.597040e-03],
                  [-8.006400e-04, -1.368800e-04, 3.761360e-03, 3.184680e-03,
                   -3.530640e-03, -1.260420e-02, -1.884744e-02, -1.750818e-02,
                   -1.648568e-02, -1.750818e-02, -1.884744e-02, -1.260420e-02,
                   -3.530640e-03, 3.184680e-03, 3.761360e-03, -1.368800e-04,
                   -8.006400e-04],
                  [-1.243400e-04, 5.621600e-04, 2.160540e-03, 3.175780e-03,
                   3.184680e-03, -1.777480e-03, -7.431700e-03, -9.056920e-03,
                   -9.637220e-03, -9.056920e-03, -7.431700e-03, -1.777480e-03,
                   3.184680e-03, 3.175780e-03, 2.160540e-03, 5.621600e-04,
                   -1.243400e-04],
                  [-6.771400e-04, -5.814600e-04, 1.460780e-03, 2.160540e-03,
                   3.761360e-03, 3.080980e-03, 4.112200e-03, 2.221220e-03,
                   5.538200e-04, 2.221220e-03, 4.112200e-03, 3.080980e-03,
                   3.761360e-03, 2.160540e-03, 1.460780e-03, -5.814600e-04,
                   -6.771400e-04],
                  [1.207800e-04, 4.460600e-04, -5.814600e-04, 5.621600e-04,
                   -1.368800e-04, 2.325540e-03, 2.889860e-03, 4.287280e-03,
                   5.589400e-03, 4.287280e-03, 2.889860e-03, 2.325540e-03,
                   -1.368800e-04, 5.621600e-04, -5.814600e-04, 4.460600e-04,
                   1.207800e-04],
                  [-4.350000e-05, 1.207800e-04, -6.771400e-04, -1.243400e-04,
                   -8.006400e-04, -1.597040e-03, -2.516800e-04, -4.202000e-04,
                   1.262000e-03, -4.202000e-04, -2.516800e-04, -1.597040e-03,
                   -8.006400e-04, -1.243400e-04, -6.771400e-04, 1.207800e-04,
                   -4.350000e-05]]))
    filters['hi0filt'] = (
        np.array([[-9.570000e-04, -2.424100e-04, -1.424720e-03, -8.742600e-04,
                   -1.166810e-03, -8.742600e-04, -1.424720e-03, -2.424100e-04,
                   -9.570000e-04],
                  [-2.424100e-04, -4.317530e-03, 8.998600e-04, 9.156420e-03,
                   1.098012e-02, 9.156420e-03, 8.998600e-04, -4.317530e-03,
                   -2.424100e-04],
                  [-1.424720e-03, 8.998600e-04, 1.706347e-02, 1.094866e-02,
                   -5.897780e-03, 1.094866e-02, 1.706347e-02, 8.998600e-04,
                   -1.424720e-03],
                  [-8.742600e-04, 9.156420e-03, 1.094866e-02, -7.841370e-02,
                   -1.562827e-01, -7.841370e-02, 1.094866e-02, 9.156420e-03,
                   -8.742600e-04],
                  [-1.166810e-03, 1.098012e-02, -5.897780e-03, -1.562827e-01,
                   7.282593e-01, -1.562827e-01, -5.897780e-03, 1.098012e-02,
                   -1.166810e-03],
                  [-8.742600e-04, 9.156420e-03, 1.094866e-02, -7.841370e-02,
                   -1.562827e-01, -7.841370e-02, 1.094866e-02, 9.156420e-03,
                   -8.742600e-04],
                  [-1.424720e-03, 8.998600e-04, 1.706347e-02, 1.094866e-02,
                   -5.897780e-03, 1.094866e-02, 1.706347e-02, 8.998600e-04,
                   -1.424720e-03],
                  [-2.424100e-04, -4.317530e-03, 8.998600e-04, 9.156420e-03,
                   1.098012e-02, 9.156420e-03, 8.998600e-04, -4.317530e-03,
                   -2.424100e-04],
                  [-9.570000e-04, -2.424100e-04, -1.424720e-03, -8.742600e-04,
                   -1.166810e-03, -8.742600e-04, -1.424720e-03, -2.424100e-04,
                   -9.570000e-04]]))
    filters['bfilts'] = (
        np.array([[6.125880e-03, -8.052600e-03, -2.103714e-02, -1.536890e-02,
                   -1.851466e-02, -1.536890e-02, -2.103714e-02, -8.052600e-03,
                   6.125880e-03, -1.287416e-02, -9.611520e-03, 1.023569e-02,
                   6.009450e-03, 1.872620e-03, 6.009450e-03, 1.023569e-02,
                   -9.611520e-03, -1.287416e-02, -5.641530e-03, 4.168400e-03,
                   -2.382180e-02, -5.375324e-02, -2.076086e-02, -5.375324e-02,
                   -2.382180e-02, 4.168400e-03, -5.641530e-03, -8.957260e-03,
                   -1.751170e-03, -1.836909e-02, 1.265655e-01, 2.996168e-01,
                   1.265655e-01, -1.836909e-02, -1.751170e-03, -8.957260e-03,
                   0.000000e+00, 0.000000e+00, 0.000000e+00, 0.000000e+00,
                   0.000000e+00, 0.000000e+00, 0.000000e+00, 0.000000e+00,
                   0.000000e+00, 8.957260e-03, 1.751170e-03, 1.836909e-02,
                   -1.265655e-01, -2.996168e-01, -1.265655e-01, 1.836909e-02,
                   1.751170e-03, 8.957260e-03, 5.641530e-03, -4.168400e-03,
                   2.382180e-02, 5.375324e-02, 2.076086e-02, 5.375324e-02,
                   2.382180e-02, -4.168400e-03, 5.641530e-03, 1.287416e-02,
                   9.611520e-03, -1.023569e-02, -6.009450e-03, -1.872620e-03,
                   -6.009450e-03, -1.023569e-02, 9.611520e-03, 1.287416e-02,
                   -6.125880e-03, 8.052600e-03, 2.103714e-02, 1.536890e-02,
                   1.851466e-02, 1.536890e-02, 2.103714e-02, 8.052600e-03,
                   -6.125880e-03],
                  [-6.125880e-03, 1.287416e-02, 5.641530e-03, 8.957260e-03,
                   0.000000e+00, -8.957260e-03, -5.641530e-03, -1.287416e-02,
                   6.125880e-03, 8.052600e-03, 9.611520e-03, -4.168400e-03,
                   1.751170e-03, 0.000000e+00, -1.751170e-03, 4.168400e-03,
                   -9.611520e-03, -8.052600e-03, 2.103714e-02, -1.023569e-02,
                   2.382180e-02, 1.836909e-02, 0.000000e+00, -1.836909e-02,
                   -2.382180e-02, 1.023569e-02, -2.103714e-02, 1.536890e-02,
                   -6.009450e-03, 5.375324e-02, -1.265655e-01, 0.000000e+00,
                   1.265655e-01, -5.375324e-02, 6.009450e-03, -1.536890e-02,
                   1.851466e-02, -1.872620e-03, 2.076086e-02, -2.996168e-01,
                   0.000000e+00, 2.996168e-01, -2.076086e-02, 1.872620e-03,
                   -1.851466e-02, 1.536890e-02, -6.009450e-03, 5.375324e-02,
                   -1.265655e-01, 0.000000e+00, 1.265655e-01, -5.375324e-02,
                   6.009450e-03, -1.536890e-02, 2.103714e-02, -1.023569e-02,
                   2.382180e-02, 1.836909e-02, 0.000000e+00, -1.836909e-02,
                   -2.382180e-02, 1.023569e-02, -2.103714e-02, 8.052600e-03,
                   9.611520e-03, -4.168400e-03, 1.751170e-03, 0.000000e+00,
                   -1.751170e-03, 4.168400e-03, -9.611520e-03, -8.052600e-03,
                   -6.125880e-03, 1.287416e-02, 5.641530e-03, 8.957260e-03,
                   0.000000e+00, -8.957260e-03, -5.641530e-03, -1.287416e-02,
                   6.125880e-03]]).T)
    filters['bfilts'] = np.negative(filters['bfilts'])
    return filters


def _sp3_filters():
    filters = {}
    filters['harmonics'] = np.array([1, 3])
    filters['mtx'] = (
        np.array([[0.5000, 0.3536, 0, -0.3536],
                  [-0.0000, 0.3536, 0.5000, 0.3536],
                  [0.5000, -0.3536, 0, 0.3536],
                  [-0.0000, 0.3536, -0.5000, 0.3536]]))
    filters['hi0filt'] = (
        np.array([[-4.0483998600E-4, -6.2596000498E-4, -3.7829999201E-5,
                   8.8387000142E-4, 1.5450799838E-3, 1.9235999789E-3,
                   2.0687500946E-3, 2.0898699295E-3, 2.0687500946E-3,
                   1.9235999789E-3, 1.5450799838E-3, 8.8387000142E-4,
                   -3.7829999201E-5, -6.2596000498E-4, -4.0483998600E-4],
                  [-6.2596000498E-4, -3.2734998967E-4, 7.7435001731E-4,
                   1.5874400269E-3, 2.1750701126E-3, 2.5626500137E-3,
                   2.2892199922E-3, 1.9755100366E-3, 2.2892199922E-3,
                   2.5626500137E-3, 2.1750701126E-3, 1.5874400269E-3,
                   7.7435001731E-4, -3.2734998967E-4, -6.2596000498E-4],
                  [-3.7829999201E-5, 7.7435001731E-4, 1.1793200392E-3,
                   1.4050999889E-3, 2.2253401112E-3, 2.1145299543E-3,
                   3.3578000148E-4, -8.3368999185E-4, 3.3578000148E-4,
                   2.1145299543E-3, 2.2253401112E-3, 1.4050999889E-3,
                   1.1793200392E-3, 7.7435001731E-4, -3.7829999201E-5],
                  [8.8387000142E-4, 1.5874400269E-3, 1.4050999889E-3,
                   1.2960999738E-3, -4.9274001503E-4, -3.1295299996E-3,
                   -4.5751798898E-3, -5.1014497876E-3, -4.5751798898E-3,
                   -3.1295299996E-3, -4.9274001503E-4, 1.2960999738E-3,
                   1.4050999889E-3, 1.5874400269E-3, 8.8387000142E-4],
                  [1.5450799838E-3, 2.1750701126E-3, 2.2253401112E-3,
                   -4.9274001503E-4, -6.3222697936E-3, -2.7556000277E-3,
                   5.3632198833E-3, 7.3032598011E-3, 5.3632198833E-3,
                   -2.7556000277E-3, -6.3222697936E-3, -4.9274001503E-4,
                   2.2253401112E-3, 2.1750701126E-3, 1.5450799838E-3],
                  [1.9235999789E-3, 2.5626500137E-3, 2.1145299543E-3,
                   -3.1295299996E-3, -2.7556000277E-3, 1.3962360099E-2,
                   7.8046298586E-3, -9.3812197447E-3, 7.8046298586E-3,
                   1.3962360099E-2, -2.7556000277E-3, -3.1295299996E-3,
                   2.1145299543E-3, 2.5626500137E-3, 1.9235999789E-3],
                  [2.0687500946E-3, 2.2892199922E-3, 3.3578000148E-4,
                   -4.5751798898E-3, 5.3632198833E-3, 7.8046298586E-3,
                   -7.9501636326E-2, -0.1554141641, -7.9501636326E-2,
                   7.8046298586E-3, 5.3632198833E-3, -4.5751798898E-3,
                   3.3578000148E-4, 2.2892199922E-3, 2.0687500946E-3],
                  [2.0898699295E-3, 1.9755100366E-3, -8.3368999185E-4,
                   -5.1014497876E-3, 7.3032598011E-3, -9.3812197447E-3,
                   -0.1554141641, 0.7303866148, -0.1554141641,
                   -9.3812197447E-3, 7.3032598011E-3, -5.1014497876E-3,
                   -8.3368999185E-4, 1.9755100366E-3, 2.0898699295E-3],
                  [2.0687500946E-3, 2.2892199922E-3, 3.3578000148E-4,
                   -4.5751798898E-3, 5.3632198833E-3, 7.8046298586E-3,
                   -7.9501636326E-2, -0.1554141641, -7.9501636326E-2,
                   7.8046298586E-3, 5.3632198833E-3, -4.5751798898E-3,
                   3.3578000148E-4, 2.2892199922E-3, 2.0687500946E-3],
                  [1.9235999789E-3, 2.5626500137E-3, 2.1145299543E-3,
                   -3.1295299996E-3, -2.7556000277E-3, 1.3962360099E-2,
                   7.8046298586E-3, -9.3812197447E-3, 7.8046298586E-3,
                   1.3962360099E-2, -2.7556000277E-3, -3.1295299996E-3,
                   2.1145299543E-3, 2.5626500137E-3, 1.9235999789E-3],
                  [1.5450799838E-3, 2.1750701126E-3, 2.2253401112E-3,
                   -4.9274001503E-4, -6.3222697936E-3, -2.7556000277E-3,
                   5.3632198833E-3, 7.3032598011E-3, 5.3632198833E-3,
                   -2.7556000277E-3, -6.3222697936E-3, -4.9274001503E-4,
                   2.2253401112E-3, 2.1750701126E-3, 1.5450799838E-3],
                  [8.8387000142E-4, 1.5874400269E-3, 1.4050999889E-3,
                   1.2960999738E-3, -4.9274001503E-4, -3.1295299996E-3,
                   -4.5751798898E-3, -5.1014497876E-3, -4.5751798898E-3,
                   -3.1295299996E-3, -4.9274001503E-4, 1.2960999738E-3,
                   1.4050999889E-3, 1.5874400269E-3, 8.8387000142E-4],
                  [-3.7829999201E-5, 7.7435001731E-4, 1.1793200392E-3,
                   1.4050999889E-3, 2.2253401112E-3, 2.1145299543E-3,
                   3.3578000148E-4, -8.3368999185E-4, 3.3578000148E-4,
                   2.1145299543E-3, 2.2253401112E-3, 1.4050999889E-3,
                   1.1793200392E-3, 7.7435001731E-4, -3.7829999201E-5],
                  [-6.2596000498E-4, -3.2734998967E-4, 7.7435001731E-4,
                   1.5874400269E-3, 2.1750701126E-3, 2.5626500137E-3,
                   2.2892199922E-3, 1.9755100366E-3, 2.2892199922E-3,
                   2.5626500137E-3, 2.1750701126E-3, 1.5874400269E-3,
                   7.7435001731E-4, -3.2734998967E-4, -6.2596000498E-4],
                  [-4.0483998600E-4, -6.2596000498E-4, -3.7829999201E-5,
                   8.8387000142E-4, 1.5450799838E-3, 1.9235999789E-3,
                   2.0687500946E-3, 2.0898699295E-3, 2.0687500946E-3,
                   1.9235999789E-3, 1.5450799838E-3, 8.8387000142E-4,
                   -3.7829999201E-5, -6.2596000498E-4, -4.0483998600E-4]]))
    filters['lo0filt'] = (
        np.array([[-8.7009997515E-5, -1.3542800443E-3, -1.6012600390E-3,
                   -5.0337001448E-4, 2.5240099058E-3, -5.0337001448E-4,
                   -1.6012600390E-3, -1.3542800443E-3, -8.7009997515E-5],
                  [-1.3542800443E-3, 2.9215801042E-3, 7.5227199122E-3,
                   8.2244202495E-3, 1.1076199589E-3, 8.2244202495E-3,
                   7.5227199122E-3, 2.9215801042E-3, -1.3542800443E-3],
                  [-1.6012600390E-3, 7.5227199122E-3, -7.0612900890E-3,
                   -3.7694871426E-2, -3.2971370965E-2, -3.7694871426E-2,
                   -7.0612900890E-3, 7.5227199122E-3, -1.6012600390E-3],
                  [-5.0337001448E-4, 8.2244202495E-3, -3.7694871426E-2,
                   4.3813198805E-2, 0.1811603010, 4.3813198805E-2,
                   -3.7694871426E-2, 8.2244202495E-3, -5.0337001448E-4],
                  [2.5240099058E-3, 1.1076199589E-3, -3.2971370965E-2,
                   0.1811603010, 0.4376249909, 0.1811603010,
                   -3.2971370965E-2, 1.1076199589E-3, 2.5240099058E-3],
                  [-5.0337001448E-4, 8.2244202495E-3, -3.7694871426E-2,
                   4.3813198805E-2, 0.1811603010, 4.3813198805E-2,
                   -3.7694871426E-2, 8.2244202495E-3, -5.0337001448E-4],
                  [-1.6012600390E-3, 7.5227199122E-3, -7.0612900890E-3,
                   -3.7694871426E-2, -3.2971370965E-2, -3.7694871426E-2,
                   -7.0612900890E-3, 7.5227199122E-3, -1.6012600390E-3],
                  [-1.3542800443E-3, 2.9215801042E-3, 7.5227199122E-3,
                   8.2244202495E-3, 1.1076199589E-3, 8.2244202495E-3,
                   7.5227199122E-3, 2.9215801042E-3, -1.3542800443E-3],
                  [-8.7009997515E-5, -1.3542800443E-3, -1.6012600390E-3,
                   -5.0337001448E-4, 2.5240099058E-3, -5.0337001448E-4,
                   -1.6012600390E-3, -1.3542800443E-3, -8.7009997515E-5]]))
    filters['lofilt'] = (
        np.array([[-4.3500000174E-5, 1.2078000145E-4, -6.7714002216E-4,
                   -1.2434000382E-4, -8.0063997302E-4, -1.5970399836E-3,
                   -2.5168000138E-4, -4.2019999819E-4, 1.2619999470E-3,
                   -4.2019999819E-4, -2.5168000138E-4, -1.5970399836E-3,
                   -8.0063997302E-4, -1.2434000382E-4, -6.7714002216E-4,
                   1.2078000145E-4, -4.3500000174E-5],
                  [1.2078000145E-4, 4.4606000301E-4, -5.8146001538E-4,
                   5.6215998484E-4, -1.3688000035E-4, 2.3255399428E-3,
                   2.8898599558E-3, 4.2872801423E-3, 5.5893999524E-3,
                   4.2872801423E-3, 2.8898599558E-3, 2.3255399428E-3,
                   -1.3688000035E-4, 5.6215998484E-4, -5.8146001538E-4,
                   4.4606000301E-4, 1.2078000145E-4],
                  [-6.7714002216E-4, -5.8146001538E-4, 1.4607800404E-3,
                   2.1605400834E-3, 3.7613599561E-3, 3.0809799209E-3,
                   4.1121998802E-3, 2.2212199401E-3, 5.5381999118E-4,
                   2.2212199401E-3, 4.1121998802E-3, 3.0809799209E-3,
                   3.7613599561E-3, 2.1605400834E-3, 1.4607800404E-3,
                   -5.8146001538E-4, -6.7714002216E-4],
                  [-1.2434000382E-4, 5.6215998484E-4, 2.1605400834E-3,
                   3.1757799443E-3, 3.1846798956E-3, -1.7774800071E-3,
                   -7.4316998944E-3, -9.0569201857E-3, -9.6372198313E-3,
                   -9.0569201857E-3, -7.4316998944E-3, -1.7774800071E-3,
                   3.1846798956E-3, 3.1757799443E-3, 2.1605400834E-3,
                   5.6215998484E-4, -1.2434000382E-4],
                  [-8.0063997302E-4, -1.3688000035E-4, 3.7613599561E-3,
                   3.1846798956E-3, -3.5306399222E-3, -1.2604200281E-2,
                   -1.8847439438E-2, -1.7508180812E-2, -1.6485679895E-2,
                   -1.7508180812E-2, -1.8847439438E-2, -1.2604200281E-2,
                   -3.5306399222E-3, 3.1846798956E-3, 3.7613599561E-3,
                   -1.3688000035E-4, -8.0063997302E-4],
                  [-1.5970399836E-3, 2.3255399428E-3, 3.0809799209E-3,
                   -1.7774800071E-3, -1.2604200281E-2, -2.0229380578E-2,
                   -1.1091699824E-2, 3.9556599222E-3, 1.4385120012E-2,
                   3.9556599222E-3, -1.1091699824E-2, -2.0229380578E-2,
                   -1.2604200281E-2, -1.7774800071E-3, 3.0809799209E-3,
                   2.3255399428E-3, -1.5970399836E-3],
                  [-2.5168000138E-4, 2.8898599558E-3, 4.1121998802E-3,
                   -7.4316998944E-3, -1.8847439438E-2, -1.1091699824E-2,
                   2.1906599402E-2, 6.8065837026E-2, 9.0580143034E-2,
                   6.8065837026E-2, 2.1906599402E-2, -1.1091699824E-2,
                   -1.8847439438E-2, -7.4316998944E-3, 4.1121998802E-3,
                   2.8898599558E-3, -2.5168000138E-4],
                  [-4.2019999819E-4, 4.2872801423E-3, 2.2212199401E-3,
                   -9.0569201857E-3, -1.7508180812E-2, 3.9556599222E-3,
                   6.8065837026E-2, 0.1445499808, 0.1773651242,
                   0.1445499808, 6.8065837026E-2, 3.9556599222E-3,
                   -1.7508180812E-2, -9.0569201857E-3, 2.2212199401E-3,
                   4.2872801423E-3, -4.2019999819E-4],
                  [1.2619999470E-3, 5.5893999524E-3, 5.5381999118E-4,
                   -9.6372198313E-3, -1.6485679895E-2, 1.4385120012E-2,
                   9.0580143034E-2, 0.1773651242, 0.2120374441,
                   0.1773651242, 9.0580143034E-2, 1.4385120012E-2,
                   -1.6485679895E-2, -9.6372198313E-3, 5.5381999118E-4,
                   5.5893999524E-3, 1.2619999470E-3],
                  [-4.2019999819E-4, 4.2872801423E-3, 2.2212199401E-3,
                   -9.0569201857E-3, -1.7508180812E-2, 3.9556599222E-3,
                   6.8065837026E-2, 0.1445499808, 0.1773651242,
                   0.1445499808, 6.8065837026E-2, 3.9556599222E-3,
                   -1.7508180812E-2, -9.0569201857E-3, 2.2212199401E-3,
                   4.2872801423E-3, -4.2019999819E-4],
                  [-2.5168000138E-4, 2.8898599558E-3, 4.1121998802E-3,
                   -7.4316998944E-3, -1.8847439438E-2, -1.1091699824E-2,
                   2.1906599402E-2, 6.8065837026E-2, 9.0580143034E-2,
                   6.8065837026E-2, 2.1906599402E-2, -1.1091699824E-2,
                   -1.8847439438E-2, -7.4316998944E-3, 4.1121998802E-3,
                   2.8898599558E-3, -2.5168000138E-4],
                  [-1.5970399836E-3, 2.3255399428E-3, 3.0809799209E-3,
                   -1.7774800071E-3, -1.2604200281E-2, -2.0229380578E-2,
                   -1.1091699824E-2, 3.9556599222E-3, 1.4385120012E-2,
                   3.9556599222E-3, -1.1091699824E-2, -2.0229380578E-2,
                   -1.2604200281E-2, -1.7774800071E-3, 3.0809799209E-3,
                   2.3255399428E-3, -1.5970399836E-3],
                  [-8.0063997302E-4, -1.3688000035E-4, 3.7613599561E-3,
                   3.1846798956E-3, -3.5306399222E-3, -1.2604200281E-2,
                   -1.8847439438E-2, -1.7508180812E-2, -1.6485679895E-2,
                   -1.7508180812E-2, -1.8847439438E-2, -1.2604200281E-2,
                   -3.5306399222E-3, 3.1846798956E-3, 3.7613599561E-3,
                   -1.3688000035E-4, -8.0063997302E-4],
                  [-1.2434000382E-4, 5.6215998484E-4, 2.1605400834E-3,
                   3.1757799443E-3, 3.1846798956E-3, -1.7774800071E-3,
                   -7.4316998944E-3, -9.0569201857E-3, -9.6372198313E-3,
                   -9.0569201857E-3, -7.4316998944E-3, -1.7774800071E-3,
                   3.1846798956E-3, 3.1757799443E-3, 2.1605400834E-3,
                   5.6215998484E-4, -1.2434000382E-4],
                  [-6.7714002216E-4, -5.8146001538E-4, 1.4607800404E-3,
                   2.1605400834E-3, 3.7613599561E-3, 3.0809799209E-3,
                   4.1121998802E-3, 2.2212199401E-3, 5.5381999118E-4,
                   2.2212199401E-3, 4.1121998802E-3, 3.0809799209E-3,
                   3.7613599561E-3, 2.1605400834E-3, 1.4607800404E-3,
                   -5.8146001538E-4, -6.7714002216E-4],
                  [1.2078000145E-4, 4.4606000301E-4, -5.8146001538E-4,
                   5.6215998484E-4, -1.3688000035E-4, 2.3255399428E-3,
                   2.8898599558E-3, 4.2872801423E-3, 5.5893999524E-3,
                   4.2872801423E-3, 2.8898599558E-3, 2.3255399428E-3,
                   -1.3688000035E-4, 5.6215998484E-4, -5.8146001538E-4,
                   4.4606000301E-4, 1.2078000145E-4],
                  [-4.3500000174E-5, 1.2078000145E-4, -6.7714002216E-4,
                   -1.2434000382E-4, -8.0063997302E-4, -1.5970399836E-3,
                   -2.5168000138E-4, -4.2019999819E-4, 1.2619999470E-3,
                   -4.2019999819E-4, -2.5168000138E-4, -1.5970399836E-3,
                   -8.0063997302E-4, -1.2434000382E-4, -6.7714002216E-4,
                   1.2078000145E-4, -4.3500000174E-5]]))
    filters['bfilts'] = (
        np.array([[-8.1125000725E-4, 4.4451598078E-3, 1.2316980399E-2,
                   1.3955879956E-2,  1.4179450460E-2, 1.3955879956E-2,
                   1.2316980399E-2, 4.4451598078E-3, -8.1125000725E-4,
                   3.9103501476E-3, 4.4565401040E-3, -5.8724298142E-3,
                   -2.8760801069E-3, 8.5267601535E-3, -2.8760801069E-3,
                   -5.8724298142E-3, 4.4565401040E-3, 3.9103501476E-3,
                   1.3462699717E-3, -3.7740699481E-3, 8.2581602037E-3,
                   3.9442278445E-2, 5.3605638444E-2, 3.9442278445E-2,
                   8.2581602037E-3, -3.7740699481E-3, 1.3462699717E-3,
                   7.4700999539E-4, -3.6522001028E-4, -2.2522680461E-2,
                   -0.1105690673, -0.1768419296, -0.1105690673,
                   -2.2522680461E-2, -3.6522001028E-4, 7.4700999539E-4,
                   0.0000000000, 0.0000000000, 0.0000000000,
                   0.0000000000, 0.0000000000, 0.0000000000,
                   0.0000000000, 0.0000000000, 0.0000000000,
                   -7.4700999539E-4, 3.6522001028E-4, 2.2522680461E-2,
                   0.1105690673, 0.1768419296, 0.1105690673,
                   2.2522680461E-2, 3.6522001028E-4, -7.4700999539E-4,
                   -1.3462699717E-3, 3.7740699481E-3, -8.2581602037E-3,
                   -3.9442278445E-2, -5.3605638444E-2, -3.9442278445E-2,
                   -8.2581602037E-3, 3.7740699481E-3, -1.3462699717E-3,
                   -3.9103501476E-3, -4.4565401040E-3, 5.8724298142E-3,
                   2.8760801069E-3, -8.5267601535E-3, 2.8760801069E-3,
                   5.8724298142E-3, -4.4565401040E-3, -3.9103501476E-3,
                   8.1125000725E-4, -4.4451598078E-3, -1.2316980399E-2,
                   -1.3955879956E-2, -1.4179450460E-2, -1.3955879956E-2,
                   -1.2316980399E-2, -4.4451598078E-3, 8.1125000725E-4],
                  [0.0000000000, -8.2846998703E-4, -5.7109999034E-5,
                   4.0110000555E-5, 4.6670897864E-3, 8.0871898681E-3,
                   1.4807609841E-2, 8.6204400286E-3, -3.1221499667E-3,
                   8.2846998703E-4, 0.0000000000, -9.7479997203E-4,
                   -6.9718998857E-3, -2.0865600090E-3, 2.3298799060E-3,
                   -4.4814897701E-3, 1.4917500317E-2, 8.6204400286E-3,
                   5.7109999034E-5, 9.7479997203E-4, 0.0000000000,
                   -1.2145539746E-2, -2.4427289143E-2, 5.0797060132E-2,
                   3.2785870135E-2, -4.4814897701E-3, 1.4807609841E-2,
                   -4.0110000555E-5, 6.9718998857E-3, 1.2145539746E-2,
                   0.0000000000, -0.1510555595, -8.2495503128E-2,
                   5.0797060132E-2, 2.3298799060E-3, 8.0871898681E-3,
                   -4.6670897864E-3, 2.0865600090E-3, 2.4427289143E-2,
                   0.1510555595, 0.0000000000, -0.1510555595,
                   -2.4427289143E-2, -2.0865600090E-3, 4.6670897864E-3,
                   -8.0871898681E-3, -2.3298799060E-3, -5.0797060132E-2,
                   8.2495503128E-2, 0.1510555595, 0.0000000000,
                   -1.2145539746E-2, -6.9718998857E-3, 4.0110000555E-5,
                   -1.4807609841E-2, 4.4814897701E-3, -3.2785870135E-2,
                   -5.0797060132E-2, 2.4427289143E-2, 1.2145539746E-2,
                   0.0000000000, -9.7479997203E-4, -5.7109999034E-5,
                   -8.6204400286E-3, -1.4917500317E-2, 4.4814897701E-3,
                   -2.3298799060E-3, 2.0865600090E-3, 6.9718998857E-3,
                   9.7479997203E-4, 0.0000000000, -8.2846998703E-4,
                   3.1221499667E-3, -8.6204400286E-3, -1.4807609841E-2,
                   -8.0871898681E-3, -4.6670897864E-3, -4.0110000555E-5,
                   5.7109999034E-5, 8.2846998703E-4, 0.0000000000],
                  [8.1125000725E-4, -3.9103501476E-3, -1.3462699717E-3,
                   -7.4700999539E-4, 0.0000000000, 7.4700999539E-4,
                   1.3462699717E-3, 3.9103501476E-3, -8.1125000725E-4,
                   -4.4451598078E-3, -4.4565401040E-3, 3.7740699481E-3,
                   3.6522001028E-4, 0.0000000000, -3.6522001028E-4,
                   -3.7740699481E-3, 4.4565401040E-3, 4.4451598078E-3,
                   -1.2316980399E-2, 5.8724298142E-3, -8.2581602037E-3,
                   2.2522680461E-2, 0.0000000000, -2.2522680461E-2,
                   8.2581602037E-3, -5.8724298142E-3, 1.2316980399E-2,
                   -1.3955879956E-2, 2.8760801069E-3, -3.9442278445E-2,
                   0.1105690673, 0.0000000000, -0.1105690673,
                   3.9442278445E-2, -2.8760801069E-3, 1.3955879956E-2,
                   -1.4179450460E-2, -8.5267601535E-3, -5.3605638444E-2,
                   0.1768419296, 0.0000000000, -0.1768419296,
                   5.3605638444E-2, 8.5267601535E-3, 1.4179450460E-2,
                   -1.3955879956E-2, 2.8760801069E-3, -3.9442278445E-2,
                   0.1105690673, 0.0000000000, -0.1105690673,
                   3.9442278445E-2, -2.8760801069E-3, 1.3955879956E-2,
                   -1.2316980399E-2, 5.8724298142E-3, -8.2581602037E-3,
                   2.2522680461E-2, 0.0000000000, -2.2522680461E-2,
                   8.2581602037E-3, -5.8724298142E-3, 1.2316980399E-2,
                   -4.4451598078E-3, -4.4565401040E-3, 3.7740699481E-3,
                   3.6522001028E-4, 0.0000000000, -3.6522001028E-4,
                   -3.7740699481E-3, 4.4565401040E-3, 4.4451598078E-3,
                   8.1125000725E-4, -3.9103501476E-3, -1.3462699717E-3,
                   -7.4700999539E-4, 0.0000000000, 7.4700999539E-4,
                   1.3462699717E-3, 3.9103501476E-3, -8.1125000725E-4],
                  [3.1221499667E-3, -8.6204400286E-3, -1.4807609841E-2,
                   -8.0871898681E-3, -4.6670897864E-3, -4.0110000555E-5,
                   5.7109999034E-5, 8.2846998703E-4, 0.0000000000,
                   -8.6204400286E-3, -1.4917500317E-2, 4.4814897701E-3,
                   -2.3298799060E-3, 2.0865600090E-3, 6.9718998857E-3,
                   9.7479997203E-4, -0.0000000000, -8.2846998703E-4,
                   -1.4807609841E-2, 4.4814897701E-3, -3.2785870135E-2,
                   -5.0797060132E-2, 2.4427289143E-2, 1.2145539746E-2,
                   0.0000000000, -9.7479997203E-4, -5.7109999034E-5,
                   -8.0871898681E-3, -2.3298799060E-3, -5.0797060132E-2,
                   8.2495503128E-2, 0.1510555595, -0.0000000000,
                   -1.2145539746E-2, -6.9718998857E-3, 4.0110000555E-5,
                   -4.6670897864E-3, 2.0865600090E-3, 2.4427289143E-2,
                   0.1510555595, 0.0000000000, -0.1510555595,
                   -2.4427289143E-2, -2.0865600090E-3, 4.6670897864E-3,
                   -4.0110000555E-5, 6.9718998857E-3, 1.2145539746E-2,
                   0.0000000000, -0.1510555595, -8.2495503128E-2,
                   5.0797060132E-2, 2.3298799060E-3, 8.0871898681E-3,
                   5.7109999034E-5, 9.7479997203E-4, -0.0000000000,
                   -1.2145539746E-2, -2.4427289143E-2, 5.0797060132E-2,
                   3.2785870135E-2, -4.4814897701E-3, 1.4807609841E-2,
                   8.2846998703E-4, -0.0000000000, -9.7479997203E-4,
                   -6.9718998857E-3, -2.0865600090E-3, 2.3298799060E-3,
                   -4.4814897701E-3, 1.4917500317E-2, 8.6204400286E-3,
                   0.0000000000, -8.2846998703E-4, -5.7109999034E-5,
                   4.0110000555E-5, 4.6670897864E-3, 8.0871898681E-3,
                   1.4807609841E-2, 8.6204400286E-3, -3.1221499667E-3]]).T)
    return filters


def _sp5_filters():
    filters = {}
    filters['harmonics'] = np.array([1, 3, 5])
    filters['mtx'] = (
        np.array([[0.3333, 0.2887, 0.1667, 0.0000, -0.1667, -0.2887],
                  [0.0000, 0.1667, 0.2887, 0.3333, 0.2887, 0.1667],
                  [0.3333, -0.0000, -0.3333, -0.0000, 0.3333, -0.0000],
                  [0.0000, 0.3333, 0.0000, -0.3333, 0.0000, 0.3333],
                  [0.3333, -0.2887, 0.1667, -0.0000, -0.1667, 0.2887],
                  [-0.0000, 0.1667, -0.2887, 0.3333, -0.2887, 0.1667]]))
    filters['hi0filt'] = (
        np.array([[-0.00033429, -0.00113093, -0.00171484,
                   -0.00133542, -0.00080639, -0.00133542,
                   -0.00171484, -0.00113093, -0.00033429],
                  [-0.00113093, -0.00350017, -0.00243812,
                   0.00631653, 0.01261227, 0.00631653,
                   -0.00243812, -0.00350017, -0.00113093],
                  [-0.00171484, -0.00243812, -0.00290081,
                   -0.00673482, -0.00981051, -0.00673482,
                   -0.00290081, -0.00243812, -0.00171484],
                  [-0.00133542, 0.00631653, -0.00673482,
                   -0.07027679, -0.11435863, -0.07027679,
                   -0.00673482, 0.00631653, -0.00133542],
                  [-0.00080639, 0.01261227, -0.00981051,
                   -0.11435863, 0.81380200, -0.11435863,
                   -0.00981051, 0.01261227, -0.00080639],
                  [-0.00133542, 0.00631653, -0.00673482,
                   -0.07027679, -0.11435863, -0.07027679,
                   -0.00673482, 0.00631653, -0.00133542],
                  [-0.00171484, -0.00243812, -0.00290081,
                   -0.00673482, -0.00981051, -0.00673482,
                   -0.00290081, -0.00243812, -0.00171484],
                  [-0.00113093, -0.00350017, -0.00243812,
                   0.00631653, 0.01261227, 0.00631653,
                   -0.00243812, -0.00350017, -0.00113093],
                  [-0.00033429, -0.00113093, -0.00171484,
                   -0.00133542, -0.00080639, -0.00133542,
                   -0.00171484, -0.00113093, -0.00033429]]))
    filters['lo0filt'] = (
        np.array([[0.00341614, -0.01551246, -0.03848215, -0.01551246,
                  0.00341614],
                 [-0.01551246, 0.05586982, 0.15925570, 0.05586982,
                  -0.01551246],
                 [-0.03848215, 0.15925570, 0.40304148, 0.15925570,
                  -0.03848215],
                 [-0.01551246, 0.05586982, 0.15925570, 0.05586982,
                  -0.01551246],
                 [0.00341614, -0.01551246, -0.03848215, -0.01551246,
                  0.00341614]]))
    filters['lofilt'] = (
        2 * np.array([[0.00085404, -0.00244917, -0.00387812, -0.00944432,
                       -0.00962054, -0.00944432, -0.00387812, -0.00244917,
                       0.00085404],
                      [-0.00244917, -0.00523281, -0.00661117, 0.00410600,
                       0.01002988, 0.00410600, -0.00661117, -0.00523281,
                       -0.00244917],
                      [-0.00387812, -0.00661117, 0.01396746, 0.03277038,
                       0.03981393, 0.03277038, 0.01396746, -0.00661117,
                       -0.00387812],
                      [-0.00944432, 0.00410600, 0.03277038, 0.06426333,
                       0.08169618, 0.06426333, 0.03277038, 0.00410600,
                       -0.00944432],
                      [-0.00962054, 0.01002988, 0.03981393, 0.08169618,
                       0.10096540, 0.08169618, 0.03981393, 0.01002988,
                       -0.00962054],
                      [-0.00944432, 0.00410600, 0.03277038, 0.06426333,
                       0.08169618, 0.06426333, 0.03277038, 0.00410600,
                       -0.00944432],
                      [-0.00387812, -0.00661117, 0.01396746, 0.03277038,
                       0.03981393, 0.03277038, 0.01396746, -0.00661117,
                       -0.00387812],
                      [-0.00244917, -0.00523281, -0.00661117, 0.00410600,
                       0.01002988, 0.00410600, -0.00661117, -0.00523281,
                       -0.00244917],
                      [0.00085404, -0.00244917, -0.00387812, -0.00944432,
                       -0.00962054, -0.00944432, -0.00387812, -0.00244917,
                       0.00085404]]))
    filters['bfilts'] = (
        np.array([[0.00277643, 0.00496194, 0.01026699, 0.01455399, 0.01026699,
                   0.00496194, 0.00277643, -0.00986904, -0.00893064,
                   0.01189859, 0.02755155, 0.01189859, -0.00893064,
                   -0.00986904, -0.01021852, -0.03075356, -0.08226445,
                   -0.11732297, -0.08226445, -0.03075356, -0.01021852,
                   0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000,
                   0.00000000, 0.00000000, 0.01021852, 0.03075356, 0.08226445,
                   0.11732297, 0.08226445, 0.03075356, 0.01021852, 0.00986904,
                   0.00893064, -0.01189859, -0.02755155, -0.01189859,
                   0.00893064, 0.00986904, -0.00277643, -0.00496194,
                   -0.01026699, -0.01455399, -0.01026699, -0.00496194,
                   -0.00277643],
                  [-0.00343249, -0.00640815, -0.00073141, 0.01124321,
                   0.00182078, 0.00285723, 0.01166982, -0.00358461,
                   -0.01977507, -0.04084211, -0.00228219, 0.03930573,
                   0.01161195, 0.00128000, 0.01047717, 0.01486305,
                   -0.04819057, -0.12227230, -0.05394139, 0.00853965,
                   -0.00459034, 0.00790407, 0.04435647, 0.09454202,
                   -0.00000000, -0.09454202, -0.04435647, -0.00790407,
                   0.00459034, -0.00853965, 0.05394139, 0.12227230,
                   0.04819057, -0.01486305, -0.01047717, -0.00128000,
                   -0.01161195, -0.03930573, 0.00228219, 0.04084211,
                   0.01977507, 0.00358461, -0.01166982, -0.00285723,
                   -0.00182078, -0.01124321, 0.00073141, 0.00640815,
                   0.00343249],
                  [0.00343249, 0.00358461, -0.01047717, -0.00790407,
                   -0.00459034, 0.00128000, 0.01166982, 0.00640815,
                   0.01977507, -0.01486305, -0.04435647, 0.00853965,
                   0.01161195, 0.00285723, 0.00073141, 0.04084211, 0.04819057,
                   -0.09454202, -0.05394139, 0.03930573, 0.00182078,
                   -0.01124321, 0.00228219, 0.12227230, -0.00000000,
                   -0.12227230, -0.00228219, 0.01124321, -0.00182078,
                   -0.03930573, 0.05394139, 0.09454202, -0.04819057,
                   -0.04084211, -0.00073141, -0.00285723, -0.01161195,
                   -0.00853965, 0.04435647, 0.01486305, -0.01977507,
                   -0.00640815, -0.01166982, -0.00128000, 0.00459034,
                   0.00790407, 0.01047717, -0.00358461, -0.00343249],
                  [-0.00277643, 0.00986904, 0.01021852, -0.00000000,
                   -0.01021852, -0.00986904, 0.00277643, -0.00496194,
                   0.00893064, 0.03075356, -0.00000000, -0.03075356,
                   -0.00893064, 0.00496194, -0.01026699, -0.01189859,
                   0.08226445, -0.00000000, -0.08226445, 0.01189859,
                   0.01026699, -0.01455399, -0.02755155, 0.11732297,
                   -0.00000000, -0.11732297, 0.02755155, 0.01455399,
                   -0.01026699, -0.01189859, 0.08226445, -0.00000000,
                   -0.08226445, 0.01189859, 0.01026699, -0.00496194,
                   0.00893064, 0.03075356, -0.00000000, -0.03075356,
                   -0.00893064, 0.00496194, -0.00277643, 0.00986904,
                   0.01021852, -0.00000000, -0.01021852, -0.00986904,
                   0.00277643],
                  [-0.01166982, -0.00128000, 0.00459034, 0.00790407,
                   0.01047717, -0.00358461, -0.00343249, -0.00285723,
                   -0.01161195, -0.00853965, 0.04435647, 0.01486305,
                   -0.01977507, -0.00640815, -0.00182078, -0.03930573,
                   0.05394139, 0.09454202, -0.04819057, -0.04084211,
                   -0.00073141, -0.01124321, 0.00228219, 0.12227230,
                   -0.00000000, -0.12227230, -0.00228219, 0.01124321,
                   0.00073141, 0.04084211, 0.04819057, -0.09454202,
                   -0.05394139, 0.03930573, 0.00182078, 0.00640815,
                   0.01977507, -0.01486305, -0.04435647, 0.00853965,
                   0.01161195, 0.00285723, 0.00343249, 0.00358461,
                   -0.01047717, -0.00790407, -0.00459034, 0.00128000,
                   0.01166982],
                  [-0.01166982, -0.00285723, -0.00182078, -0.01124321,
                   0.00073141, 0.00640815, 0.00343249, -0.00128000,
                   -0.01161195, -0.03930573, 0.00228219, 0.04084211,
                   0.01977507, 0.00358461, 0.00459034, -0.00853965,
                   0.05394139, 0.12227230, 0.04819057, -0.01486305,
                   -0.01047717, 0.00790407, 0.04435647, 0.09454202,
                   -0.00000000, -0.09454202, -0.04435647, -0.00790407,
                   0.01047717, 0.01486305, -0.04819057, -0.12227230,
                   -0.05394139, 0.00853965, -0.00459034, -0.00358461,
                   -0.01977507, -0.04084211, -0.00228219, 0.03930573,
                   0.01161195, 0.00128000, -0.00343249, -0.00640815,
                   -0.00073141, 0.01124321, 0.00182078, 0.00285723,
                   0.01166982]]).T)
    return filters
