"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addForwarder = void 0;
const crypto = require("crypto");
const lambda = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_logs_destinations_1 = require("@aws-cdk/aws-logs-destinations");
const loglevel_1 = require("loglevel");
const SUBSCRIPTION_FILTER_PREFIX = "DatadogSubscriptionFilter";
function generateForwaderConstructId(forwarderArn) {
    loglevel_1.default.debug("Generating construct Id for Datadog Lambda Forwarder");
    return "forwarder" + crypto.createHash("sha256").update(forwarderArn).digest("hex");
}
function generateSubscriptionFilterName(functionArn, forwarderArn) {
    const subscriptionFilterValue = crypto
        .createHash("sha256")
        .update(functionArn)
        .update(forwarderArn)
        .digest("hex");
    const subscriptionFilterValueLength = subscriptionFilterValue.length;
    const subscriptionFilterName = SUBSCRIPTION_FILTER_PREFIX +
        subscriptionFilterValue.substring(subscriptionFilterValueLength - 8, subscriptionFilterValueLength);
    return subscriptionFilterName;
}
function addForwarder(scope, lambdaFunctions, forwarderArn) {
    const forwarderConstructId = generateForwaderConstructId(forwarderArn);
    let forwarder;
    if (scope.node.tryFindChild(forwarderConstructId)) {
        forwarder = scope.node.tryFindChild(forwarderConstructId);
    }
    else {
        forwarder = lambda.Function.fromFunctionArn(scope, forwarderConstructId, forwarderArn);
    }
    const forwarderDestination = new aws_logs_destinations_1.LambdaDestination(forwarder);
    lambdaFunctions.forEach((lam) => {
        const subscriptionFilterName = generateSubscriptionFilterName(lam.functionArn, forwarderArn);
        loglevel_1.default.debug(`Adding log subscription ${subscriptionFilterName} for ${lam.functionName}`);
        lam.logGroup.addSubscriptionFilter(subscriptionFilterName, {
            destination: forwarderDestination,
            filterPattern: aws_logs_1.FilterPattern.allEvents(),
        });
    });
}
exports.addForwarder = addForwarder;
//# sourceMappingURL=data:application/json;base64,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