"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpGatewayRoutePathMatch = exports.HttpRoutePathMatch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Defines HTTP route matching based on the URL path of the request.
 *
 * @stability stable
 */
class HttpRoutePathMatch {
    /**
     * The value of the path must match the specified value exactly.
     *
     * The provided `path` must start with the '/' character.
     *
     * @param path the exact path to match on.
     * @stability stable
     */
    static exactly(path) {
        return new HttpRouteWholePathMatch({ exact: path });
    }
    /**
     * The value of the path must match the specified regex.
     *
     * @param regex the regex used to match the path.
     * @stability stable
     */
    static regex(regex) {
        return new HttpRouteWholePathMatch({ regex: regex });
    }
    /**
     * The value of the path must match the specified prefix.
     *
     * @param prefix the value to use to match the beginning of the path part of the URL of the request.
     * @stability stable
     */
    static startsWith(prefix) {
        return new HttpRoutePrefixPathMatch(prefix);
    }
}
exports.HttpRoutePathMatch = HttpRoutePathMatch;
_a = JSII_RTTI_SYMBOL_1;
HttpRoutePathMatch[_a] = { fqn: "@aws-cdk/aws-appmesh.HttpRoutePathMatch", version: "1.139.0" };
class HttpRoutePrefixPathMatch extends HttpRoutePathMatch {
    constructor(prefix) {
        super();
        this.prefix = prefix;
        if (prefix && prefix[0] !== '/') {
            throw new Error(`Prefix Path for the match must start with \'/\', got: ${prefix}`);
        }
    }
    bind(_scope) {
        return {
            prefixPathMatch: this.prefix,
        };
    }
}
class HttpRouteWholePathMatch extends HttpRoutePathMatch {
    constructor(match) {
        super();
        this.match = match;
        if (match.exact && match.exact[0] !== '/') {
            throw new Error(`Exact Path for the match must start with \'/\', got: ${match.exact}`);
        }
    }
    bind(_scope) {
        return {
            wholePathMatch: this.match,
        };
    }
}
/**
 * Defines HTTP gateway route matching based on the URL path of the request.
 *
 * @stability stable
 */
class HttpGatewayRoutePathMatch {
    /**
     * The value of the path must match the specified prefix.
     *
     * @param prefix the value to use to match the beginning of the path part of the URL of the request.
     * @param rewriteTo Specify either disabling automatic rewrite or rewriting to specified prefix path.
     * @stability stable
     */
    static startsWith(prefix, rewriteTo) {
        return new HttpGatewayRoutePrefixPathMatch(prefix, rewriteTo);
    }
    /**
     * The value of the path must match the specified value exactly.
     *
     * The provided `path` must start with the '/' character.
     *
     * @param path the exact path to match on.
     * @param rewriteTo the value to substitute for the matched part of the path of the gateway request URL As a default, retains original request's URL path.
     * @stability stable
     */
    static exactly(path, rewriteTo) {
        return new HttpGatewayRouteWholePathMatch({ exact: path }, rewriteTo);
    }
    /**
     * The value of the path must match the specified regex.
     *
     * @param regex the regex used to match the path.
     * @param rewriteTo the value to substitute for the matched part of the path of the gateway request URL As a default, retains original request's URL path.
     * @stability stable
     */
    static regex(regex, rewriteTo) {
        return new HttpGatewayRouteWholePathMatch({ regex }, rewriteTo);
    }
}
exports.HttpGatewayRoutePathMatch = HttpGatewayRoutePathMatch;
_b = JSII_RTTI_SYMBOL_1;
HttpGatewayRoutePathMatch[_b] = { fqn: "@aws-cdk/aws-appmesh.HttpGatewayRoutePathMatch", version: "1.139.0" };
class HttpGatewayRoutePrefixPathMatch extends HttpGatewayRoutePathMatch {
    constructor(prefixPathMatch, rewriteTo) {
        super();
        this.prefixPathMatch = prefixPathMatch;
        this.rewriteTo = rewriteTo;
        if (prefixPathMatch[0] !== '/') {
            throw new Error('Prefix path for the match must start with \'/\', '
                + `got: ${prefixPathMatch}`);
        }
        if (rewriteTo) {
            if (prefixPathMatch[prefixPathMatch.length - 1] !== '/') {
                throw new Error('When prefix path for the rewrite is specified, prefix path for the match must end with \'/\', '
                    + `got: ${prefixPathMatch}`);
            }
            if (rewriteTo[0] !== '/' || rewriteTo[rewriteTo.length - 1] !== '/') {
                throw new Error('Prefix path for the rewrite must start and end with \'/\', '
                    + `got: ${rewriteTo}`);
            }
        }
    }
    bind(_scope) {
        return {
            prefixPathMatch: this.prefixPathMatch,
            prefixPathRewrite: this.rewriteTo === undefined
                ? undefined
                : {
                    defaultPrefix: this.rewriteTo === '' ? 'DISABLED' : undefined,
                    value: this.rewriteTo === '' ? undefined : this.rewriteTo,
                },
        };
    }
}
class HttpGatewayRouteWholePathMatch extends HttpGatewayRoutePathMatch {
    constructor(wholePathMatch, exactPathRewrite) {
        super();
        this.wholePathMatch = wholePathMatch;
        this.exactPathRewrite = exactPathRewrite;
        if (wholePathMatch.exact && wholePathMatch.exact[0] !== '/') {
            throw new Error(`Exact Path for the match must start with \'/\', got: ${wholePathMatch.exact}`);
        }
        if (exactPathRewrite === '') {
            throw new Error('Exact Path for the rewrite cannot be empty. Unlike startsWith() method, no automatic rewrite on whole path match');
        }
        if (exactPathRewrite && exactPathRewrite[0] !== '/') {
            throw new Error(`Exact Path for the rewrite must start with \'/\', got: ${exactPathRewrite}`);
        }
    }
    bind(_scope) {
        return {
            wholePathMatch: this.wholePathMatch,
            wholePathRewrite: this.exactPathRewrite === undefined ? undefined : { exact: this.exactPathRewrite },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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