import * as React from 'react';
import { ReactWidget, showErrorMessage } from '@jupyterlab/apputils';
import { simSetupConfigIcon } from '../style/icons';
/**
 * This class implements a main area widget (React widget)
 * for configuring a simulation setup.
 */
export class SimSetupConfigureWidget extends ReactWidget {
    /**
     * Create an instance of the widget.
     * @param options - widget initialization options
     * @returns widget instance
     */
    constructor(options) {
        super(options);
        // Define CSS id.
        this.id = 'mosaik-docker-sim-setup-config';
        // Set widget title icon.
        this.title.icon = simSetupConfigIcon;
        // Set widget title label.
        this.title.label = 'Sim Setup Configuration';
        // Make widget closable.
        this.title.closable = true;
        this._model = options.model;
        this._configData = options.configData;
    }
    /**
     * Render the widget.
     * @returns React widget
     */
    render() {
        return (React.createElement(SimSetupConfigureComponent, { model: this._model, widget: this, configData: this._configData }));
    }
    /**
     * Retrieve the configuration data specified by the user.
     * @returns simulation setup configuration
     */
    getConfigData() {
        // Retrieve HTML input fields.
        const inputNodesList = this.node.querySelectorAll('input');
        return {
            scenarioFile: inputNodesList[0].value,
            dockerFile: inputNodesList[1].value,
            extraFiles: inputNodesList[2].value.split(','),
            extraDirs: inputNodesList[3].value.split(','),
            results: inputNodesList[4].value.split(',')
        };
    }
}
/**
 * This class implements the React component for class SimSetupConfigureWidget.
 */
export class SimSetupConfigureComponent extends React.Component {
    /**
     * Create an instance of the widget's React component.
     * @param props - React component properties
     * @returns React component instance
     */
    constructor(props) {
        super(props);
        this._simSetupDir = this.props.model.simSetupRoot;
    }
    /**
     * Render the React element.
     * @returns React element
     */
    render() {
        // Retrieve the initial configuration data.
        const { orchestrator } = this.props.configData;
        const scenarioFile = orchestrator.scenarioFile;
        const dockerFile = orchestrator.dockerFile;
        const extraFiles = orchestrator.extraFiles.join(', ');
        const extraDirs = orchestrator.extraDirs.join(', ');
        const results = orchestrator.results.join(', ');
        // Define the HTML structure of the widget (header, input fields and update button).
        return (React.createElement("div", { className: "jp-Widget" },
            React.createElement("div", { className: "jp-Content" },
                React.createElement("span", { className: "jp-Content-header" },
                    "Simulation setup location: ",
                    this._simSetupDir),
                React.createElement("span", { className: "jp-Content-input-header" }, "Name of the mosaik scenario file:"),
                React.createElement("input", { defaultValue: scenarioFile, className: "jp-Content-input" }),
                React.createElement("span", { className: "jp-Content-input-header" }, "Path to the Dockerfile for the simulation orchestrator image:"),
                React.createElement("input", { defaultValue: dockerFile, className: "jp-Content-input" }),
                React.createElement("span", { className: "jp-Content-input-header" }, "Additional files to be added to the simulation orchestrator image (comma-separated list):"),
                React.createElement("input", { defaultValue: extraFiles, className: "jp-Content-input" }),
                React.createElement("span", { className: "jp-Content-input-header" }, "Additional folders to be added to the simulation orchestrator image (comma-separated list):"),
                React.createElement("input", { defaultValue: extraDirs, className: "jp-Content-input" }),
                React.createElement("span", { className: "jp-Content-input-header" }, "Results files or folders produced by the simulation that should be retrieved after the simulation has finished (comma-separated list):"),
                React.createElement("input", { defaultValue: results, className: "jp-Content-input" })),
            React.createElement("div", { className: "jp-Dialog-span" },
                React.createElement("button", { className: "jp-mod-reject jp-mod-styled", onClick: async () => {
                        const data = this.props.widget.getConfigData();
                        try {
                            await this.props.model.configureSimSetup(data, this._simSetupDir);
                        }
                        catch (error) {
                            console.error(error);
                            showErrorMessage('An error occurred while attempting to display the simulation status!\n', error);
                        }
                    } }, "UPDATE CONFIGURATION"))));
    }
}
