import { URLExt } from '@jupyterlab/coreutils';
import { KernelManager, ServerConnection } from '@jupyterlab/services';
import { UUID } from '@lumino/coreutils';
/**
 * This namespace contains the interface definitions and methods required
 * for the low-level interaction with the functionality provided by
 * package mosaik_docker_jl on the server.
 */
export var MosaikDockerAPI;
(function (MosaikDockerAPI) {
    /**
     * Send an HTTP request to the server (mosaik_docker_jl backend).
     * The request will be parsed there and the appropriate API will be called (blocking).
     * The results from this API call are then returned (asynchronously).
     * @param endPoint - request endpoints (as defined by the mosaik_docker_jl backend)
     *
     * @param method - HTTP method
     * @param request - payload of HTTP request
     * @returns Low-level API response
     */
    async function sendRequest(endPoint, method = 'GET', request = null) {
        // Construct complete request.
        let fullRequest;
        if (request === null) {
            fullRequest = {
                method: method
            };
        }
        else {
            fullRequest = {
                method: method,
                body: JSON.stringify(request)
            };
        }
        // Retrieve server connection settings.
        const settings = ServerConnection.makeSettings();
        // Specify URL for communicating with the mosaik_docker_jl backend.
        const requestUrl = URLExt.join(settings.baseUrl, 'mosaik_docker_jl', // API Namespace
        endPoint);
        // Send request to server.
        let response;
        try {
            response = await ServerConnection.makeRequest(requestUrl, fullRequest, settings);
        }
        catch (error) {
            return Promise.reject(new ServerConnection.NetworkError(error));
        }
        // Retrieve low-level API response.
        let data;
        try {
            data = await response.json();
        }
        catch (error) {
            return Promise.reject({ error, response });
        }
        if (!response.ok) {
            return Promise.reject(new ServerConnection.ResponseError(response));
        }
        return Promise.resolve(data);
    }
    MosaikDockerAPI.sendRequest = sendRequest;
    /**
     * Execute Python code on the server (via a dedicated Python kernel).
     * In the Python code, an instance of class CallbackComm (imported from
     * mosaik_docker_jl.comm) named 'comm' is available, which can be used
     * to send back data and trigger the execution of the callback functions
     * 'onMsg' and 'onClose'.
     *
     * @param commTargetPrefix - prefix used for the kernel comm target
     * @param executeCode - Python code to be executed
     * @param onMsg - data from the kernel can be sent back to the frontend asynchronously via this callback function
     * @param onClose - this callback function is executed when the kernel closes
     * @returns execution status
     */
    async function executeWithCallbacks(commTargetPrefix, executeCode, onMsg, onClose) {
        const commTarget = commTargetPrefix + UUID.uuid4();
        // Start a python kernel
        const kernelManager = new KernelManager();
        const kernel = await kernelManager.startNew({ name: 'python' });
        // Register a new comm target and set the callbacks.
        kernel.registerCommTarget(commTarget, (comm, commMsg) => {
            if (commMsg.content.target_name !== commTarget) {
                return;
            }
            comm.onMsg = onMsg;
            comm.onClose = onClose;
        });
        // Add boilerplate code for initializing the callback interface (comm).
        const code = 'from mosaik_docker_jl.comm import CallbackComm\n' +
            `comm = CallbackComm( '${commTarget}' )\n` +
            executeCode;
        // Execute the code on the kernel.
        const reply = await kernel.requestExecute({
            code: code
        }).done;
        const status = await reply.content.status;
        // The kernel is no longer needed, try to shut it down.
        try {
            await kernel.shutdown();
        }
        catch (error) {
            return Promise.reject({ status: 'kernel-error', error: error });
        }
        if (status === 'error') {
            const errorMessage = reply.content;
            return Promise.reject({ status: status, error: errorMessage.evalue });
        }
        return Promise.resolve({ status: status });
    }
    MosaikDockerAPI.executeWithCallbacks = executeWithCallbacks;
})(MosaikDockerAPI || (MosaikDockerAPI = {}));
