import { ILayoutRestorer, JupyterFrontEnd } from '@jupyterlab/application';
import { FileBrowser } from '@jupyterlab/filebrowser';
import { ISignal } from '@lumino/signaling';
import { IMosaikDockerExtension, MosaikDocker } from './tokens';
export declare namespace MosaikDockerExtension {
    /**
     * Definition of initialization options of class MosaikDockerExtension.
     */
    interface IOptions {
        /** Frontend application. */
        app: JupyterFrontEnd;
        /** File browser extension */
        fileBrowser: FileBrowser;
        /** Application layout restorer */
        restorer: ILayoutRestorer | null;
    }
}
/**
 * Main extension class, implementing the mosaik-docker extension model.
 * The model is supposed to act on the currently activated simulation setup.
 * A simulation setup is activated by navigating to it with the help of the
 * file browser (i.e., the simulation setup's root folder or one of its
 * subfolders is the current working directory).
 */
export declare class MosaikDockerExtension implements IMosaikDockerExtension {
    /**
     * Returns an instance of the mosaik-docker extension model.
     *
     * @param options - initialization options
     * @returns instance of extension model
     */
    constructor(options: MosaikDockerExtension.IOptions);
    /**
     * Retrieve the installed version of the extension.
     * @returns version information
     */
    getVersion(): Promise<MosaikDocker.IVersion>;
    /**
     * Create a new simulation setup in the current working directory.
     * @param name - name of new simulation setup
     * @returns creation status
     */
    createSimSetup(name: string): Promise<MosaikDocker.ICreateSimSetupStatus>;
    /**
     * Configure a simulation setup.
     * @param data - simulation setup configuration data
     * @param dir - path to simulation setup (active simulation setup if not specified)
     * @returns configuration status
     */
    configureSimSetup(data: MosaikDocker.IOrchestratorConfigData, dir?: string): Promise<MosaikDocker.IConfigureSimSetupStatus>;
    /**
     * Build currently active simulation setup.
     */
    buildSimSetup(): Promise<void>;
    /**
     * Check currently active simulation setup.
     * @returns check status
     */
    checkSimSetup(): Promise<MosaikDocker.ICheckSimSetupStatus>;
    /**
     * Delete currently active simulation setup.
     * @returns deletion status
     */
    deleteSimSetup(): Promise<MosaikDocker.IDeleteSimSetupStatus>;
    /**
     * Start a new simulation for the currently active simulation setup.
     * @returns simulation status
     */
    startSim(): Promise<MosaikDocker.IStartSimStatus>;
    /**
     * Cancel a running simulation of the currently active simulation setup.
     * @param simId - ID of the running simulation to be cancelled
     * @returns cancel status
     */
    cancelSim(simId: string): Promise<MosaikDocker.ICancelSimStatus>;
    /**
     * Clear a finished simulation (i.e., remove its ID from the list of
     * finished simulations).
     * @param simId - ID of the finished simulation to be cleared
     * @returns clearing status
     */
    clearSim(simId: string): Promise<MosaikDocker.IClearSimStatus>;
    /**
     * Retrieve the results of a finished simulation and store them in a
     * subfolder of the simulation setup root folder. The subfolder name
     * is the simulation ID.
     * @param simId - ID of the finished simulation for which the results are to be retrieved
     * @returns results retrieval status
     */
    getSimResults(simId: string): Promise<MosaikDocker.IGetSimResultsStatus>;
    /**
     * Get the IDs of all running and finished simulations for the currently
     * active simulation setup.
     * @returns IDs of all running and finished simulations
     */
    getSimIds(): Promise<MosaikDocker.ISimIds>;
    /**
     * Inquire the status of all running and finished simulations for the
     * currently active simulation setup.
     * @returns status of all running and finished simulations
     */
    getSimStatus(): Promise<MosaikDocker.ISimStatus>;
    /**
     * Get the configuration data of the currently active simulation setup.
     * @returns configuration data
     */
    getSimSetupConfigData(): Promise<MosaikDocker.IConfigData>;
    /**
     * Display the status of all running and finished simulations for the
     * currently active simulation setup in a separate main area widget.
     */
    displaySimStatus(): Promise<void>;
    /**
     * Display the configuration data for the currently active simulation
     * setup in a separate main area widget. Changes can be applied to the
     * simulation setup configuration.
     */
    displaySimSetupConfiguration(): Promise<void>;
    /**
     * This flag indicates if the current working directory is part of a
     * valid simulation setup.
     */
    get isValidSimSetup(): boolean;
    /**
     * Points to the root directory of the currently active simulation setup
     * (absolute path).
     */
    get simSetupRoot(): string;
    /**
     * Points to the user's JupyterLab home directory (absolute path).
     */
    get userHomeDir(): string;
    /**
     * Signal that indicates whether the state of the extension model has changed.
     */
    get modelChanged(): ISignal<this, void>;
    /**
     * Get whether the model is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Dispose of the resources held by the model.
     */
    dispose(): void;
    /**
     * This function is called every time the working directory changes,
     * in order to check if the working directory is (a subfolder of) a
     * valid simulation setup.
     * @private
     * @param emitter - file browser model
     */
    private _checkIfValidSimDir;
    /**
     * Try to retrieve the path to the currently active simulation setup's
     * root directory. If the current working directory is not (a subfolder
     * of) a valid simulation setup, set the validity flag to false.
     * @private
     */
    private _getSimSetupRoot;
    /**
     * Signal for notifying that the extension state has changed.
     * ALso update the sim status widget if it is active.
     * @private
     */
    private _notifyStateChanged;
    /**
     * Retrieve absolute path to the user's JupyterLab home directory.
     * @private
     */
    private _retrieveUserHomeDir;
    private _app;
    private _fileBrowser;
    private _simStatusWidget;
    private _simStatusWidgetTracker;
    private _isValidSimSetup;
    private _simSetupRoot;
    private _userHomeDir;
    private _modelChanged;
    private _isDisposed;
}
