from typing import Union, List

from .constants import NOTES, ROOTS


def note_to_midi(note: Union[str, int]) -> int:
    """
    Convert a note name (eg: "C3") into its corresponding midi note value (eg: 60).
    If passed a valid midi note, this function will simply return it.

    Args:
        note: A valid note name from `midi_utils.constants.NOTES`
              or a midi note between 0 and 127.
    """
    if isinstance(note, str):
        note = note.upper()
        if note in NOTES:
            return NOTES[note]
        raise ValueError(f'"{note}" is not a valid note.')
    if note < 0 or note > 127:
        raise ValueError(
            f'"{note}" is not a valid midi note. Must be between 0 and 127'
        )
    return note


def root_to_midi(root: Union[str, int]) -> int:
    """
    Convert a root name (eg: "C") into its corresponding midi note  .
    If passed a valid midi note value, this function will simply return it.

    Args:
        root: A valid root name from `midi_utils.constants.NOTES`
              or a midi note between 0 and 11.
    """
    if isinstance(root, str):
        root = root.upper()
        if root in ROOTS:
            return ROOTS[root]
        raise ValueError(
            f'"{root}" is not a valid root name. Choose from: {", ".join(ROOTS.keys())}'
        )
    if root not in ROOTS.values():
        raise ValueError(
            f'"{root}" is not a valid midi root. Choose from: {", ".join(ROOTS.values())}'
        )
    return root
