"""Unit tests for the manifest formatter."""

import json

from dbt_score.evaluation import ModelResultsType
from dbt_score.formatters.manifest_formatter import ManifestFormatter
from dbt_score.rule import RuleViolation
from dbt_score.scoring import Score


def test_manifest_formatter_model(
    capsys,
    default_config,
    manifest_loader,
    model1,
    rule_severity_low,
    rule_severity_medium,
    rule_severity_critical,
):
    """Ensure the formatter has the correct output after model evaluation."""
    formatter = ManifestFormatter(
        manifest_loader=manifest_loader, config=default_config
    )
    results: ModelResultsType = {
        rule_severity_low: None,
        rule_severity_medium: Exception("Oh noes"),
        rule_severity_critical: RuleViolation("Error"),
    }
    formatter.model_evaluated(model1, results, Score(10.0, "🥇"))
    stdout = capsys.readouterr().out
    assert stdout == ""


def test_manifest_formatter_project(
    capsys,
    default_config,
    manifest_loader,
    model1,
    model2,
    rule_severity_low,
    rule_severity_medium,
    rule_severity_critical,
):
    """Ensure the formatter has the correct output after project evaluation."""
    formatter = ManifestFormatter(
        manifest_loader=manifest_loader, config=default_config
    )
    result1: ModelResultsType = {
        rule_severity_low: None,
        rule_severity_medium: Exception("Oh noes"),
        rule_severity_critical: RuleViolation("Error"),
    }
    result2: ModelResultsType = {
        rule_severity_low: None,
        rule_severity_medium: None,
        rule_severity_critical: None,
    }

    formatter.model_evaluated(model1, result1, Score(5.0, "🚧"))
    formatter.model_evaluated(model2, result2, Score(10.0, "🥇"))
    formatter.project_evaluated(Score(7.5, "🥉"))
    stdout = capsys.readouterr().out
    new_manifest = json.loads(stdout)
    assert new_manifest["nodes"]["model.package.model1"]["meta"]["score"] == 5.0
    assert new_manifest["nodes"]["model.package.model1"]["meta"]["badge"] == "🚧"
    assert new_manifest["nodes"]["model.package.model2"]["meta"]["score"] == 10.0
    assert new_manifest["nodes"]["model.package.model2"]["meta"]["badge"] == "🥇"
