import enum
from django.db import models
from django.db.models.base import ModelBase
from django.utils import translation, timezone

_ = translation.gettext_lazy


class NumeratorManager(models.Manager):
    def get_queryset(self):
        return super().get_queryset()

    def get_by_natural_key(self, ctype, date_start, date_end):
        return self.get(ctype=ctype, date_start=date_start, date_end=date_end)


class NumeratorReset(enum.Enum):
    YEARLY = 'YEAR'
    MONTHLY = 'MONTH'
    FIXED = 'NEVER'


class Numerator(models.Model):
    """ ContentTypeCounter is used as autogenerated number register
        like autonumber for uuid based Model """

    class Meta:
        verbose_name = _('Numerator')
        verbose_name_plural = _('Numerators')
        unique_together = ('app_label', 'model', 'prefix', 'reset_mode', 'year', 'month')

    objects = NumeratorManager()

    app_label = models.CharField(
        max_length=50,
        verbose_name=_('application'))
    model = models.CharField(
        max_length=50,
        verbose_name=_('model'))
    prefix = models.CharField(
        max_length=50,
        null=True, blank=True,
        verbose_name=_('prefix'))
    reset_mode = models.CharField(
        max_length=50,
        choices=[(str(x.value), str(x.name)) for x in NumeratorReset],
        default=NumeratorReset.YEARLY.value,
        verbose_name=_('reset mode'))
    year = models.IntegerField()
    month = models.IntegerField(default=0)
    counter = models.PositiveIntegerField(
        default=0, verbose_name=_('Counter'))

    def __str__(self):
        return str(self.model)

    def natural_key(self):
        natural_key = (self.app_label, self.model, self.year, self.month)
        return natural_key

    def increase_counter(self):
        self.counter += 1
        return self.counter

    def decrease_counter(self):
        i = 1 if self.counter > 0 else 0
        self.counter -= i
        return self.counter

    def reset_counter(self):
        self.counter = 0
        return self.counter

    def get_mode(self):
        return Numerator.YEARLY

    @staticmethod
    def get_for_instance(obj):
        opts = obj._meta
        defaults = {
            'app_label': opts.app_label,
            'model': (opts.model_name if not obj.parent_prefix else obj.parent_model),
            'prefix': obj.get_doc_prefix(),
            'year': obj.get_date_field().year,
            'month': obj.get_date_field().month if obj.reset_mode == NumeratorReset.MONTHLY else 0
        }
        get_or_create = Numerator.objects.get_or_create
        ct_counter, created = get_or_create(**defaults, defaults=defaults)
        return ct_counter


class NumeratorMeta(ModelBase):
    """ Provide extra fields to child Model """

    def __new__(mcs, name, bases, attrs, **kwargs):
        new_class = super().__new__(mcs, name, bases, attrs, **kwargs)
        abstract = getattr(new_class._meta, 'abstract', False)
        created_at_field = models.DateTimeField(
            default=timezone.now, verbose_name=_("created at"))
        date_field = getattr(attrs, 'created_at', None)
        if not date_field and not abstract:
            new_class.add_to_class('created_at', created_at_field)
        return new_class


class NumeratorMixinBase(models.Model):
    class Meta:
        abstract = True

    zero_fill = 4
    doc_prefix = ''
    numerator = None
    parent_prefix = False
    parent_model = NotImplemented
    inner_id_field = NotImplemented
    create_date_field = NotImplemented
    reset_mode = NumeratorReset.YEARLY

    reg_number = models.PositiveIntegerField(
        null=True,
        blank=True,
        editable=False,
        verbose_name=_('Reg number'))

    def get_doc_prefix(self):
        """
            Give numerator unique prefix, doc prefix is Mandatory for
            Polymorphic Based Child Model, each child should implement unique doc_prefix,
            Override this method if custom prefix is needed
            """
        doc_prefix = getattr(self, 'doc_prefix', '')
        return doc_prefix

    def get_date_field(self):
        return getattr(self, self.create_date_field)

    def get_inner_id_field(self):
        return getattr(self, self.inner_id_field)

    def format_date(self, form="%y%m"):
        """ Format date to string """
        return self.get_date_field().strftime(form)

    def format_number(self):
        """ Format register number digit lead by zero """
        return str(self.reg_number).zfill(self.zero_fill)

    def format_inner_id(self):
        """ Inner ID final format """
        form = [
            self.get_doc_prefix(),
            self.format_date(),
            self.format_number()
        ]
        inner_id = '{}{}{}'.format(*form)
        return setattr(self, self.inner_id_field, inner_id)

    def get_numerator(self):
        """ Get or create numerator """
        numerator = Numerator.get_for_instance(self)
        return numerator

    def update_inner_id(self):
        """
        Register and get Numerator instance for
        this model, Get latest counter value and then
        generate and set inner_id
        """
        # Get Numerator instance for this model
        if self.numerator is None:
            self.numerator = self.get_numerator()

        # If reg_number is None get new one
        if self.reg_number is None:
            self.reg_number = self.numerator.increase_counter()

        # If reg number > counter set new counter value
        if self.reg_number > self.numerator.counter:
            self.numerator.counter = self.reg_number

        return self.format_inner_id()

    def save(self, *args, **kwargs):
        if self.numerator is None:
            self.update_inner_id()
        self.numerator.save()
        super().save(*args, **kwargs)


class NumeratorMixin(NumeratorMixinBase):
    """ Mixin for Numerator Model """

    class Meta:
        abstract = True

    inner_id = models.CharField(
        null=True, blank=True, unique=True,
        editable=False, max_length=50,
        verbose_name=_('Inner ID')
    )
    created_at = models.DateTimeField(
        default=timezone.now,
        verbose_name=_("created at"))

    inner_id_field = 'inner_id'
    create_date_field = 'created_at'
