import re
import operator as op
from copy import copy
from dataclasses import dataclass
from typing import Any

import click
import yaml
from linkml_runtime.utils.formatutils import camelcase, underscore
from linkml_runtime.utils.schemaview import SchemaView

from linkml_runtime_api.apiroot import ApiRoot

@dataclass
class ApiGenerator(ApiRoot):
    """
    Generates an API schema given a data schema
    """

    def serialize(self, container_class=None):
        sv = self.schemaview
        cns = sv.all_class(imports=False).keys()
        if container_class != None:
            cns = self._get_top_level_classes(container_class)

        src = sv.schema
        name = f'{src.name}_api'
        classes = {
            "LocalChange" : {
                "slots" : [
                    "value",
                    "path"
                ],
                "mixin" : True
            },
            "LocalQuery" : {
                "mixin" : True,
                "slots" : [
                    "target_class",
                    "path"
                ]
            }
        }
        schema = {

            "id" : "https://w3id.org/linkml/tests/kitchen_sink_api",
            "name" : name,
            "description" : f"API for querying and manipulating objects from the {src.name} schema",
            "prefixes" : {
                name : "https://w3id.org/linkml/tests/kitchen_sink_api/",
                "linkml" : "https://w3id.org/linkml/"
            },
            "default_prefix" : name,
            "imports" : [
                "linkml:types",
                src.name
            ],
            "default_range" : "string",
            "slots" : {
                "value" : {
                    "inlined" : True
                },
                "path" : {},
                "constraints" : {
                    "range" : "Any"
                },
                "id_value" : {},
                "target_class" : {}
            },
            "classes": classes
        }

        cmts =  ["This is an autogenerated class"]

        for cn in cns:
            c = sv.get_class(cn)
            cn_camel = camelcase(cn)
            if not c.abstract and not c.mixin:
                classes[f'Add{cn_camel}'] = {
                    "description": f'A change action that adds a {cn_camel} to a database',
                    "comments": copy(cmts),
                    "mixins" : "LocalChange",
                    "slot_usage" : {
                        "value" : {
                            "range" : cn,
                            "inlined" : True
                        }
                    }
                }
                classes[f'Remove{cn_camel}'] = {
                    "description": f'A change action that remoaves a {cn_camel} to a database',
                    "comments": copy(cmts),
                    "mixins" : "LocalChange",
                    "slot_usage" : {
                        "value" : {
                            "range" : cn,
                            "inlined" : True
                        }
                    }
                }
            classes[f'{cn_camel}Query'] = {
                "description": f'A query object for instances of {cn_camel} from a database',
                "comments": copy(cmts),
                "mixins" : "LocalChange",
                "slots" : [
                    "constraints"
                ],
            }
            classes[f'{cn_camel}FetchById'] = {
                "description": f'A query object for fetching an instance of {cn_camel} from a database by id',
                "comments": copy(cmts),
                "mixins" : "LocalChange",
                "slots" : [
                    "id_value"
                ],
            }
        return yaml.safe_dump(schema, sort_keys=False)

@click.command()
@click.option('-R', '--container-class', help="name of class that contains top level objects")
@click.argument('schema')
def cli(schema,  **args):
    """ Generate API """
    sv = SchemaView(schema)
    gen = ApiGenerator(schemaview=sv)
    print(gen.serialize(**args))


if __name__ == '__main__':
    cli()
