"""
validataclass
Copyright (c) 2021, binary butterfly GmbH and contributors
Use of this source code is governed by an MIT-style license that can be found in the LICENSE file.
"""

import dataclasses
import sys
from typing import Any, Optional, NoReturn

from validataclass.validators import Validator
from .defaults import Default, NoDefault, DefaultFactory, DefaultUnset

__all__ = [
    'validataclass_field',
]


def validataclass_field(
    validator: Validator,
    default: Any = NoDefault,
    *,
    metadata: Optional[dict] = None,
    _name: Optional[str] = None,  # noqa (undocumented parameter, only used internally)
    **kwargs
):
    """
    Define a dataclass field compatible with DataclassValidator.

    Wraps the regular `dataclasses.field()` function, but has special parameters to add validator metadata to the field.

    Additional keyword arguments will be passed to `dataclasses.field()`, with some exceptions:
    - 'default' is handled by this function to set metadata. It can be either a direct value or a `Default` object. It
      is then converted to a direct value (or factory) if neccessary and passed to `dataclasses.field()`.
    - 'default_factory' is not allowed. Use `default=DefaultFactory(...)` instead.
    - 'init' is not allowed. To create a non-init field, use `dataclasses.field(init=False)` instead.

    Parameters:
        validator: Validator to use for validating the field (saved as metadata)
        default: Default value to use when the field does not exist in the input data (preferably a `Default` object)
        metadata: Base dictionary for field metadata, gets merged with the metadata generated by this function
        **kwargs: Additional keyword arguments that are passed to `dataclasses.field()`
    """
    # If metadata is specified as argument, use it as the base for the field's metadata
    if metadata is None:
        metadata = {}

    # Check for incompatible keyword arguments
    if 'init' in kwargs:
        raise ValueError('Keyword argument "init" is not allowed in validator field.')
    if 'default_factory' in kwargs:
        raise ValueError('Keyword argument "default_factory" is not allowed in validator field (use default=DefaultFactory(...) instead).')

    # Add validator metadata
    metadata['validator'] = validator

    # Ensure default is a Default object (or any subclass of Default)
    if default is dataclasses.MISSING:
        default = NoDefault
    elif not isinstance(default, Default):
        default = Default(default)

    if default is not NoDefault:
        # Add default values to metadata
        metadata['validator_default'] = default

        # Set regular dataclass default (depending on the type of default)
        if type(default) is Default or default is DefaultUnset:
            kwargs['default'] = default.get_value()
        elif type(default) is DefaultFactory:
            kwargs['default_factory'] = default.factory
        else:
            # Fallback to a lambda
            kwargs['default_factory'] = lambda: default.get_value()

    # Compatibility for Python 3.9 and older: Use a workaround to allow required and optional fields to be defined in
    # any order. (In Python 3.10 the kw_only=True option for dataclasses is introduced, which can be used instead.)
    if default is NoDefault and sys.version_info < (3, 10):  # pragma: ignore-py-gte-310
        # Use a default_factory that raises an exception for required fields.
        kwargs['default_factory'] = lambda: _raise_field_required(_name)

    # Create a dataclass field with our metadata
    return dataclasses.field(metadata=metadata, **kwargs)


def _raise_field_required(name: str) -> NoReturn:  # pragma: ignore-py-gte-310
    """
    Raises a TypeError exception. Used for required fields (only in Python 3.9 or lower where the kw_only option is not
    supported yet).
    """
    raise TypeError(f"Missing required keyword-only argument: '{name}'" if name else 'Missing required keyword-only argument')
