# AUTOGENERATED! DO NOT EDIT! File to edit: ../../../nbs/02_data.external.iaea.ipynb.

# %% auto 0
__all__ = ['load_wetchem', 'get_replicates', 'bundle']

# %% ../../../nbs/02_data.external.iaea.ipynb 3
#import subprocess
import pandas as pd
from fastcore.basics import patch
from fastcore.xtras import mkdir
from pathlib import Path
import glob
import re
from tqdm import tqdm
from typing import Dict, Callable

from ...readers import read_spa
from ...sig import interp

# %% ../../../nbs/02_data.external.iaea.ipynb 4
def load_wetchem(fname_target:str, # Excel file containing wet chemistry measurements
                 col_names:list[str]=[
                     'id', 'year', 'crops', 'kd_cs133', 
                     'ex_cs_total_cs_ratio', 'cs137', 'cs137_total', 
                     'ex_cs137', 'ex_k2o', 'ph', 'c',
                     'n', 'c_n_ratio', 'ex_mgo', 'ex_cao', 
                     'cec', 'pretreatment'], # Wet chemistry excel file column names
                 kwargs:dict={'skiprows': 3}) -> pd.DataFrame:  # Loaded Wet Chemistry
    "Load wet chemistry measurements"
    df = pd.read_excel(fname_target, **kwargs)
    df.columns = col_names
    return df

# %% ../../../nbs/02_data.external.iaea.ipynb 6
def get_replicates(src_dir_spectra, pattern):
    fnames = glob.glob(src_dir_spectra+'/*.SPA')
    spectra_paths = [Path(f) for f in fnames if re.match(pattern, Path(f).name)]
    return spectra_paths

# %% ../../../nbs/02_data.external.iaea.ipynb 8
def bundle(df:pd.DataFrame, # Wet chemistry
           src_dir_spectra:str, # Directory containing infrared spectra
           fn_get_replicates:Callable, # Get list of path of replicates given a regexp pattern
           dest_dir:str, # Destination directory
           analytes:list[str]=['ph', 'cec'], # Analytes of interest
           wn_range:tuple=(4000, 650), # Max and min wavenumbers 
           compressed=True, # True if folder should be compressed
          ):
    """Bundle IAEA dataset as follows:

        - /dest_dir/
        -          /smp_id/
        -                  target.csv
        -                  spectrum_replicate_XX.csv
    """
    dest_dir = Path(dest_dir)        
    mkdir(dest_dir, overwrite=True)
    
    # iterate and create tree structure
    for index, row in tqdm(df.iterrows(), total=df.shape[0]):
        path = dest_dir / str(row['id'])
        if not path.exists():
            path.mkdir()
            target = row[['id'] + analytes]
            target.index.name = 'analyte'
            target.name = 'value'
            target.to_csv(path/'target.csv')
            
        # read, process and bundle spectra    
        fnames = fn_get_replicates(src_dir_spectra, pattern=str(row['id'])+'[a-z]')
        for fname in fnames:
            absorbance, wn = interp(read_spa(fname), between=wn_range)
            spectrum = pd.Series(data=absorbance, index=wn, name='absorbance')
            spectrum.index.name = 'wavenumber'
            out_fname = str(path/fname.stem) + '.csv'
            spectrum.to_csv(out_fname)
