"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const utils_1 = require("./private/utils");
class VirtualNodeBase extends cdk.Resource {
    grantStreamAggregatedResources(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['appmesh:StreamAggregatedResources'],
            resourceArns: [this.virtualNodeArn],
        });
    }
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 */
class VirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.backends = new Array();
        this.listeners = new Array();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualNodeProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.mesh = props.mesh;
        this.serviceDiscoveryConfig = (_b = props.serviceDiscovery) === null || _b === void 0 ? void 0 : _b.bind(this);
        (_c = props.backends) === null || _c === void 0 ? void 0 : _c.forEach(backend => this.addBackend(backend));
        (_d = props.listeners) === null || _d === void 0 ? void 0 : _d.forEach(listener => this.addListener(listener));
        const accessLogging = (_e = props.accessLog) === null || _e === void 0 ? void 0 : _e.bind(this);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            meshOwner: utils_1.renderMeshOwner(this.env.account, this.mesh.env.account),
            spec: {
                backends: cdk.Lazy.any({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.any({ produce: () => this.listeners.map(listener => listener.listener) }, { omitEmptyArray: true }),
                backendDefaults: props.backendDefaults !== undefined
                    ? {
                        clientPolicy: {
                            tls: utils_1.renderTlsClientPolicy(this, (_f = props.backendDefaults) === null || _f === void 0 ? void 0 : _f.tlsClientPolicy),
                        },
                    }
                    : undefined,
                serviceDiscovery: renderServiceDiscovery(this.serviceDiscoveryConfig),
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualNodeAccessLog,
                } : undefined,
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualNode given an ARN
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new class extends VirtualNodeBase {
            constructor() {
                super(...arguments);
                this.virtualNodeArn = virtualNodeArn;
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).splitArn(virtualNodeArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
                this.virtualNodeName = cdk.Fn.select(2, this.parsedArn);
            }
        }(scope, id);
    }
    /**
     * Import an existing VirtualNode given its name
     */
    static fromVirtualNodeAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualNodeAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromVirtualNodeAttributes);
            }
            throw error;
        }
        return new class extends VirtualNodeBase {
            constructor() {
                super(...arguments);
                this.mesh = attrs.mesh;
                this.virtualNodeName = attrs.virtualNodeName;
                this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualNode`,
                    resourceName: this.virtualNodeName,
                });
            }
        }(scope, id);
    }
    /**
     * Utility method to add an inbound listener for this VirtualNode
     *
     * Note: At this time, Virtual Nodes support at most one listener. Adding
     * more than one will result in a failure to deploy the CloudFormation stack.
     * However, the App Mesh team has plans to add support for multiple listeners
     * on Virtual Nodes and Virtual Routers.
     *
     * @see https://github.com/aws/aws-app-mesh-roadmap/issues/120
     */
    addListener(listener) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualNodeListener(listener);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addListener);
            }
            throw error;
        }
        if (!this.serviceDiscoveryConfig) {
            throw new Error('Service discovery information is required for a VirtualNode with a listener.');
        }
        this.listeners.push(listener.bind(this));
    }
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to
     */
    addBackend(backend) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_Backend(backend);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addBackend);
            }
            throw error;
        }
        this.backends.push(backend.bind(this).virtualServiceBackend);
    }
}
exports.VirtualNode = VirtualNode;
_a = JSII_RTTI_SYMBOL_1;
VirtualNode[_a] = { fqn: "@aws-cdk/aws-appmesh.VirtualNode", version: "1.153.1" };
function renderServiceDiscovery(config) {
    return config
        ? {
            dns: config === null || config === void 0 ? void 0 : config.dns,
            awsCloudMap: config === null || config === void 0 ? void 0 : config.cloudmap,
        }
        : undefined;
}
//# sourceMappingURL=data:application/json;base64,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