# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['dynamicprompts',
 'dynamicprompts.commands',
 'dynamicprompts.generators',
 'dynamicprompts.parser',
 'dynamicprompts.samplers',
 'dynamicprompts.wildcards',
 'dynamicprompts.wildcards.collection',
 'dynamicprompts.wildcards.tree']

package_data = \
{'': ['*']}

install_requires = \
['jinja2>=3.1.2,<4.0.0',
 'pyparsing>=3.0.9,<4.0.0',
 'requests>=2.28.1,<3.0.0',
 'tqdm>=4.64.1,<5.0.0']

extras_require = \
{':extra == "magicprompt"': ['transformers[torch]>=4.19.2'],
 'attentiongrabber': ['spacy>=3.4.4,<4.0.0']}

setup_kwargs = {
    'name': 'dynamicprompts',
    'version': '0.21.0',
    'description': 'Dynamic prompts templating library for Stable Diffusion',
    'long_description': '# Dynamic Prompts\n\n\n![MIT](https://img.shields.io/github/license/adieyal/dynamicprompts)\n&nbsp;-&nbsp;\n![GitHub Workflow Status](https://img.shields.io/github/actions/workflow/status/adieyal/dynamicprompts/test.yml)\n[![Codecov](https://img.shields.io/codecov/c/github/adieyal/dynamicprompts)](https://app.codecov.io/gh/adieyal/dynamicprompts)\n&nbsp;-&nbsp;\n[![PyPI](https://img.shields.io/pypi/v/dynamicprompts)](https://pypi.org/project/dynamicprompts) ![PyPI - Downloads](https://img.shields.io/pypi/dm/dynamicprompts)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/dynamicprompts)\n\nDynamicprompts is a Python library that provides developers with a flexible and intuitive templating language for generating prompts for text-to-image generators like Stable Diffusion, MidJourney or Dall-e 2. Dynamicprompts lets you create and manage sophisticated prompt generation workflows that seamlessly integrate with your existing text-to-image generation pipelines.\n\nIt includes:\n* An easy-to-learn templating language that lets you create a number of unique prompts from a single template.\n* Support for wildcard files as placeholders in templates.\n* A mechanism for creating a wildcard library. Text, JSON, and YAML files are supported.\n* Exhaustive generation of all prompts from a template.\n* Variable assignment enabling re-usable prompt snippets. (coming soon)\n* Supports Magic Prompt which automatically spices up your prompt with modifiers\n* Provides an I\'m Feeling Lucky feature which uses the semantic search on Lexica.art to find similar prompts.\n* For systems that support attention syntax, Attention Grabber will emphasis random phrases in your prompt.\n* Jinja-powered templating for advanced prompt creation.\n\nThe dynamicprompts library powers the [Dynamic Prompts](https://github.com/adieyal/sd-dynamic-prompts) extension for Automatic1111.\n\n## Table of contents\n\n* [Dynamic Prompts](#dynamic-prompts)\n   * [Quick overview of the templating language:](#quick-overview-of-the-templating-language)\n      * [Variants](#variants)\n      * [Choosing multiple variants](#choosing-multiple-variants)\n      * [Wildcards](#wildcards)\n      * [Let\'s try a real-world prompt](#lets-try-a-real-world-prompt)\n      * [Use whitespace for readability](#use-whitespace-for-readability)\n   * [Installation](#installation)\n   * [Quick Start](#quick-start)\n      * [Combinatorial Generation](#combinatorial-generation)\n      * [Magic Prompt](#magic-prompt)\n       * [I\'m feeling lucky](#im-feeling-lucky)\n       * [Attention Generator](#attention-generator)\n    * [Template syntax](#template-syntax)\n      * [Combinations](#combinations)\n      * [Wildcard files](#wildcard-files)\n      * [Nesting](#nesting)\n         * [Combinations](#combinations-1)\n         * [Wildcard files](#wildcard-files-1)\n      * [Comments](#comments)\n      * [Whitespace](#whitespace)\n      * [Samplers](#samplers)\n         * [Finite Samplers](#finite-samplers)\n         * [Non-finite samplers](#non-finite-samplers)\n         * [Comparison between samplers](#comparison-between-samplers)\n         * [Mixing samplers in the same prompt](#mixing-samplers-in-the-same-prompt)\n      * [Syntax customisation](#syntax-customisation)\n    * [Wildcard Collections](#wildcard-collections)\n    * [Dynamic Prompts in the wild.](#dynamic-prompts-in-the-wild)\n\n\n\n\n## Quick overview of the templating language:\n\n### Variants\n```\n{summer|autumn|winter|spring} is coming\n```\nRandomly generate one of:\n```\nsummer is coming\nautumn is coming\nwinter is coming\nspring is coming\n```\n\n### Choosing multiple variants\nThis syntax `{2$$ and $$A|B|C}` will choose two values from the list:\n```\nA and B\nA and C\nB and A\nB and C\nC and A\nC and B\n```\n\n### Wildcards\n```\n__season__ is coming\n```\nRandomly selects a value from season.txt in your wildcard directory.\n\n\n### Let\'s try a real-world prompt\nOne prompt template can generate a family of prompts:\n\n```\nFunky pop {yoda|darth vader|jabba the hutt|princess leia|chewbacca|luke skywalker} figurine, made of {wood|plastic|metal|stone}, product studio shot, on a white background, diffused lighting, centered\n```\n\n<img src="images/funkypop.jpg" style="width:50%">\n\n<br/>\n\n\nNow, how about two characters at the same time:\n\n```\nFunky pop {2$$ and $$yoda|darth vader|jabba the hutt|princess leia|chewbacca|luke skywalker} figurine, made of {wood|plastic|metal|stone}, product studio shot, on a white background, diffused lighting, centered\n```\n\n<img src="images/funkypop2.jpg" style="width:50%">\n\n\n<br/>\n\n### Use whitespace for readability\n```\n# Add comments like this\nFunky pop\n    {2$$ and $$\n  \t    yoda\n\t\t|darth vader\n\t\t|jabba the hutt\n\t\t|princess leia\n\t\t|chewbacca\n\t\t|luke skywalker\n\t}\n\tfigurine, made of\n\t{\n\t\twood\n\t\t|plastic\n\t\t|metal\n\t\t|stone\n\t}, product studio shot, on a white background, diffused lighting, centered\n```\n\n\nUse wildcards for re-usable lists:\n\n```\n# starwars.txt\nyoda\ndarth vader\njabba the hutt\nprincess leia\nchewbacca\nluke skywalker\n```\n\n```\n# material.txt\nwood\nplastic\nmetal\nstone\n```\n\n```\n# studio-shot.txt\nproduct studio shot, on a white background, diffused lighting, centered\n```\n\nNow compose your prompt like this:\n```\nFunky pop __starwars__ figurine, made of __material__, __studio-shot__\n```\n\nand easily change it to:\n```\nFunky pop __celebrities__ figurine, made of __material__, __studio-shot__\n```\n\n\nHat tip to [publicprompts](https://publicprompts.art/) for the funky pop prompt.\n\nMore complete documentation can be found [below](#syntax).\n\n\n## Installation\n\n`pip install dynamicprompts`\n\nAdditional functionality (see below) can be installed with this command:\n\n`pip install "dynamicprompts[magicprompt, attentiongrabber]"`\n\n## Quick Start\n\nUse the RandomPromptGenerator to create 5 random prompts using a given template:\n\n```python\nfrom dynamicprompts.generators import RandomPromptGenerator\n\ngenerator = RandomPromptGenerator()\ngenerator.generate("I love {red|green|blue} roses", num_images=5)\n\n>> [\'I love blue roses\', \'I love red roses\', \'I love green roses\', \'I love red roses\', \'I love red roses\']\n```\n\nIf you want to use wildcards, instantiate a WildcardManager:\n\n```python\nfrom pathlib import Path\nfrom dynamicprompts.generators import RandomPromptGenerator\nfrom dynamicprompts.wildcardmanager import WildcardManager\n\nwm = WildcardManager(Path("/path/to/wildcard/directory"))\n\ngenerator = RandomPromptGenerator(wildcard_manager=wm)\n```\n\nAssuming you have a file called colours.txt in /path/to/wildcards/directory with one colour per line, e.g.\n\n```\nred\ngreen\nblue\npurple\nyellow\n```\n\nthen\n\n```python\ngenerator.generate("I love __colours__ roses", num_prompts)\n>> [\'I love pink roses\', \'I love violet roses\', \'I love white roses\', \'I love violet roses\', \'I love blue roses\']\n```\n\n\n### Combinatorial Generation\nInstead of generating random prompts from a template, combinatorial generation produces every possible prompt from the given string. For example:\n\n`I {love|hate} {New York|Chicago} in {June|July|August}`\n\nwill produce:\n> I love New York in June<br>\n> I love New York in July<br>\n> I love New York in August<br>\n> I love Chicago in June<br>\n> I love Chicago in July<br>\n> I love Chicago in August<br>\n> I hate New York in June<br>\n> I hate New York in July<br>\n> I hate New York in August<br>\n> I hate Chicago in June<br>\n> I hate Chicago in July<br>\n> I hate Chicago in August<br>\n\nIf a `__wildcard__` is provided, then a new prompt will be produced for every value in the wildcard file. For example:\n\n`My favourite season is __seasons__`\n\nwill produce:\n> My favourite season is Summer<br>\n> My favourite season is August<br>\n> My favourite season is Winter<br>\n> My favourite season is Sprint<br>\n\n#### Usage\n\n```python\nfrom dynamicprompts.generators import CombinatorialPromptGenerator\n\ngenerator = CombinatorialPromptGenerator()\ngenerator.generate("I love {red|green|blue} roses", max_prompts=5)\n\n>> [\'I love red roses\', \'I love green roses\', \'I love blue roses\']\n```\n\nNotice that only 3 prompts were generated, even though we requested 5. Since there are only three options, i.e. red, green, and blue, only 3 unique prompts can be created. `num_prompts` in this case acts as an upper bound. Combinatorial generation can very quickly produce many more prompts than you intended. `num_prompts` is a safeguard to ensure that you don\'t accidentally produced thousands or tens of thousands of prompts.\n\nConsider this template:\n\n`My favourite colours are __colours__, __colours__, and __colours__`\n\nIf colours.txt contains 10 different colours, a combinatorial enumeration of that template will create `10 * 10 * 10 = 1000` different prompts. e.g.\n\n> My favourite colours are red, green, and blue<br>\n> My favourite colours are red, green, and yellow<br>\n> My favourite colours are red, green, and purple<br>\n> My favourite colours are red, blue, and yellow<br>\n> My favourite colours are red, blue, and purple<br>\n> ...<br>\n\n\n### Magic Prompt\nUsing [Gustavosta](https://huggingface.co/Gustavosta/MagicPrompt-Stable-Diffusion)\'s MagicPrompt model, automatically generate new prompts from the input. Trained on 80,000 prompts from [Lexica.art](lexica.art), it can help give you interesting new prompts on a given subject. Here are some automatically generated variations for "dogs playing football":\n\n> dogs playing football, in the streets of a japanese town at night, with people watching in wonder, in the style of studio ghibli and makoto shinkai, highly detailed digital art, trending on artstation<br>\n> dogs playing football, in the background is a nuclear explosion. photorealism. hq. hyper. realistic. 4 k. award winning.<br>\n> dogs playing football, in the background is a nuclear explosion. photorealistic. realism. 4 k wideshot. cinematic. unreal engine. artgerm. marc simonetti. jc leyendecker<br>\n\nThis is compatible with the wildcard syntax described above.\n\n#### Usage\n\n```python\nfrom dynamicprompts.generators import RandomPromptGenerator\nfrom dynamicprompts.generators.magicprompt import MagicPromptGenerator\n\ngenerator = RandomPromptGenerator()\nmagic_generator = MagicPromptGenerator(generator, device=0) # device = 0 for CUDA or -1 for CPU\n\nnum_prompts = 5\ngenerator.generate("I love {red|green|blue} roses", num_prompts)\n\n>> [\'I love red roses trending on artstation #vividart #pixiv\', \'I love red roses trending on artstation artwork\', \'I love blue roses breakfast club, cute, intricate, highly detailed, digital painting, artstation, concept art, smooth, sharp focus, illustration, unreal engine 5, 8 k, art by artgerm and greg rutkowski and alphonse mucha\', \'I love green roses I love green flowers, smile, Tristan Eaton, victo ngai, artgerm, RHADS, ross draws\', \'I love red roses smile, Tristan Eaton, victo ngai, artgerm, RHADS, ross draws\']\n\n```\n\nThe first time you use it, the model is downloaded. It is approximately 500mb and so will take some time depending on how fast your connection is. It will also take a few seconds on first activation as the model is loaded into memory. Note, if you\'re low in VRAM, you might get a Cuda error. My GPU uses less than 8GB by YMMV.\n\nMagic Prompt is not available by default, you need to install it as follows:\n\n`pip install "dynamicprompts[magicprompt]"`\n\n#### Other models\nThere a few alternatives to Gustavosta\'s model available. You can try:\n\n```\nmagic_generator = MagicPromptGenerator(generator, "AUTOMATIC/promptgen-lexart")\nmagic_generator = MagicPromptGenerator(generator, "AUTOMATIC/promptgen-majinai-safe")\nmagic_generator = MagicPromptGenerator(generator, "AUTOMATIC/promptgen-majinai-unsafe")\n```\n\nYou can find a longer list [here](https://github.com/adieyal/sd-dynamic-prompts/blob/main/config/magicprompt_models.txt)\nNote that each model requires a download of large model files.\n\n### I\'m feeling lucky\nUse the [lexica.art](https://lexica.art) API to create random prompts. Useful if you\'re looking for inspiration, or are simply too lazy to think of your own prompts. When this option is selected, the template is used as a search string. For example, prompt "Mech warrior" might return:\n\n> A large robot stone statue in the middle of a forest by Greg Rutkowski, Sung Choi, Mitchell Mohrhauser, Maciej Kuciara, Johnson Ting, Maxim Verehin, Peter Konig, final fantasy , 8k photorealistic, cinematic lighting, HD, high details, atmospheric,\n\n> a beautiful portrait painting of a ( ( ( cyberpunk ) ) ) armor by simon stalenhag and pascal blanche and alphonse mucha and nekro. in style of digital art. colorful comic, film noirs, symmetry, brush stroke, vibrating colors, hyper detailed. octane render. trending on artstation\n\n> symmetry!! portrait of a robot astronaut, floral! horizon zero dawn machine, intricate, elegant, highly detailed, digital painting, artstation, concept art, smooth, sharp focus, illustration, art by artgerm and greg rutkowski and alphonse mucha, 8 k\n\n<img src="images/feeling-lucky.png">\n\n#### Usage\n\n```python\nfrom dynamicprompts.generators import RandomPromptGenerator\nfrom dynamicprompts.generators.feelinglucky import FeelingLuckyGenerator\n\ngenerator = RandomPromptGenerator()\nlucky_generator = FeelingLuckyGenerator(generator)\n\nnum_prompts = 5\nlucky_generator.generate("I love {red|green|blue} roses", num_prompts)\n\n>> [\'“ guns and roses ” \', \'🌹🥀🏜. 🌌🌠⭐. 💯. \', \'tattoo design, stencil, beautiful japanese girls face, roses and ivy surrounding by artgerm, artgerm, cat girl, anime \', \'rose made of glass dramatic lighting\', \'a wireframe render of a red rose\']\n\n```\n\n### Attention Generator\nIf you are using [Automatic1111](https://github.com/AUTOMATIC1111/stable-diffusion-webui/) or a similar frontend to Stable Diffusion that uses attention syntax, e.g. `(some text:1.4)`, AttentionGenerator will randomly add attention to various phrases in your prompt. This injects a small amount of randomness into your prompt.\n\n#### Usage\n\n```python\nfrom dynamicprompts.generators import RandomPromptGenerator\nfrom dynamicprompts.generators.attentiongenerator import AttentionGenerator\n\ngenerator = RandomPromptGenerator()\nattention_generator = AttentionGenerator(generator)\n\nprompt = "a portrait an anthropomorphic panda mage casting a spell, wearing mage robes, landscape in background, cute, dnd character art portrait, by jason felix and peter mohrbacher, cinematic lighting"\nattention_generator.generate(prompt, num_prompts=1)\n\n>> [\'a portrait an anthropomorphic panda mage casting a spell, wearing (mage robes:0.77), landscape in background, cute, dnd character art portrait, by jason felix and peter mohrbacher, cinematic lighting\']\n\n```\n\n<img src="images/emphasis.png">\n\nNote, AttentionGenerator is not installed by default as it needs additional libraries to run. Use this command to install it:\n\n`pip install "dynamicprompts[attentiongrabber]"`\n\nOne your first use it, a model will automatically be downloaded.\n\n## Template syntax\n\nThe templating language understands 3 different core constructs:\n* Literals - this is plain text, e.g. `A red rose`\n* [wildcards](https://github.com/adieyal/sd-dynamic-prompts/blob/main/docs/SYNTAX.md#wildcard-files) - a wildcard represent a variable that is populated from a file, e.g. `A __colours__ rose` - a prompt will be generated by randomly choosing a value from a file called colours.txt in the [wildcards directory](https://github.com/adieyal/sd-dynamic-prompts#wildcard_dir)\n* [variants or combinations](https://github.com/adieyal/sd-dynamic-prompts/blob/main/docs/SYNTAX.md#combinations) - `A {red|pink|white} rose` - Dynamic Prompts will chose one of the colours at random when generating the prompt. They are called combinations because syntax is available to choose more than one value at once, `A {2$$red|pink|white} rose` will generate one of: A red,pink rose, A red,white rose, A pink,red rose, .... etc.\n\n### Combinations\n\t{2$$opt1|opt2|opt3}\n\nThis will randomly combine two of the options for every batch, separated with a comma.  In this case, "opt1, opt2" or "opt2, opt3", or "opt1, opt3" or the same pairs in the reverse order.\n\n\t{1-3$$opt1|opt2|opt3}\n\nThis will use a random number of options between 1 and 3 for each batch.\n\nIf the number of combinations chosen is greater than the number of options listed then options may be repeated in the output.\nIf the number of combinations chosen is less than or equal to the number of options listed then the same options will not be chosen more than once.\n\n\t{4$$and$$opt1|opt2|opt3|opt4|opt5}\n\nThis will choose 4 options and join them together with \'and\' instead of the default comma. When there are multiple $$ tokens then the first item is the number of options to choose and the second option is the joiner to use.\n\n\t{-$$opt1|opt2|opt3}\n\nAn omitted minimum is assumed to be 0 and an omitted maximum is assumed to be the number of options.\n\n\t{opt1|opt2|opt3}\n\nIf you omit the `$$` prefix, one item will be selected. (Equivalent to `1$$`)\n\nOptions are chosen randomly with replacement. This means that `{2$$opt1|opt2}` can return any of the following:\n\n> opt1, opt1<br>\n> opt1, opt2<br>\n> opt2, opt1<br>\n> opt2, opt2<br>\n\nThis is useful in conjunction with wildcards (see below).\n\n\nOptions can be assigned relative weights using a `::` prefix operator.\n\n\tphoto of a {3::blue|red} ball\n\nThis will generate 3 photos of a blue ball per every 1 photo of a red ball.\n\n<img src="images/weighting-colours.png">\n\n\tphoto of a {blue|0.25::red} ball\n\nDecimals also work as expected: this will generate 4 photos of a blue ball per every 1 photo of a red ball.\n\n\tphoto portrait of a {59::white|21::latino|14::black|8::asian} {man|woman}\n\nThis would generate photo portraits of men and women of different races, proportional to the 2020 U.S. census.\n<img src="images/weighting-us-population.png">\n\nIf you omit the `::` prefix, it will have a default weight of 1.0. (Equivalent to `1::prompt`)\n\n### Wildcard files\nWildcards are text files (ending in .txt). Each line contains a term, artist name, or modifier. The wildcard file can then be embedded in your prompt by removing the .txt extension and surrounding it with double underscores. e.g:\n\n\tMy favourite colour is __colours__\n\nEmpty lines and lines starting with `#` are ignored. This can be used to add comments or disable sections of the file.\n\nMixing Combinations and Wildcards can be useful. For example,\n\n\ta photo of a {2-4$$and$$__adjective__} house\n\nwill choose between 2 and 4 options from adjective.txt, join them together with "and", for results such as "a photo of a cozy and ancient and delicate house"\n\n\n### Nesting\n\nMany constructed can nest sub-prompts. This means that you can create more advanced templates. Here are some examples:\n\n#### Combinations\nYou can nest inside combinations.\n\n    {__seasons__|__timeofday__}\n\n\nThis will then either choose a season from seasons.txt or a time of day from timeofday.txt.\n\nCombinations can also be nested inside other combinations, e.g.\n\n    {{a|b|c}|d}\n\nYou can even nest complete prompts in combinations:\n\n\t{A {small|large} __monster__|A {scary|friendly} ghost}\n\nThis produce one of:\n\n> A small dragon (assuming dragon is contained in the monster.txt file)<br>\n> A large dragon<br>\n> A scary ghost<br>\n> A friendly ghost<br>\n\nIf you find that your prompts are becoming too complicated to read, consider using [whitespace](#whitespace)\n\n#### Wildcard files\nWildcard files are processed recursively. If a wildcard file contains a row with dynamic syntax, then that will be resolved as well. For example if seasons.txt contains the following rows:\n\n\tSummer\n\tWinter\n\t{Autumn|Fall}\n\tSpring\n\nif the 3rd row is chosen, then either Autumn or Fall will be selected. You could go pretty wild e.g.\n\n\tSummer\n\t__winter_in_different_languages__\n\t{Autumn|Fall}\n\tSpring\n\n### Comments\nPython-style comments are supported:\n\n    Test string\n    # This  a comment until the end of the line\n\n### Whitespace\nIn most cases, whitespace is ignored which allows you to create more expressive and readable prompts, e.g.\n\n\twisdom {\n    \twoman, __colours__ eyes, braided hair\n    \t|man using a __war/weapons/swords/european__, red turban\n    \t|dwarf wielding a warhammer, __colours__ beard\n\t},\n\tknows the meaning of life, warrior, hyper-realistic, peaceful, dark fantasy, unreal engine, 8k\n\n### Samplers\n[Note, this is an advanced feature and you probably don\'t need to worry about it.]\n\nBehind the scenes, Dynamic Prompts uses samplers to select an option from a wildcard or variant. Samplers can be classed as either finite or non-finite.\n\n#### Finite Samplers\nWhen sampling using a finite sampler, once the options are exhausted, the sampler no-longer returns any values.\n\nThe only finite sampler currently available is the  **Combinatorial Sampler**.  It will exhaustively generate all possible combinations and then stop, e.g. `{A|B|C}` will produce:\n\n```\nA\nB\nC\n```\n\nCombinatorialPromptGenerators uses a combinatorial sampler by default.\n\n#### Non-finite samplers\nNon-finite samplers can be used to generate an infinite number of prompts. They are useful for generating prompts that are not limited by the number of options available in a wildcard file or combination.\n\n**Random Sampler** - this sampler randomly picks an option. In this case `A`, `B`, and `C` are all equally likely to be chosen, e.g.\n```\nA\nC\nB\nB\nA\nC\n...\n```\n\nUnsurprisingly, RandomPromptGenerators uses a random sampler by default.\n\n**Cyclical Sampler** - a cyclical sampler will choose an item in a round-robin fashion. Every time you sample using this sampler, it will return the next option in the list. Once you have exhausted the list it starts again. e.g.:\n```\nA\nB\nC\nA\nB\n...\n```\n\nBoth Random and Cyclical samplers treat a wildcard or variant in isolation, whereas the Combinatorial sampler combines all the wildcards and variants in the prompt and treats them as a single unit.  The examples below should make this clearer.\n\n#### Comparison between samplers\nConsider the prompt `{A|B|C} {X|Y}`. If we use a random sampler, we might get the following prompts:\n\n```\nA X\nC Y\nB X\nA Y\nC X\nC X\n```\n\nIf we use a cyclical sampler, we will generate the following prompts:\n```\nA X\nB Y\nC X\nA Y\nB X\nC Y\nA X\n...\n\n```\n\nFinally, if we use a combinatorial sampler, we will generate exactly 6 prompts:\n```\nA X\nA Y\nB X\nB Y\nC X\nC Y\n```\n\n#### Mixing samplers in the same prompt\nWhen parsing a prompt template, every variant and wildcard is assigned a sampler. If a sampler is not explicitly set, then the default sampler is used. You can explictly set the sampler, using the syntax `{!A|B|C}` or `__!wildcard__` for combinatorial, `{~A|B|C}` or `__~wildcard__` for random and `{@A|B|C}` or `__@wildcard__` for cyclical.\n\nExamples:\nIn combinatorial mode, the template `{A|B|C} {@X|Y}` will automatically be converted to `{!A|B|C} {@X|Y}`. This will generate the following prompts:\n```\nA X\nB Y\nC X\n```\n\nThis template only produces 3 prompts because the combinatorial sampler is exhausted  after producing `A`, `B`, and `C`.\n\nSimilarly, `{!A|B|C} {~X|Y}` might generate the following prompts:\n```\nA Y\nB X\nC X\n```\n\nCompare this with `{!A|B|C} {!X|Y}` which will generate the following prompts:\n```\nA X\nA Y\nB X\nB Y\nC X\nC Y\n```\n\nThe default mode is propagated to all nested variants and wildcards, e.g. In random mode, this prompt:\n```\n{A|B|C}\n{@X|Y|\n    {1|2|3}\n}\n```\n\nwill be converted to:\n```\n{~A|B|C}\n{@X|Y|\n\t{@1|2|3}\n}\n```\n\nOne final note, finite variants and wildcards cannot be sampled inside non-finite variants and wildcards. For example, in this prompt:\n```\n{~A|B|{!X|Y}}\n```\n\nthe combinatorial syntax is ignored and the template will be converted to:\n```\n{~A|B|{~X|Y}}\n```\n\n\n### Syntax customisation\nTo address potential syntax clashes with other tools it is possible to change various tokens. Instead of `{red|green|blue}` you can configure the library to use the `<` `>` pair instead, e.g. `<red|green|blue>`. You can also change the `__` used in wildcards. So instead of `__colours__`, you can configure wildcards to use `**`, e.g. `**colours**`\n```python\n\nfrom pathlib import Path\nfrom dynamicprompts.wildcards import WildcardManager\nfrom dynamicprompts.generators import RandomPromptGenerator\nfrom dynamicprompts.parser.config import ParserConfig\n\nWILDCARD_DIR = Path("/path/to/wildcards/directory")\nwm = WildcardManager(WILDCARD_DIR)\nparser_config = ParserConfig(variant_start="<", variant_end=">", wildcard_wrap="**")\n\ngenerator = RandomPromptGenerator(wm, parser_config=parser_config)\n\n```\n## Wildcard Collections\nYou can bootstrap your wildcard library by using our [pre-existing collections](https://github.com/adieyal/sd-dynamic-prompts/tree/main/collections). You\'ll find just under 80,000 wildcards divided into 1900 files. Feel free to pick and choose or take them in their entirety.\n\n## Dynamic Prompts in the wild.\nDynamic Prompts has been used in:\n1. [SD Dynamic Prompts](https://github.com/adieyal/sd-dynamic-prompts/edit/main/README.md) Auto1111 extension\n2. Deforum 0.7 [colab](https://colab.research.google.com/drive/1qtYHUwFl9ocLyzDRL1_MlpQluV32ndoT?usp=sharing)\n',
    'author': 'Adi Eyal',
    'author_email': 'adi@clearforest.io',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/adieyal/dynamicprompts',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
