"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LaunchTemplateRequireImdsv2Aspect = exports.InstanceRequireImdsv2Aspect = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const ec2_generated_1 = require("../ec2.generated");
const instance_1 = require("../instance");
const launch_template_1 = require("../launch-template");
/**
 * Base class for Aspect that makes IMDSv2 required.
 */
class RequireImdsv2Aspect {
    constructor(props) {
        var _c;
        this.suppressWarnings = (_c = props === null || props === void 0 ? void 0 : props.suppressWarnings) !== null && _c !== void 0 ? _c : false;
    }
    /**
     * Adds a warning annotation to a node, unless `suppressWarnings` is true.
     *
     * @param node The scope to add the warning to.
     * @param message The warning message.
     */
    warn(node, message) {
        if (this.suppressWarnings !== true) {
            cdk.Annotations.of(node).addWarning(`${RequireImdsv2Aspect.name} failed on node ${node.node.id}: ${message}`);
        }
    }
}
/**
 * Aspect that applies IMDS configuration on EC2 Instance constructs.
 *
 * This aspect configures IMDS on an EC2 instance by creating a Launch Template with the
 * IMDS configuration and associating that Launch Template with the instance. If an Instance
 * is already associated with a Launch Template, a warning will (optionally) be added to the
 * construct node and it will be skipped.
 *
 * To cover Instances already associated with Launch Templates, use `LaunchTemplateImdsAspect`.
 */
class InstanceRequireImdsv2Aspect extends RequireImdsv2Aspect {
    constructor(props) {
        var _c;
        super(props);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_InstanceRequireImdsv2AspectProps(props);
        this.suppressLaunchTemplateWarning = (_c = props === null || props === void 0 ? void 0 : props.suppressLaunchTemplateWarning) !== null && _c !== void 0 ? _c : false;
    }
    visit(node) {
        if (!(node instanceof instance_1.Instance)) {
            return;
        }
        if (node.instance.launchTemplate !== undefined) {
            this.warn(node, 'Cannot toggle IMDSv1 because this Instance is associated with an existing Launch Template.');
            return;
        }
        const launchTemplate = new ec2_generated_1.CfnLaunchTemplate(node, 'LaunchTemplate', {
            launchTemplateData: {
                metadataOptions: {
                    httpTokens: 'required',
                },
            },
        });
        if (core_1.FeatureFlags.of(node).isEnabled(cxapi.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME)) {
            launchTemplate.launchTemplateName = cdk.Names.uniqueId(launchTemplate);
        }
        else {
            launchTemplate.launchTemplateName = `${node.node.id}LaunchTemplate`;
        }
        node.instance.launchTemplate = {
            launchTemplateName: launchTemplate.launchTemplateName,
            version: launchTemplate.getAtt('LatestVersionNumber').toString(),
        };
    }
    warn(node, message) {
        if (this.suppressLaunchTemplateWarning !== true) {
            super.warn(node, message);
        }
    }
}
exports.InstanceRequireImdsv2Aspect = InstanceRequireImdsv2Aspect;
_a = JSII_RTTI_SYMBOL_1;
InstanceRequireImdsv2Aspect[_a] = { fqn: "@aws-cdk/aws-ec2.InstanceRequireImdsv2Aspect", version: "1.152.0" };
/**
 * Aspect that applies IMDS configuration on EC2 Launch Template constructs.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-metadataoptions.html
 */
class LaunchTemplateRequireImdsv2Aspect extends RequireImdsv2Aspect {
    constructor(props) {
        super(props);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_LaunchTemplateRequireImdsv2AspectProps(props);
    }
    visit(node) {
        if (!(node instanceof launch_template_1.LaunchTemplate)) {
            return;
        }
        const launchTemplate = node.node.tryFindChild('Resource');
        const data = launchTemplate.launchTemplateData;
        if (cdk.isResolvableObject(data)) {
            this.warn(node, 'LaunchTemplateData is a CDK token.');
            return;
        }
        const metadataOptions = data.metadataOptions;
        if (cdk.isResolvableObject(metadataOptions)) {
            this.warn(node, 'LaunchTemplateData.MetadataOptions is a CDK token.');
            return;
        }
        const newData = {
            ...data,
            metadataOptions: {
                ...metadataOptions,
                httpTokens: 'required',
            },
        };
        launchTemplate.launchTemplateData = newData;
    }
}
exports.LaunchTemplateRequireImdsv2Aspect = LaunchTemplateRequireImdsv2Aspect;
_b = JSII_RTTI_SYMBOL_1;
LaunchTemplateRequireImdsv2Aspect[_b] = { fqn: "@aws-cdk/aws-ec2.LaunchTemplateRequireImdsv2Aspect", version: "1.152.0" };
//# sourceMappingURL=data:application/json;base64,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