/** include config file generated by configure
 *  (i.e., know what grids are present, etc)
 *  this should always be included first */
#include <config.h>
/** standard headers **/
#include <iostream>

#include <dune/common/version.hh>
/** dune (mpi, field-vector and grid type for dgf) **/
#include <dune/common/fvector.hh>
#include <dune/common/timer.hh>

#include <dune/alugrid/common/meshquality.hh>

#include "io.hh"

#define BALL 1

/** numerical scheme **/
#include "piecewisefunction.hh"
#include "fvscheme.hh"

/** adaptation scheme **/
#include "adaptation.hh"

/** pde and problem description **/
#include "problem-globalrefine.hh"

#include "diagnostics.hh"
#include "paralleldgf.hh"

// method
// ------
void method ( const std::string& dgffile, int startLvl, int maxLvl,
              const std::string outpath, const int mpiSize )
{
  typedef Dune::GridSelector::GridType Grid;

  const int startLevel = startLvl * Dune :: DGFGridInfo< Grid > :: refineStepsForHalf();
  const int maxLevel   = maxLvl   * Dune :: DGFGridInfo< Grid > :: refineStepsForHalf();

  /** type of pde to solve **/
  typedef GlobalRefineModel< Grid::dimensionworld > ModelType;
  ModelType model( 0 );

  /* Grid construction ... */
  // create grid pointer and release to free memory of GridPtr
  Grid* gridPtr = Dune::CreateParallelGrid< Grid >::create( dgffile ).release();

  Grid &grid = *gridPtr;

  //if( grid.comm().size() == 1 )
  //{
  // ALUGridSFC::printSpaceFillingCurve( grid.leafGridView() );
  //}

  grid.loadBalance();
  //grid.finalizeGridCreation() ;
  const bool verboseRank = grid.comm().rank() == 0 ;

  // avoid output here
  std::string outPath( "./" );

  // create the diagnostics object
  Dune::Diagnostics< Grid> diagnostics( grid.comm(), 1);

  std::ofstream meshqlty (  outpath  + "meshquality.gnu" );
  // investigate mesh quality
  meshQuality( grid.leafGridView(), 0.0, meshqlty );

  /* ... some global refinement steps */
  if( verboseRank )
    std::cout << "globalRefine: " << startLevel << std::endl;
  grid.globalRefine( startLevel );

  /* get view to leaf grid */
  typedef Grid::Partition< Dune::Interior_Partition >::LeafGridView GridView;
  GridView gridView = grid.leafGridView< Dune::Interior_Partition >();

  /* construct data vector for solution */
  typedef PiecewiseFunction< GridView, Dune::FieldVector< double, ModelType::dimRange > > DataType;
  DataType solution( gridView );
  /* initialize data */
  solution.initialize( model.problem() );

  /* create finite volume scheme */
  typedef FiniteVolumeScheme< DataType, ModelType > FVScheme;
  FVScheme scheme( gridView, model );

  /* create VTK writer for data sequqnce */
  Dune::VTKSequenceWriter< GridView >* vtkOut = 0 ;
  if( outPath != "none" )
  {
    vtkOut = new Dune::VTKSequenceWriter< GridView >(  gridView, "solution", outPath, ".", Dune::VTK::nonconforming );
    VTKData< DataType >::addPartitioningData( grid.comm().rank(), *vtkOut );
  }

  /* create adaptation method */
  const int initialBalanceCounter = std::max( int(model.problem().balanceStep() - maxLevel), int(1) );
  typedef LeafAdaptation< Grid, DataType > AdaptationType;
  AdaptationType adaptation( grid, model.problem().balanceStep(), initialBalanceCounter );

  if( vtkOut )
  {
    /* output the initial grid and the solution */
    vtkOut->write( 0.0 );
  }

  /* print info about initialization */
  if ( verboseRank )
    std::cout << "Intialization done!" << std::endl;

  for( int i = 0; i < maxLevel; ++i )
  {
    // mark grid for initial refinement
    GridMarker< Grid > gridMarker( grid, startLevel, maxLevel );
    scheme.mark( 0, solution, gridMarker );
    // adapt grid
    if( gridMarker.marked() )
      adaptation( solution );
    // initialize solution for new grid
    solution.initialize( model.problem() );
  }

  meshQuality( grid.leafGridView(), 1.0, meshqlty );

  if( vtkOut )
  {
    /* output the initial grid and the solution */
    vtkOut->write( 1.0 );
  }

  /* print info about initialization */
  if ( verboseRank )
    std::cout << "Refinement done!" << std::endl;

  // flush diagnostics
  diagnostics.flush();

  delete vtkOut ;
  // delete grid
  delete gridPtr ;
}
/***************************************************
 ** main program with parameters:                 **
 ** 1) number of problem to use (initial data...) **
 ** 2) number of global refinement steps          **
 ** 3) maximal level to use during refinement     **
 ***************************************************/
int main ( int argc , char **argv )
try
{
  /* initialize MPI, finalize is done automatically on exit */
  Dune::MPIHelper &mpi = Dune::MPIHelper::instance( argc, argv );

  int variant = 0;
  int threshold = 2;
  int useAnnouncedEdge = 1 ;
  int maxLevel = 1;

  if( argc < 3 )
  {
    /* display usage */
    if( mpi.rank() == 0 )
    {
      std::cout << "Usage: " << argv[ 0 ] << " [dgffile] [variant] [threshold] [useAnnouncedEdge]" << std::endl;
      std::cout << "variant = [0,1,2,3]" << std::endl;
      std::cout << "threshold = 0,...,25" << std::endl;
      std::cout << "useAnncounedEdge = 0, 1" << std::endl;
      std::cout << "Using default values variant = " << variant << ", threshold = " << threshold << ", useAnnouncedEdge = " << useAnnouncedEdge  << std::endl;
    }
    if( argc < 2 )
      return 0;
  }
  else
  {
    variant   = argc >= 3 ? atoi( argv[2] ) : variant;
    threshold = argc >= 4 ? atoi( argv[3] ) : threshold;
    useAnnouncedEdge = argc >= 5 ? atoi( argv[4] ) : useAnnouncedEdge;
    maxLevel = (argc >= 6 ) ? atoi( argv[ 5 ] ) : maxLevel;
  }

  std::string dgffile( argv[1] );

  BisectionCompatibilityParameters::variant() = variant;
  BisectionCompatibilityParameters::threshold() = threshold;
  BisectionCompatibilityParameters::useAnnouncedEdge() = useAnnouncedEdge;

  // meassure program time
  Dune::Timer timer ;

  /* get level to use for computationa */
  const int startLevel = 0;

  std::size_t pos1 = 0;//dgffile.find("/dgf/") + 5;
  std::size_t pos2 = dgffile.find(".dgf");
  std::string path =  "./meshquality/" + dgffile.substr(pos1, pos2-pos1) + "/";
  Dune::Fem::createDirectory( path );
  path += std::to_string( variant) + "-" + std::to_string(threshold) + "-" + std::to_string(useAnnouncedEdge) + "-" ;
  method( dgffile, startLevel, maxLevel, path, mpi.size() );

#ifdef HAVE_MPI
  MPI_Barrier ( MPI_COMM_WORLD );
#endif

  if( mpi.rank() == 0 )
    std::cout << "Program finished: CPU time = " << timer.elapsed() << " sec." << std::endl;

  /* done */
  return 0;
}
catch( const std::exception &e )
{
  std::cout << "STL ERROR: " << e.what() << std::endl;
  return 1;
}
catch( const ALUGrid::ALUGridException& e )
{
  std::cout << "ALUGrid ERROR: " << e.what() << std::endl;
  return 1;
}
catch( ... )
{
  std::cout << "Unknown ERROR" << std::endl;
  return 1;
}
