function Cy=GetCy(alpha,beta,fi,dnos,Wz,V,ba,sb)
% --------------------------------------------------------------
%       Вычисление коэффициента нормальной силы Cy.
% --------------------------------------------------------------
% Cy = GetCy(alpha,beta,fi,dnos,Wz,V)  вычисляет коэффициент нормальной силы
%    в зависимости от углов атаки, скольжения, отклонения стабилизатора, а 
%    также от угловой скорости тангажа.
% alpha - угол атаки, рад.
% beta - угол скольжения, рад.
% fi - отклонение стабилизатора, рад.
% dnos - отклонение носков крыла, рад.
% V - воздушная скорость, м/с
% Wz - угловая скорость тангажа, рад/с
%
% Зависимости заданы в виде таблиц.
% Таблицы интерполяции хранятся в оперативной памяти для ускорения
% численного расчёта.
% --------------------------------------------------------------
persistent init_mdl
persistent alpha1 alpha2 beta1 fi1 Cy1 Cy_nos1 Cywz1 dCywz_nos1 dCy_sb1
persistent interpCy interpCy_nos interpCywz1 interpCywz2 interpdCy_sb

% Инициализация
if isempty(init_mdl)

init_mdl = 1;
    
alpha1 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45 50 55 60 70 80 90]);
alpha2 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45]);
beta1 = deg2rad([-30 -25 -20 -15 -10 -8 -6 -4 -2 0 2 4 6 8 10 15 20 25 30]);
fi1 = deg2rad([-25 -10 0 10 25]);

% Зависимость коэффициента нормальной силы от угла атаки, угла скольжения
% и отклонения стабилизатора.
% Cy = f(alpha, beta, fi)

% fi = -25 grad
Cy1(:,:,1) = [1.1940    1.2720    1.3110    1.3560    1.3960    1.3470    1.3390    1.3140    1.3210    1.3150    1.3370    1.3320    1.3400    1.3380    1.2940  1.2350    1.1850    1.1400    1.1000;
    0.9960    1.0570    1.0900    1.1210    1.1280    1.1290    1.1310    1.1430    1.1580    1.1710    1.1770    1.1420    1.1480    1.1310    1.1370  1.1300    1.1000    1.0600    1.0050;
    0.7930    0.8320    0.8410    0.8560    0.8870    0.8888    0.8990    0.9090    0.9150    0.9250    0.9100    0.8920    0.8890    0.8810    0.8750  0.8350    0.8210    0.8150    0.7800;
    0.4100    0.4100    0.4200    0.4250    0.4510    0.4640    0.4740    0.4720    0.4740    0.4690    0.4600    0.4540    0.4470    0.4460    0.4400  0.4240    0.4050    0.3940    0.4030;
    0.1800    0.1550    0.1350    0.1300    0.1410    0.1490    0.1540    0.1530    0.1510    0.1550    0.1540    0.1510    0.1470    0.1380    0.1290  0.1190    0.1370    0.1230    0.1590;
   -0.0900   -0.1300   -0.1600   -0.1800   -0.1840   -0.1860   -0.1820   -0.1870   -0.1870   -0.1890   -0.1930   -0.1910   -0.1930   -0.1950   -0.1940  -0.1870   -0.1710   -0.1330   -0.0990;
   -0.3400   -0.4050   -0.4600   -0.4980   -0.5110   -0.5180   -0.5260   -0.5350   -0.5340   -0.5300   -0.5320   -0.5250   -0.5200   -0.5210   -0.5150  -0.4980   -0.4650   -0.4020   -0.3410;
   -0.6100   -0.6650   -0.7200   -0.7700   -0.8060   -0.8180   -0.8370   -0.8490   -0.8510   -0.8560   -0.8540   -0.8550   -0.8550   -0.8360   -0.8270  -0.8010   -0.7380   -0.6640   -0.6020;
   -0.8700   -0.9500   -1.0150   -1.0800   -1.1220   -1.1370   -1.1490   -1.1540   -1.1560   -1.1690   -1.1510   -1.1480   -1.1460   -1.1350   -1.1290  -1.0770   -0.9940   -0.9430   -0.8730;
   -1.1700   -1.2350   -1.2950   -1.3550   -1.4060   -1.4050   -1.4290   -1.4410   -1.4460   -1.4460   -1.4520   -1.4490   -1.4550   -1.4400   -1.4150  -1.3560   -1.2880   -1.2170   -1.1670;
   -1.3150   -1.3800   -1.4450   -1.5150   -1.5810   -1.6710   -1.6970   -1.7140   -1.7190   -1.7170   -1.7200   -1.7090   -1.6840   -1.6700   -1.6510  -1.5800   -1.4770   -1.4630   -1.3890;
   -1.5200   -1.5700   -1.6350   -1.7100   -1.7880   -1.8180   -1.8380   -1.8890   -1.9100   -1.9090   -1.9090   -1.8930   -1.8910   -1.8460   -1.8000  -1.7210   -1.6400   -1.5900   -1.5310;
   -1.6000   -1.6700   -1.7300   -1.8100   -1.8910   -1.9070   -1.9110   -1.9830   -2.0160   -2.0370   -1.9320   -1.9900   -1.9690   -1.8360   -1.9180  -1.8390   -1.7550   -1.6710   -1.6300;
   -1.5600   -1.6150   -1.6850   -1.7500   -1.8540   -1.9910   -2.0330   -1.9390   -2.0030   -1.9850   -2.0200   -2.0400   -1.9130   -1.9180   -1.9460  -1.9110   -1.8240   -1.6890   -1.6630;
   -1.3000   -1.4800   -1.6000   -1.7200   -1.8800   -1.9240   -1.9130   -1.8660   -1.8790   -1.9590   -1.9920   -2.0170   -2.0300   -1.9420   -2.0020  -1.8700   -1.7380   -1.6230   -1.4470;
   -1.7050   -1.7950   -1.8250   -1.8500   -1.9380   -1.9590   -2.0120   -1.9990   -1.9690   -2.0100   -1.9650   -1.8470   -1.8950   -1.9280   -1.9650  -1.7550   -1.6970   -1.7060   -1.6180;
   -1.7000   -1.7400   -1.7300   -1.8950   -1.9330   -1.8800   -1.9070   -1.8980   -1.8920   -1.9160   -1.9360   -1.8770   -1.9330   -1.9520   -1.9150  -1.7800   -1.7500   -1.7500   -1.6880;
   -1.6900   -1.7400   -1.7350   -1.8300   -1.8130   -1.8640   -2.0040   -1.9500   -1.9250   -1.9570   -1.9050   -1.8330   -1.9320   -1.9520   -1.8930  -1.8000   -1.8530   -1.7990   -1.7910;
   -1.9350   -1.9500   -1.9450   -1.9200   -1.8720   -1.8380   -1.9080   -1.9490   -1.8260   -1.8160   -1.8370   -1.7550   -1.8480   -1.8580   -1.7740  -1.8100   -1.8640   -1.8850   -1.8340;
   -1.9600   -1.9350   -1.8500   -1.8700   -1.9530   -2.0360   -2.0130   -1.9680   -1.9900   -1.9780   -1.9570   -1.9560   -1.9620   -2.0480   -1.9700  -1.8950   -1.8900   -1.9690   -1.9700];

% fi = -10 grad
Cy1(:,:,2) = [1.1490    1.2140    1.2640    1.2940    1.3270    1.2830    1.2660    1.2450    1.2340    1.2280    1.2580    1.2570    1.2680    1.2650    1.2360  1.1960    1.1540    1.1090    1.0630;
    0.9480    0.9950    1.0210    1.0470    1.0430    1.0400    1.0370    1.0420    1.0500    1.0590    1.0660    1.0480    1.0510    1.0400    1.0530  1.0510    1.0310    0.9990    0.9550;
    0.7550    0.7780    0.7770    0.7880    0.8010    0.7990    0.8030    0.8040    0.8120    0.8150    0.8130    0.8050    0.8040    0.8000    0.7970  0.7760    0.7690    0.7650    0.7470;
    0.3200    0.3200    0.3270    0.3320    0.3500    0.3650    0.3700    0.3720    0.3570    0.3560    0.3520    0.3490    0.3430    0.3370    0.3280  0.3170    0.3090    0.2950    0.3060;
    0.0860    0.0610    0.0410    0.0390    0.0520    0.0560    0.0620    0.0620    0.0610    0.0640    0.0620    0.0610    0.0580    0.0530    0.0470  0.0350    0.0490    0.0440    0.0760;
   -0.1920   -0.2320   -0.2620   -0.2790   -0.2800   -0.2840   -0.2810   -0.2870   -0.2870   -0.2870   -0.2890   -0.2910   -0.2890   -0.2910   -0.2930  -0.2920   -0.2750   -0.2440   -0.2050;
   -0.4550   -0.5220   -0.5750   -0.6110   -0.6240   -0.6320   -0.6410   -0.6470   -0.6500   -0.6500   -0.6510   -0.6460   -0.6420   -0.6380   -0.6350  -0.6220   -0.5870   -0.5240   -0.4630;
   -0.7140   -0.7840   -0.8460   -0.8980   -0.9330   -0.9490   -0.9670   -0.9760   -0.9770   -0.9800   -0.9800   -0.9780   -0.9770   -0.9630   -0.9510  -0.9230   -0.8640   -0.7900   -0.7170;
   -1.0050   -1.0880   -1.1610   -1.2230   -1.2630   -1.2840   -1.2990   -1.3060   -1.3020   -1.3060   -1.2920   -1.2890   -1.2870   -1.2790   -1.2660  -1.2200   -1.1460   -1.0900   -1.0050;
   -1.3130   -1.3780   -1.4450   -1.5090   -1.5600   -1.5660   -1.5830   -1.5900   -1.5950   -1.5940   -1.5970   -1.5950   -1.5950   -1.5840   -1.5660  -1.5050   -1.4450   -1.3740   -1.3070;
   -1.4180   -1.4980   -1.5780   -1.6630   -1.7460   -1.8250   -1.8480   -1.8610   -1.8610   -1.8630   -1.8630   -1.8560   -1.8360   -1.8160   -1.7950  -1.7350   -1.6130   -1.5570   -1.4700;
   -1.5420   -1.6290   -1.7190   -1.8190   -1.9190   -1.9770   -2.0330   -2.0640   -2.0790   -2.0900   -2.0810   -2.0750   -2.0670   -2.0340   -1.9890  -1.8960   -1.8000   -1.6960   -1.6130;
   -1.6710   -1.7680   -1.8620   -1.9670   -2.0740   -2.0770   -2.1510   -2.1840   -2.1990   -2.2160   -2.1920   -2.1940   -2.0840   -2.1100   -2.1110  -1.9960   -1.9000   -1.7870   -1.7020;
   -1.6150   -1.5770   -1.7700   -1.9630   -2.1300   -2.2170   -2.1840   -2.2160   -2.3060   -2.2630   -2.3040   -2.3040   -2.2420   -2.2350   -2.2100  -2.1210   -1.8910   -1.6540   -1.6960;
   -1.4060   -1.5920   -1.7160   -1.9440   -2.0260   -2.0810   -2.0810   -2.0330   -2.0310   -2.0970   -2.1180   -2.1310   -2.1420   -2.0620   -2.1290  -2.0470   -1.8190   -1.6950   -1.5090;
   -1.6880   -1.7380   -1.7210   -1.8090   -2.0140   -2.0480   -2.1120   -2.1000   -2.0580   -2.0880   -2.0670   -1.9720   -2.0160   -2.0190   -2.0250  -1.8200   -1.7320   -1.7490   -1.6690;
   -1.7240   -1.7930   -1.8000   -1.7560   -1.9490   -1.9230   -1.9750   -1.9900   -2.0050   -2.0510   -2.0210   -1.9140   -1.9560   -1.9980   -1.9850  -1.7920   -1.8360   -1.8290   -1.7600;
   -1.7430   -1.7540   -1.8110   -1.7810   -1.8390   -1.8970   -2.0040   -1.9990   -1.9860   -2.0270   -1.9430   -1.8350   -1.9250   -1.9930   -1.9210  -1.8630   -1.8930   -1.8360   -1.8250;
   -1.9350   -1.9930   -1.9790   -1.9910   -1.9280   -1.8770   -1.9310   -1.9810   -1.8920   -1.9160   -1.9380   -1.8560   -1.9430   -1.9470   -1.8480  -1.9110   -1.8990   -1.9130   -1.8550;
   -1.9900   -2.0090   -1.9500   -1.9790   -2.0060   -2.0850   -2.0190   -2.0070   -2.0190   -1.9980   -1.9900   -2.0040   -2.0360   -2.1020   -2.0030  -1.9760   -1.9470   -2.0060   -1.9870];

% fi = 0 grad
Cy1(:,:,3) = [1.0910    1.1400    1.2030    1.2150    1.2390    1.2010    1.1710    1.1570    1.1220    1.1160    1.1560    1.1600    1.1750    1.1720    1.1610  1.1450    1.1150    1.0700    1.0150;
    0.9050    0.9390    0.9590    0.9800    0.9670    0.9600    0.9540    0.9510    0.9530    0.9590    0.9660    0.9640    0.9650    0.9590    0.9780  0.9800    0.9700    0.9450    0.9100;
    0.7130    0.7180    0.7060    0.7110    0.7050    0.6990    0.6960    0.6870    0.6970    0.6920    0.7050    0.7080    0.7100    0.7100    0.7090  0.7100    0.7100    0.7100    0.7100;
    0.2650    0.2650    0.2700    0.2750    0.2880    0.3050    0.3060    0.3110    0.2850    0.2870    0.2860    0.2850    0.2800    0.2710    0.2600  0.2510    0.2510    0.2340    0.2470;
   -0.0060   -0.0300   -0.0500   -0.0500   -0.0360   -0.0350   -0.0280   -0.0270   -0.0270   -0.0250   -0.0280   -0.0280   -0.0290   -0.0310   -0.0340  -0.0480   -0.0370   -0.0330   -0.0060;
   -0.2750   -0.3150   -0.3450   -0.3600   -0.3590   -0.3640   -0.3620   -0.3680   -0.3680   -0.3670   -0.3680   -0.3720   -0.3680   -0.3700   -0.3730  -0.3770   -0.3600   -0.3350   -0.2920;
   -0.5500   -0.6200   -0.6700   -0.7050   -0.7190   -0.7270   -0.7370   -0.7410   -0.7470   -0.7500   -0.7500   -0.7460   -0.7440   -0.7360   -0.7350  -0.7250   -0.6890   -0.6260   -0.5640;
   -0.8250   -0.9100   -0.9800   -1.0350   -1.0690   -1.0890   -1.1050   -1.1110   -1.1110   -1.1120   -1.1120   -1.1080   -1.1060   -1.0980   -1.0830  -1.0530   -0.9990   -0.9240   -0.8400;
   -1.1150   -1.2000   -1.2800   -1.3400   -1.3790   -1.4050   -1.4210   -1.4310   -1.4220   -1.4180   -1.4080   -1.4050   -1.4030   -1.3960   -1.3820  -1.3360   -1.2710   -1.2100   -1.1130;
   -1.3750   -1.4400   -1.5100   -1.5750   -1.6260   -1.6350   -1.6500   -1.6550   -1.6590   -1.6580   -1.6600   -1.6580   -1.6550   -1.6460   -1.6310  -1.5700   -1.5130   -1.4420   -1.3670;
   -1.5200   -1.6150   -1.7100   -1.8100   -1.9100   -1.9770   -1.9970   -2.0060   -2.0020   -2.0080   -2.0060   -2.0010   -1.9810   -1.9610   -1.9390  -1.8880   -1.7990   -1.6510   -1.5500;
   -1.5550   -1.6650   -1.7700   -1.8850   -1.9980   -2.0730   -2.1520   -2.1710   -2.1820   -2.2000   -2.1860   -2.1860   -2.1740   -2.1490   -2.1040  -2.0020   -1.8970   -1.7600   -1.6630;
   -1.7150   -1.8300   -1.9450   -2.0650   -2.1880   -2.1830   -2.3010   -2.3100   -2.3140   -2.3280   -2.3550   -2.3210   -2.1560   -2.2810   -2.2310  -2.0950   -1.9910   -1.8600   -1.7470;
   -1.6250   -1.5700   -1.7850   -2.0000   -2.1780   -2.2720   -2.2100   -2.2640   -2.3580   -2.3110   -2.3530   -2.3500   -2.2990   -2.2900   -2.2550  -2.1570   -2.0020   -1.6480   -1.7020;
   -1.5700   -1.7350   -1.9000   -2.0500   -2.1650   -2.2540   -2.2880   -2.2580   -2.2580   -2.3260   -2.3120   -2.2900   -2.2770   -2.1840   -2.2390  -2.1090   -1.9860   -1.8480   -1.6490;
   -1.7750   -1.9000   -1.9700   -2.0550   -2.1760   -2.1840   -2.2230   -2.2110   -2.1960   -2.2520   -2.2350   -2.1450   -2.1820   -2.1650   -2.1520  -2.0250   -1.9650   -1.9310   -1.8000;
   -1.9000   -1.9350   -1.9600   -1.9950   -2.1280   -2.1110   -2.1730   -2.1830   -2.1810   -2.2080   -2.1900   -2.0940   -2.1310   -2.1500   -2.1140  -1.9900   -1.9860   -1.9620   -1.8760;
   -1.9300   -1.9450   -1.9400   -1.9200   -1.9290   -2.0210   -2.1610   -2.1600   -2.1200   -2.1340   -2.0850   -2.0110   -2.1080   -2.1250   -2.0630  -1.9700   -2.0160   -1.9470   -1.9240;
   -2.0000   -2.0450   -2.0750   -2.0800   -2.0450   -1.9940   -2.0480   -2.0920   -1.9920   -2.0040   -2.0190   -1.9300   -2.0140   -2.0180   -1.9190  -2.0000   -2.0340   -2.0320   -1.9580;
   -1.9600   -1.9500   -1.9000   -2.0100   -2.0600   -2.1580   -2.1120   -2.1170   -2.1450   -2.1400   -2.1130   -2.1070   -2.1010   -2.1690   -2.0730  -2.0740   -2.0380   -2.0880   -2.0600];

% fi = 10 grad
Cy1(:,:,4) = [1.0210    1.0660    1.1160    1.1260    1.1390    1.1080    1.1030    1.0700    1.0410    1.0390    1.0710    1.0760    1.0890    1.0860    1.0780   1.0700    1.0460    1.0070    0.9570;
    0.8150    0.8380    0.8460    0.8630    0.8540    0.8480    0.8440    0.8410    0.8460    0.8490    0.8560    0.8520    0.8530    0.8480    0.8630   0.8630    0.8560    0.8410    0.8200;
    0.6220    0.6180    0.6030    0.6090    0.6060    0.6020    0.5990    0.5920    0.6000    0.5960    0.6050    0.6070    0.6090    0.6090    0.6090   0.6070    0.6070    0.6110    0.6190;
    0.1810    0.1760    0.1790    0.1840    0.1980    0.2120    0.2130    0.2150    0.2020    0.2050    0.2020    0.1980    0.1920    0.1830    0.1800    0.1650    0.1630    0.1550    0.1690;
   -0.0690   -0.1000   -0.1250   -0.1310   -0.1220   -0.1200   -0.1140   -0.1120   -0.1150   -0.1140   -0.1170   -0.1170   -0.1210   -0.1230   -0.1260   -0.1340   -0.1220   -0.1090   -0.0750;
   -0.3390   -0.4000   -0.4440   -0.4740   -0.4800   -0.4800   -0.4810   -0.4860   -0.4870   -0.4900   -0.4900   -0.5040   -0.4960   -0.4910   -0.4930  -0.4830   -0.4580   -0.4120   -0.3540;
   -0.5850   -0.6300   -0.7150   -0.7680   -0.8060   -0.8100   -0.8240   -0.8330   -0.8440   -0.8490   -0.8510   -0.8420   -0.8460   -0.8310   -0.8250  -0.8010   -0.7520   -0.6760   -0.5950;
   -0.8430   -0.9470   -1.0310   -1.0970   -1.1330   -1.1470   -1.1670   -1.1750   -1.1820   -1.1770   -1.1710   -1.1760   -1.1750   -1.1710   -1.1580  -1.1200   -1.0470   -0.9540   -0.8640;
   -1.1040   -1.2000   -1.2870   -1.3560   -1.4040   -1.4310   -1.4460   -1.4530   -1.4450   -1.4420   -1.4350   -1.4300   -1.4340   -1.4280   -1.4140  -1.3570   -1.2860   -1.2150   -1.1100;
   -1.3620   -1.4580   -1.5600   -1.6550   -1.7410   -1.7710   -1.7710   -1.7820   -1.7940   -1.7890   -1.7870   -1.7910   -1.7750   -1.7750   -1.7600  -1.6460   -1.5820   -1.4730   -1.3700;
   -1.5200   -1.6300   -1.7400   -1.8540   -1.9680   -2.0370   -2.0700   -2.0810   -2.0830   -2.0820   -2.0800   -2.0700   -2.0540   -2.0390   -2.0160  -1.9290   -1.7900   -1.6730   -1.5630;
   -1.6900   -1.8560   -2.0060   -2.1360   -2.2520   -2.2550   -2.2600   -2.3260   -2.3170   -2.3080   -2.3550   -2.3410   -2.3020   -2.2590   -2.2190  -2.0810   -1.9380   -1.7600   -1.6790;
   -1.8490   -1.9490   -2.0540   -2.1690   -2.2900   -2.3610   -2.3430   -2.3750   -2.2840   -2.4110   -2.4190   -2.4020   -2.3450   -2.3330   -2.3030  -2.1630   -2.0650   -1.9710   -1.8780;
   -1.5900   -1.4840   -1.7410   -2.0000   -2.1930   -2.2790   -2.1860   -2.2620   -2.3950   -2.3060   -2.3730   -2.3690   -2.2950   -2.2930   -2.2620  -2.1750   -1.9930   -1.5550   -1.6610;
   -1.7070   -1.8910   -2.0130   -2.2550   -2.1410   -2.2000   -2.2040   -2.1650   -2.1790   -2.2610   -2.2830   -2.2810   -2.2940   -2.1820   -2.2180  -2.1780   -1.9360   -1.8140   -1.6300;
   -1.7350   -1.8380   -1.8440   -1.9040   -2.1330   -2.1590   -2.2170   -2.2090   -2.1840   -2.2310   -2.1860   -2.0680   -2.1150   -2.1450   -2.1800   -1.9510   -1.8910   -1.8850   -1.7820;
   -1.7990   -1.8890   -1.9170   -1.9420   -2.0970   -2.0650   -2.1120   -2.1230   -2.1400   -2.1850   -2.1640   -2.0650   -2.1070   -2.1420   -2.1210  -1.9660   -1.9410   -1.9130   -1.8230;
   -1.7530   -1.7520   -1.7970   -1.7790   -1.9870   -2.0480   -2.1570   -2.1490   -2.0480   -2.2680   -2.1780   -2.0640   -2.1420   -2.1610   -2.1020  -1.8940   -1.9120   -1.8670   -1.8680;
   -2.0670   -2.1230   -2.1070   -2.1450   -2.0530   -1.9110   -1.9740   -2.0240   -1.9260   -1.9400   -1.9670   -1.8910   -1.9780   -1.9760   -1.8720  -1.9640   -1.9260   -1.9420   -1.8860;
   -2.0080   -2.0200   -1.9550   -2.0760   -2.0260   -2.1160   -2.0610   -2.0570   -2.0730   -2.0570   -2.0340   -2.0330   -2.0300   -2.0950   -1.9960   -2.0460   -1.9250   -1.9900   -1.9780];

% fi = 25 grad
Cy1(:,:,5) = [0.7230    0.7500    0.7440    0.7440    0.7110    0.7090    0.6970    0.6970    0.6960    0.7100    0.7040    0.7150    0.7200    0.7210    0.7210  0.7500    0.7500    0.7400    0.7100;
    0.5120    0.4950    0.4610    0.4650    0.4700    0.4700    0.4710    0.4670    0.4830    0.4760    0.4810    0.4720    0.4750    0.4730    0.4730   0.4650    0.4700    0.4900    0.5150;
    0.2490    0.2120    0.1860    0.1950    0.2030    0.2050    0.2020    0.2030    0.2070    0.2050    0.2000    0.1940    0.1980    0.1980    0.2010   0.1850    0.1900    0.2100    0.2500;
    0.1000    0.0900    0.0900    0.0950    0.1110    0.1220    0.1220    0.1220    0.1210    0.1250    0.1210    0.1140    0.1070    0.0970    0.1010   0.0810    0.0770    0.0770    0.0930;
   -0.1500   -0.1900   -0.2200   -0.2350   -0.2320   -0.2240   -0.2240   -0.2210   -0.2270   -0.2280   -0.2310   -0.2320   -0.2390   -0.2410   -0.2440  -0.2440   -0.2320   -0.2070   -0.1640;
   -0.3850   -0.4600   -0.5150   -0.5550   -0.5660   -0.5630   -0.5660   -0.5710   -0.5720   -0.5780   -0.5780   -0.5990   -0.5880   -0.5770   -0.5790  -0.5590   -0.5280   -0.4670   -0.3980;
   -0.6200   -0.6900   -0.7600   -0.8300   -0.8920   -0.8910   -0.9100   -0.9240   -0.9390   -0.9460   -0.9490   -0.9360   -0.9450   -0.9240   -0.9130  -0.8760   -0.8130   -0.7240   -0.6260;
   -0.8650   -0.9900   -1.0900   -1.1700   -1.2080   -1.2150   -1.2390   -1.2500   -1.2690   -1.2530   -1.2400   -1.2550   -1.2550   -1.2570   -1.2450   -1.1980   -1.1030   -0.9900   -0.8910;
   -1.0550   -1.1950   -1.3200   -1.4300   -1.5190   -1.5500   -1.5640   -1.5580   -1.5550   -1.5540   -1.5630   -1.5490   -1.5770   -1.5780   -1.5630  -1.4530   -1.3580   -1.2410   -1.0970;
   -1.3600   -1.4600   -1.5700   -1.6700   -1.7630   -1.7970   -1.7940   -1.8060   -1.8200   -1.8140   -1.8110   -1.8160   -1.7980   -1.8000   -1.7850   -1.6610   -1.5950   -1.4790   -1.3710;
   -1.5200   -1.6350   -1.7500   -1.8700   -1.9890   -2.0580   -2.0950   -2.1070   -2.1120   -2.1080   -2.1060   -2.0940   -2.0790   -2.0660   -2.0430  -1.9430   -1.8040   -1.6810   -1.5680;
   -1.6150   -1.7500   -1.8750   -1.9950   -2.1110   -2.1540   -2.2000   -2.2400   -2.2420   -2.2480   -2.2610   -2.2550   -2.2310   -2.1980   -2.1550   -2.0370   -1.9150   -1.7600   -1.6700;
   -1.7750   -1.8750   -1.9800   -2.0950   -2.2160   -2.2870   -2.2690   -2.3010   -2.2100   -2.3370   -2.3450   -2.3280   -2.2710   -2.2590   -2.2290   -2.0890   -1.9910   -1.8970   -1.8040;
   -1.7400   -1.8450   -1.9250   -2.0000   -2.1300   -2.2510   -2.2860   -2.2700   -2.2390   -2.3270   -2.2890   -2.2880   -2.3120   -2.2820   -2.2320  -2.0990   -2.0300   -1.9450   -1.8340;
   -1.5700   -1.7400   -1.9000   -2.0500   -2.1560   -2.2160   -2.2030   -2.1580   -2.1750   -2.2610   -2.2660   -2.2620   -2.2550   -2.1530   -2.2010  -2.1100   -1.9840   -1.8410   -1.6370;
   -1.7000   -1.8100   -1.8800   -1.9500   -2.0430   -2.1700   -2.1840   -2.1110   -2.2040   -2.2310   -2.2030   -2.1020   -2.1350   -2.1730   -2.1070  -1.9500   -1.8920   -1.8770   -1.7650;
   -1.7950   -1.8950   -1.9600   -2.0200   -2.1130   -2.0940   -2.1240   -2.1240   -2.1340   -2.1740   -2.1770   -2.1030   -2.1530   -2.1760   -2.1440  -2.0000   -1.9600   -1.9300   -1.8380;
   -1.7800   -1.7850   -1.7900   -1.8100   -1.8730   -1.9430   -2.0590   -2.2740   -2.0000   -2.2590   -2.2110   -1.8850   -2.2210   -2.2120   -2.1250  -1.9700   -1.9070   -1.8820   -1.8740;
   -1.9500   -1.9800   -1.9800   -1.9600   -1.9110   -1.8810   -1.9550   -2.0050   -1.8940   -1.8990   -2.0090   -2.0140   -2.1010   -2.0140   -1.8250  -1.8400   -1.8990   -1.9220   -1.8730;
   -1.9250   -1.9200   -1.8700   -1.8850   -1.9690   -2.0710   -2.0290   -2.0390   -2.0700   -2.0690   -2.0260   -2.0050   -2.0000   -2.0850   -2.0070  -1.9350   -1.9480   -1.9890   -1.9510];

% Переход к человеческим осям
Cy1 = -Cy1;

% Коэффициент нормальной силы, носки выпущены.  носки УБРАНЫ
% Cy_nos = f(alpha2, beta)
Cy_nos1 = [1.1830    1.2460    1.2790    1.2900    1.3690    1.3640    1.2970    1.2770    1.2760    1.2560    1.2810    1.2800    1.3120    1.3150    1.3060  1.2270    1.2160    1.1830    1.1200;
    0.9600    1.0180    1.0550    1.0930    1.0580    1.0390    1.0310    1.0190    1.0250    1.0350    1.0330    1.0420    1.0430    1.0560    1.0560  1.0910    1.0530    1.0160    0.9580;
    0.7090    0.7100    0.7020    0.7040    0.7010    0.7100    0.7300    0.7290    0.7290    0.7250    0.7290    0.7280    0.7280    0.7230    0.7110   0.7140    0.7120    0.7200    0.7190;
    0.2220    0.2160    0.2310    0.2270    0.2400    0.2430    0.2440    0.2490    0.2490    0.2480    0.2480    0.2420    0.2390    0.2350    0.2290   0.2160    0.2200    0.2050    0.2110;
   -0.0660   -0.0840   -0.0900   -0.1050   -0.1040   -0.0990   -0.1070   -0.0990   -0.0990   -0.1000   -0.1010   -0.1040   -0.1040   -0.1040   -0.1060  -0.1070   -0.0920   -0.0860   -0.0680;
   -0.3170   -0.3470   -0.3900   -0.4140   -0.4200   -0.4170   -0.4170   -0.4210   -0.4240   -0.4280   -0.4210   -0.4280   -0.4220   -0.4230   -0.4250  -0.4190   -0.3950   -0.3520   -0.3220;
   -0.5690   -0.6190   -0.6790   -0.7030   -0.7280   -0.7650   -0.7720   -0.7740   -0.7720   -0.7740   -0.7700   -0.7670   -0.7610   -0.7540   -0.7560  -0.7310   -0.7070   -0.6450   -0.5970;
   -0.8530   -0.9290   -1.0180   -1.0700   -1.0980   -1.1160   -1.1144   -1.1510   -1.1420   -1.1390   -1.1350   -1.1180   -1.1120   -1.1070   -1.0990  -1.0710   -1.0190   -0.9300   -0.8540;
   -1.1060   -1.1680   -1.2280   -1.3140   -1.3480   -1.3590   -1.3620   -1.3520   -1.3570   -1.3550   -1.3710   -1.3760   -1.3700   -1.3790   -1.3990  -1.3650   -1.2790   -1.2190   -1.1570;
   -1.3140   -1.4070   -1.4650   -1.5060   -1.5640   -1.5980   -1.6280   -1.6470   -1.6460   -1.6500   -1.6420   -1.6410   -1.6180   -1.5990   -1.5850  -1.5270   -1.4860   -1.4280   -1.3350;
   -1.4960   -1.5100   -1.5890   -1.6920   -1.7750   -1.8140   -1.8460   -1.8750   -1.8790   -1.8830   -1.8910   -1.8760   -1.8430   -1.8380   -1.8110  -1.7280   -1.6250   -1.5460   -1.5320;
   -1.5940   -1.6940   -1.8070   -1.8750   -1.9570   -1.9760   -2.0320   -2.0600   -2.0700   -2.0770   -2.0380   -2.0390   -2.0280   -2.0050   -1.9860  -1.9040   -1.8360   -1.7130   -1.6230;
   -1.6830   -1.7550   -1.9120   -1.9990   -2.1110   -2.1490   -2.1470   -2.2040   -2.2070   -2.2040   -2.2050   -2.1950   -2.1930   -2.1740   -2.1330  -2.0210   -1.9340   -1.7770   -1.7050;
   -1.6640   -1.7830   -1.8590   -1.9620   -2.0300   -2.1290   -1.9170   -2.1430   -2.0500   -2.2080   -2.2010   -2.1820   -2.0770   -2.2090   -2.1260  -2.0580   -1.9550   -1.8790   -1.7600];

Cy_nos1 = -Cy_nos1;

% Производная продольной силы по угловой скорости тангажа
% Cywz = f(alpha1)
Cywz1 = [-23.9000  -23.9000  -23.9000  -29.5000  -29.5000  -30.5000  -31.3000  -30.1000  -27.7000  -28.2000  -29.0000  -29.8000  -38.3000  -35.3000  -32.3000 -27.3000  -25.2000  -27.3000   -9.3500   -2.1600]';
Cywz1 = -Cywz1;

% Влияние носков
% dCywz_nos = f(alpha2)
dCywz_nos1 = [15.1000   15.1000   15.1000    3.7000    0.6000   -1.3000    0.3000   -3.8000   -4.6000   -0.2000   -2.7000   -3.5000   -1.3000   -0.6500]';
dCywz_nos1 = -dCywz_nos1;

dCy_sb1 = [-0.38580 -0.38580 -0.38580 -0.38580 -0.38580 -0.26850 -0.30210 -0.42480 -0.20940 -0.09690 0.04380 0.09470 0.00140 -0.00970 -0.01530 -0.05200 -0.00100 -0.02020 -0.03690 -0.03690];
dCy_sb1 = -dCy_sb1;

interpCy = csaps({alpha1, beta1, fi1}, Cy1, 1.0 - 1.0e-6);
interpCy_nos = csaps({alpha2, beta1}, Cy_nos1, 1.0 - 1.0e-6);
interpCywz1 = pchip(alpha1, Cywz1);
interpCywz2 = pchip(alpha2, dCywz_nos1);
interpdCy_sb = pchip(alpha1, dCy_sb1);
end

ntraj = size(alpha, 2);
Cy = zeros(1, ntraj);
Cy0 = zeros(1, ntraj);
Cy_nos = zeros(1, ntraj);
Cywz = zeros(1, ntraj);
dCy_sb = zeros(1, ntraj);
for i = 1 : ntraj
    Cy(i) = fnval(interpCy, {(alpha(i)), (beta(i)), (fi(i))});
    Cy0(i) = fnval(interpCy, {(alpha(i)), (beta(i)), (0)});
    Cy_nos(i) = fnval(interpCy_nos, {(alpha(i)), (beta(i))});
    Cywz(i) = fnval(interpCywz1, alpha(i)) + fnval(interpCywz2, alpha(i)).*(dnos / deg2rad(25));
    dCy_sb(i) = fnval(interpdCy_sb, alpha(i));
end

% Сборка
dCy_nos = Cy_nos - Cy0;

Cy = Cy + dCy_nos.*(dnos / deg2rad(25)) + Cywz.*((Wz.*ba)./(2 * V)) + dCy_sb.*(sb / deg2rad(60));
