"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.accountIfDifferentFromStack = exports.regionIfDifferentFromStack = void 0;
const cdk = require("@aws-cdk/core");
/**
 * Make a Token that renders to given region if used in a different stack, otherwise undefined
 */
function regionIfDifferentFromStack(region) {
    return cdk.Token.asString(new StackDependentToken(region, stack => stack.region));
}
exports.regionIfDifferentFromStack = regionIfDifferentFromStack;
/**
 * Make a Token that renders to given account if used in a different stack, otherwise undefined
 */
function accountIfDifferentFromStack(account) {
    return cdk.Token.asString(new StackDependentToken(account, stack => stack.account));
}
exports.accountIfDifferentFromStack = accountIfDifferentFromStack;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(originalValue, fn) {
        this.originalValue = originalValue;
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        const stackValue = this.fn(cdk.Stack.of(context.scope));
        // Don't render if the values are definitely the same. If the stack
        // is unresolved we don't know, better output the value.
        if (!cdk.Token.isUnresolved(stackValue) && stackValue === this.originalValue) {
            return undefined;
        }
        return this.originalValue;
    }
    toString() {
        return cdk.Token.asString(this);
    }
    toJSON() {
        return this.originalValue;
    }
}
//# sourceMappingURL=data:application/json;base64,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