"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const prettier_1 = require("./prettier");
const common_1 = require("../common");
const component_1 = require("../component");
const json_1 = require("../json");
const yaml_1 = require("../yaml");
/**
 * Represents eslint configuration.
 */
class Eslint extends component_1.Component {
    constructor(project, options) {
        super(project);
        this._plugins = new Array();
        this._extends = new Array();
        this.nodeProject = project;
        project.addDevDeps("eslint@^8", "@typescript-eslint/eslint-plugin@^5", "@typescript-eslint/parser@^5", "eslint-import-resolver-node", "eslint-import-resolver-typescript", "eslint-plugin-import");
        if (options.aliasMap) {
            project.addDevDeps("eslint-import-resolver-alias");
        }
        const devdirs = options.devdirs ?? [];
        const dirs = [...options.dirs, ...devdirs];
        const fileExtensions = options.fileExtensions ?? [".ts"];
        this._allowDevDeps = new Set((devdirs ?? []).map((dir) => `**/${dir}/**`));
        const lintProjenRc = options.lintProjenRc ?? true;
        const lintProjenRcFile = options.lintProjenRcFile ?? common_1.PROJEN_RC;
        this.eslintTask = project.addTask("eslint", {
            description: "Runs eslint against the codebase",
            exec: [
                "eslint",
                `--ext ${fileExtensions.join(",")}`,
                "--fix",
                "--no-error-on-unmatched-pattern",
                ...dirs,
                ...(lintProjenRc && lintProjenRcFile ? [lintProjenRcFile] : []),
            ].join(" "),
        });
        project.testTask.spawn(this.eslintTask);
        // exclude some files
        project.npmignore?.exclude("/.eslintrc.json");
        this._formattingRules = {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            indent: ["off"],
            "@typescript-eslint/indent": ["error", 2],
            // Style
            quotes: ["error", "single", { avoidEscape: true }],
            "comma-dangle": ["error", "always-multiline"],
            "comma-spacing": ["error", { before: false, after: true }],
            "no-multi-spaces": ["error", { ignoreEOLComments: false }],
            "array-bracket-spacing": ["error", "never"],
            "array-bracket-newline": ["error", "consistent"],
            "object-curly-spacing": ["error", "always"],
            "object-curly-newline": ["error", { multiline: true, consistent: true }],
            "object-property-newline": [
                "error",
                { allowAllPropertiesOnSameLine: true },
            ],
            "keyword-spacing": ["error"],
            "brace-style": ["error", "1tbs", { allowSingleLine: true }],
            "space-before-blocks": ["error"],
            curly: ["error", "multi-line", "consistent"],
            "@typescript-eslint/member-delimiter-style": ["error"],
            // Require semicolons
            semi: ["error", "always"],
            // Max line lengths
            "max-len": [
                "error",
                {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                },
            ],
            // Don't unnecessarily quote properties
            "quote-props": ["error", "consistent-as-needed"],
        };
        this.rules = {
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            "@typescript-eslint/no-require-imports": ["error"],
            // Require all imported dependencies are actually declared in package.json
            "import/no-extraneous-dependencies": [
                "error",
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true,
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            "import/no-unresolved": ["error"],
            // Require an ordering on all imports
            "import/order": [
                "warn",
                {
                    groups: ["builtin", "external"],
                    alphabetize: { order: "asc", caseInsensitive: true },
                },
            ],
            // Cannot import from the same module twice
            "no-duplicate-imports": ["error"],
            // Cannot shadow names
            "no-shadow": ["off"],
            "@typescript-eslint/no-shadow": ["error"],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            "key-spacing": ["error"],
            // No multiple empty lines
            "no-multiple-empty-lines": ["error"],
            // One of the easiest mistakes to make
            "@typescript-eslint/no-floating-promises": ["error"],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            "no-return-await": ["off"],
            "@typescript-eslint/return-await": ["error"],
            // Useless diff results
            "no-trailing-spaces": ["error"],
            // Must use foo.bar instead of foo['bar'] if possible
            "dot-notation": ["error"],
            // Are you sure | is not a typo for || ?
            "no-bitwise": ["error"],
            // Member ordering
            "@typescript-eslint/member-ordering": [
                "error",
                {
                    default: [
                        "public-static-field",
                        "public-static-method",
                        "protected-static-field",
                        "protected-static-method",
                        "private-static-field",
                        "private-static-method",
                        "field",
                        // Constructors
                        "constructor",
                        // Methods
                        "method",
                    ],
                },
            ],
        };
        // Overrides for .projenrc.js
        this.overrides = [
            {
                files: [lintProjenRcFile || common_1.PROJEN_RC],
                rules: {
                    "@typescript-eslint/no-require-imports": "off",
                    "import/no-extraneous-dependencies": "off",
                },
            },
        ];
        this.ignorePatterns = options.ignorePatterns ?? [
            "*.js",
            `!${lintProjenRcFile || common_1.PROJEN_RC}`,
            "*.d.ts",
            "node_modules/",
            "*.generated.ts",
            "coverage",
        ];
        const tsconfig = options.tsconfigPath ?? "./tsconfig.json";
        this.addPlugins("@typescript-eslint");
        this.addPlugins("import");
        this.addExtends("plugin:import/typescript");
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins: () => this._plugins,
            parser: "@typescript-eslint/parser",
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: "module",
                project: tsconfig,
            },
            extends: () => this._extends,
            settings: {
                "import/parsers": {
                    "@typescript-eslint/parser": [".ts", ".tsx"],
                },
                "import/resolver": {
                    ...(options.aliasMap && {
                        alias: {
                            map: Object.entries(options.aliasMap).map(([k, v]) => [k, v]),
                            extensions: options.aliasExtensions,
                        },
                    }),
                    node: {},
                    typescript: {
                        project: tsconfig,
                        ...(options.tsAlwaysTryTypes !== false && { alwaysTryTypes: true }),
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: () => ({ ...this._formattingRules, ...this.rules }),
            overrides: this.overrides,
        };
        if (options.yaml) {
            new yaml_1.YamlFile(project, ".eslintrc.yml", {
                obj: this.config,
                marker: true,
            });
        }
        else {
            new json_1.JsonFile(project, ".eslintrc.json", {
                obj: this.config,
                // https://eslint.org/docs/latest/user-guide/configuring/configuration-files#comments-in-configuration-files
                marker: true,
                allowComments: true,
            });
        }
        // if the user enabled prettier explicitly _or_ if the project has a
        // `Prettier` component, we shall tweak our configuration accordingly.
        if (options.prettier || prettier_1.Prettier.of(project)) {
            this.enablePrettier();
        }
    }
    /**
     * Returns the singletone Eslint component of a project or undefined if there is none.
     */
    static of(project) {
        const isEslint = (c) => c instanceof Eslint;
        return project.components.find(isEslint);
    }
    /**
     * Add an eslint rule.
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * Adds an eslint plugin
     * @param plugins The names of plugins to add
     */
    addPlugins(...plugins) {
        this._plugins.push(...plugins);
    }
    /**
     * Add an eslint override.
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * Do not lint these files.
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * Adds an `extends` item to the eslint configuration.
     * @param extendList The list of "extends" to add.
     */
    addExtends(...extendList) {
        this._extends.push(...extendList);
    }
    /**
     * Add a glob file pattern which allows importing dev dependencies.
     * @param pattern glob pattern.
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.add(pattern);
    }
    /**
     * Enables prettier for code formatting.
     */
    enablePrettier() {
        this.nodeProject.addDevDeps("prettier", "eslint-plugin-prettier", "eslint-config-prettier");
        this.addPlugins("prettier");
        this._formattingRules = {
            "prettier/prettier": ["error"],
        };
        this.addExtends("prettier", "plugin:prettier/recommended");
    }
    renderDevDepsAllowList() {
        return Array.from(this._allowDevDeps);
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.javascript.Eslint", version: "0.67.87" };
//# sourceMappingURL=data:application/json;base64,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