# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['metabase', 'metabase.mbql', 'metabase.resources']

package_data = \
{'': ['*']}

install_requires = \
['pandas>=1.0.0,<2.0.0', 'requests']

setup_kwargs = {
    'name': 'metabase-python',
    'version': '0.3.3',
    'description': 'Python wrapper for interacting with the Metabase API.',
    'long_description': '# metabase-python\n[![main](https://github.com/chasleslr/metabase-python/actions/workflows/main.yml/badge.svg)](https://github.com/chasleslr/metabase-python/actions/workflows/main.yml)\n[![codecov](https://codecov.io/gh/chasleslr/metabase-python/branch/main/graph/badge.svg?token=15G7HOQ1CM)](https://codecov.io/gh/chasleslr/metabase-python)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\nAn unofficial Python library for the [Metabase API](https://www.metabase.com/learn/administration/metabase-api).\n\n\n## Installation\n\n```\npip install metabase-python\n```\n\n## Usage\n\n### Connection\n\nStart by creating an instance of Metabase with your credentials.\n```python\nfrom metabase import Metabase\n\nmetabase = Metabase(\n    host="<host>",\n    user="<username/email>",\n    password="<password>",\n)\n```\n\n### Interacting with Endpoints\nYou can then interact with any of the supported endpoints through the classes included in this package. Methods that\ninstantiate an object from the Metabase API require the `using` parameter which expects an instance of `Metabase` such\nas the one we just instantiated above. All changes are reflected in Metabase instantly.\n\n```python\nfrom metabase import User\n\n# get all objects\nusers = User.list(using=metabase)\n\n# get an object by ID\nuser = User.get(1, using=metabase)\n\n# attributes are automatically loaded and available in the instance\nif user.is_active:\n    print("User is active!")\n\n# update any available attribute\nuser.update(is_superuser=True)\n\n# delete an object\nuser.delete()\n\n# create an object\nnew_user = User.create(\n    using=metabase,\n    first_name="<first_name>",\n    last_name="<last_name>",\n    email="<email>",\n    password="<password>"\n)\n```\n\nThe methods `.list()`, `.get()`, `.create()`, `.update()`, `.delete()` are available on all\nendpoints that support them in Metabase API.\n\nSome endpoints also support additional methods:\n\n```python\nfrom metabase import User\n\nuser = User.get(1, using=metabase)\n\nuser.reactivate()   # Reactivate user\nuser.send_invite()  # Resend the user invite email for a given user.\n```\n\nHere\'s a slightly more advanced example:\n```python\nfrom metabase import User, PermissionGroup, PermissionMembership\n\n# create a new PermissionGroup\nmy_group = PermissionGroup.create(name="My Group", using=metabase)\n\nfor user in User.list():\n    # add all users to my_group\n    PermissionMembership.create(\n        group_id=my_group.id,\n        user_id=user.id,\n        using=metabase,\n    )\n```\n\n### Querying & MBQL\n\nYou can also execute queries and get results back as a Pandas DataFrame. You can provide the exact MBQL, or use\nthe `Query` object to compile MBQL (i.e. Metabase Query Language) from Python classes included in this package.\n\n```python\nfrom metabase import Dataset, Query, Count, GroupBy, TemporalOption\n\ndataset = Dataset.create(\n    database=1,\n    type="query",\n    query={\n        "source-table": 1,\n        "aggregation": [["count"]],\n        "breakout": ["field", 7, {"temporal-unit": "year"},],\n    },\n    using=metabase,\n)\n\n# compile the MBQL above using the Query object\ndataset = Dataset.create(\n    database=1,\n    type="query",\n    query=Query(\n        table_id=2,\n        aggregations=[Count()],\n        group_by=[GroupBy(id=7, option=TemporalOption.YEAR)]\n    ).compile(),\n    using=metabase\n)\n\ndf = dataset.to_pandas()\n```\n\nAs shown above, the `Query` object allows you to easily compile MBQL from Python objects. Here is a\nmore complete example:\n```python\nfrom metabase import Query, Sum, Average, Metric, Greater, GroupBy, BinOption, TemporalOption\n\nquery = Query(\n    table_id=5,\n    aggregations=[\n        Sum(id=5),                                  # Provide the ID for the Metabase field\n        Average(id=5, name="Average of Price"),     # Optionally, you can provide a name\n        Metric.get(5)                               # You can also provide your Metabase Metrics\n    ],\n    filters=[\n        Greater(id=1, value=5.5)                    # Filter for values of FieldID 1 greater than 5.5\n    ],\n    group_by=[\n        GroupBy(id=4),                              # Group by FieldID 4\n        GroupBy(id=5, option=BinOption.AUTO),       # You can use Metabase\'s binning feature for numeric fields\n        GroupBy(id=5, option=TemporalOption.YEAR)   # Or it\'s temporal option for date fields\n    ]\n)\n\nprint(query.compile())\n{\n    \'source-table\': 5,\n    \'aggregation\': [\n        [\'sum\', [\'field\', 5, None]],\n        [\'aggregation-options\', [\'avg\', [\'field\', 5, None]], {\'name\': \'Average of Price\', \'display-name\': \'Average of Price\'}],\n        ["metric", 5]\n    ],\n    \'breakout\': [\n        [\'field\', 4, None],\n        [\'field\', 5, {\'binning\': {\'strategy\': \'default\'}}],\n        [\'field\', 5, {\'temporal-unit\': \'year\'}]\n    ],\n    \'filter\': [\'>\', [\'field\', 1, None], 5.5]\n}\n```\n\nThis can also be used to more easily create `Metric` objects.\n\n```python\nfrom metabase import Metric, Query, Count, EndsWith, CaseOption\n\n\nmetric = Metric.create(\n    name="Gmail Users",\n    description="Number of users with a @gmail.com email address.",\n    table_id=2,\n    definition=Query(\n        table_id=1,\n        aggregations=[Count()],\n        filters=[EndsWith(id=4, value="@gmail.com", option=CaseOption.CASE_INSENSITIVE)]\n    ).compile(),\n    using=metabase\n)\n```\n\n\n\n## Endpoints\n\nFor a full list of endpoints and methods, see [Metabase API](https://www.metabase.com/docs/latest/api-documentation.html).\n\n| Endpoints             | Support    | Notes |\n|-----------------------|:----------:|-------|\n| Activity              |  ❌        |       |\n| Alert                 |  ❌        |       |\n| Automagic dashboards  |  ❌        |       |\n| Card                  |  ✅        |       |\n| Collection            |  ❌        |       |\n| Dashboard             |  ❌        |       |\n| Database              |  ✅        |       |\n| Dataset               |  ✅        |       |\n| Email                 |  ❌        |       |\n| Embed                 |  ❌        |       |\n| Field                 |  ✅        |       |\n| Geojson               |  ❌        |       |\n| Ldap                  |  ❌        |       |\n| Login history         |  ❌        |       |\n| Metric                |  ✅        |       |\n| Native query snippet  |  ❌        |       |\n| Notify                |  ❌        |       |\n| Permissions           |  ✅        |       |\n| Premium features      |  ❌        |       |\n| Preview embed         |  ❌        |       |\n| Public                |  ❌        |       |\n| Pulse                 |  ❌        |       |\n| Revision              |  ❌        |       |\n| Search                |  ❌        |       |\n| Segment               |  ✅        |       |\n| Session               |  ❌        |       |\n| Setting               |  ❌        |       |\n| Setup                 |  ❌        |       |\n| Slack                 |  ❌        |       |\n| Table                 |  ✅        |       |\n| Task                  |  ❌        |       |\n| Tiles                 |  ❌        |       |\n| Transform             |  ❌        |       |\n| User                  |  ✅        |       |\n| Util                  |  ❌        |       |\n\n## Contributing\nContributions are welcome!\n\n## License\nThis library is distributed under the MIT license.\n',
    'author': 'Charles Lariviere',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/chasleslr/metabase-python',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
