# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aws_sqs_consumer']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.16.0,<2.0.0',
 'botocore>=1.20.0,<2.0.0',
 'single-source>=0.2.0,<0.3.0']

setup_kwargs = {
    'name': 'aws-sqs-consumer',
    'version': '0.0.13',
    'description': 'AWS SQS Consumer',
    'long_description': '# Python AWS SQS Consumer\n\n![PyPI](https://img.shields.io/pypi/v/aws-sqs-consumer?color=blue)\n![Build passing](https://github.com/FlyweightGroup/aws_sqs_consumer_python/actions/workflows/tests.yml/badge.svg?event=push)\n\nWrite Amazon Simple Queue Service (SQS) consumers in Python with simplified interface. Define your logic to process an SQS message. After you\'re done processing, messages are deleted from the queue.\n\nHeavily inspired from [https://github.com/bbc/sqs-consumer](https://github.com/bbc/sqs-consumer), a similar JavaScript interface.\n\n## Installation\n\n```\npip install aws-sqs-consumer\n```\n\n## Usage\n\n### Simple usage\n\n```python\nfrom aws_sqs_consumer import Consumer, Message\n\nclass SimpleConsumer(Consumer):\n    def handle_message(self, message: Message):\n        # Write your logic to handle a single `message`.\n        print("Received message: ", message.Body)\n\nconsumer = SimpleConsumer(\n    queue_url="https://sqs.eu-west-1.amazonaws.com/12345678901/test_queue",\n    polling_wait_time_ms=5\n)\nconsumer.start()\n```\n\n* `consumer.start()` will block the main thread.\n* Consumer uses [SQS long polling](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-short-and-long-polling.html#sqs-long-polling) by default with configurable wait time between polls (`polling_wait_time_ms`).\n* By default, messages are processed one by one. The `handle_message` method must be finished for processing the next one. For receiving messages in batches, use the `batch_size` attribute. [See all attributes](#attributes).\n* Messages are deleted from the queue after `handle_message` is successfully completed. \n* Raising an exception in the handler function will not delete the message from the queue. Define your behavior for handling exceptions by overriding `handle_processing_exception(message, exception)` method.  See [Handling exceptions](#handling-exceptions)\n\n### Receiving messages in batches\n\nSQS supports receiving messages in batches. Setting `batch_size > 1` will fetch multiple messages in a single call to SQS API. Override `handle_message_batch(messages)` method to process the message batch.\n\nNote that only after `handle_message_batch` is finished, the next batch of messages is fetched. Maximum supported `batch_size` is `10`.\n\n```python\nfrom typing import List\nfrom aws_sqs_consumer import Consumer, Message\n\nclass BatchConsumer(Consumer):\n    def handle_message_batch(self, messages: List[Message]):\n        print(f"Received {len(messages)} Messages")\n        for message in messages:\n            print(f"\\t{message.Body}")\n\nconsumer = BatchConsumer(\n    queue_url="https://sqs.eu-west-1.amazonaws.com/12345678901/test_queue",\n    batch_size=5,\n    polling_wait_time_ms=5,\n)\nconsumer.start()\n```\n\n### Handling exceptions\n\n```python\nfrom aws_sqs_consumer import Consumer, Message\n\nclass SimpleConsumer(Consumer):\n    def handle_message(self, message: Message):\n        print(f"Processing message: {message.Body}")\n        raise Exception("Something went wrong!")\n\n    def handle_processing_exception(self, message: Message, exception):\n        # Define your logic to handle exception\n        print(f"Exception occured while processing: {exception}")\n\nconsumer = SimpleConsumer(\n    queue_url="https://sqs.eu-west-1.amazonaws.com/12345678901/test_queue",\n    polling_wait_time_ms=5\n)\nconsumer.start()\n```\n\n* Override `handle_batch_processing_exception(messages: List[Message], exception)` in case of `batch_size` > 1.\n\n### Long and short polling\n\n* **Short polling** - If you set `wait_time_seconds=0`, it is short polling. If you also set `polling_wait_time_ms=0` (which is default), you will be making a lot of (unregulated) HTTP calls to AWS.\n* **Long polling** - With `wait_time_seconds > 0`, it is long polling.\n\nFor a detailed explanation, refer [Amazon SQS short and long polling](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-short-and-long-polling.html).\n\n### AWS Credentials\n\nConsumer uses [`boto3`](https://boto3.amazonaws.com/v1/documentation/api/latest/guide/quickstart.html) for interacting with SQS. Simplest option is to set the following environment variables:\n\n```bash\nexport AWS_SECRET_ACCESS_KEY=...\nexport AWS_ACCESS_KEY_ID=...\n```\n\nIf you want to manually configure the credentials, pass custom `boto3.Client` object to `Consumer`:\n\n```python\nimport boto3\nfrom aws_sqs_consumer import Consumer, Message\n\nclass SimpleConsumer(Consumer):\n    def handle_message(self, message: Message):\n        print(f"Received message: {message.Body}")\n\nsqs_client = boto3.client(\n    \'sqs\',\n    aws_access_key_id=ACCESS_KEY,\n    aws_secret_access_key=SECRET_KEY,\n    aws_session_token=SESSION_TOKEN\n)\n\nconsumer = SimpleConsumer(\n    queue_url="https://sqs.eu-west-1.amazonaws.com/12345678901/test_queue",\n    polling_wait_time_ms=5,\n    sqs_client=sqs_client\n)\nconsumer.start()\n```\n\nSee [`boto3` latest credentials guideline](https://boto3.amazonaws.com/v1/documentation/api/latest/guide/credentials.html).\n\n## API\n\n### `Consumer(...)`\n\nCreates a new SQS consumer. Default parameters:\n\n```python\nconsumer = Consumer(\n    queue_url, # REQUIRED\n    region="eu-west-1",\n    sqs_client=None,\n    attribute_names=[],\n    message_attribute_names=[],\n    batch_size=1,\n    wait_time_seconds=1,\n    visibility_timeout_seconds=None,\n    polling_wait_time_ms=0\n)\n```\n\n#### Attributes\n\n| Attribute                                                                                                                     | Description                                                                                                                                                                                                                                                                                                                         | Default       | Example(s)                                                                                                                            |\n|-------------------------------------------------------------------------------------------------------------------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------|---------------------------------------------------------------------------------------------------------------------------------------|\n| `queue_url` (`str`)                                                                                                           | SQS Queue URL                                                                                                                                                                                                                                                                                                                       | **REQUIRED**  | `"https://sqs.eu-west-1.amazonaws.com/12345678901/test_queue"`                                                                        |\n| `region` (`str`)                                                                                                              | AWS region for internally creating an SQS client using `boto3`                                                                                                                                                                                                                                                                      | `"eu-west-1"` | `"us-east-1"`, `"ap-south-1"`                                                                                                         |\n| `sqs_client` ([`boto3.SQS.Client`](https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/sqs.html#id57)) | Override this to pass your own SQS client. This takes precedence over `region`                                                                                                                                                                                                                                                      | `None`        | `sqs_client = boto3.client("sqs", region_name="ap-south-1")`                                                                          |\n| `attribute_names` (`list`)                                                                                                    | List of attributes that need to be returned along with each message.                                                                                                                                                                                                                                                                | `[]`          | - `["All"]` - Returns all values.<br>- `["ApproximateFirstReceiveTimestamp", "ApproximateReceiveCount", "SenderId", "SentTimestamp"]` |\n| `message_attribute_names` (`list`)                                                                                            | List of names of message attributes, i.e. metadata you have passed to each message while sending to the queue.                                                                                                                                                                                                                      | `[]`          | `["CustomAttr1", "CustomAttr2"]`                                                                                                      |\n| `batch_size` (`int`)                                                                                                          | Number of messages to return at once. (Maximum `10`)<br><br>- If `batch_size = 1`, override `handle_message(message)` and `handle_processing_exception(message, exception)` methods.<br>- If `batch_size > 1`, override `handle_message_batch(messages)` and `handle_batch_processing_exception(messages, exception)` methods.<br>  | `1`           |                                                                                                                                       |\n| `wait_time_seconds` (`int`)                                                                                                   | The duration (in seconds) for which the call waits for a message to arrive in the queue before returning. If a message is available, the call returns sooner than `wait_time_seconds`.                                                                                                                                              | `1`           |                                                                                                                                       |\n| `visibility_timeout_seconds` (`int`)                                                                                          | The duration (in seconds) that the received messages are hidden from subsequent retrieve requests after being retrieved. <br><br>If this is `None`, visibility timeout of the queue is used.                                                                                                                                        | `None`        | `30`                                                                                                                                  |\n| `polling_wait_time_ms` (`int`)                                                                                                | The duration (in ms) between two subsequent polls.                                                                                                                                                                                                                                                                                  | `0`           | `2000` (2 seconds)                                                                                                                    |\n\n#### `consumer.start()`\n\nStarts polling the queue. This blocks the main thread.\n\n#### `consumer.stop()`\n\nStops polling the queue. You can use this method only if you run the `consumer` in a separate thread.\n\n#### `handle_message(message)`\n\nOverride this method to define logic for handling a single message. By default, this does nothing (i.e. `pass`). This is called only if `batch_size=1`.\n\n```python\nfrom aws_sqs_consumer import Consumer, Message\n\nclass SimpleConsumer(Consumer):\n    def handle_message(self, message: Message):\n        print(f"Processing message: {message.Body}")\n```\n\n#### `handle_processing_exception(message, exception)`\n\nOverride this method to handle any exception processing the message, including message deletion. By default, stack trace is printed to the console. This is called only if `batch_size=1`.\n\nSee [Handling exceptions](#handling-exceptions).\n\n#### `handle_message_batch(messages)`\n\nOverride this method to define logic for handling a message batch. By default, this does nothing (i.e. `pass`). This is called only if `batch_size > 1`.\n\nSee [Receiving messages in batches](#receiving-messages-in-batches).\n\n#### `handle_batch_processing_exception(messages, exception)`\n\nOverride this method to handle any exception processing a message batch, including message batch deletion. By default, stack trace is printed to the console. This is called only if `batch_size > 1`.\n\n```python\nfrom typing import List\nfrom aws_sqs_consumer import Consumer, Message\n\nclass BatchConsumer(Consumer):\n    def handle_message_batch(self, messages: List[Message]):\n        print(f"Received {len(messages)} Messages")\n        raise Exception("Failed to process message batch")\n\n    def handle_batch_processing_exception(messages: List[Message], exception):\n        print(f"Exception occurred while processing message batch: {exception}")\n```\n\n### `Message`\n\n`Message` represents a single SQS message. It is defined as a Python `dataclass` with the following attributes:\n\n* `MessageId` (`str`) - A unique identifier for the message.\n* `ReceiptHandle` (`str`) - An identifier associated with the act of receiving the message.\n* `MD5OfBody` (`str`) - An MD5 digest of the non-URL-encoded message body string.\n* `Body` (`str`) - The message\'s contents.\n* `Attributes` (`Dict[str, str]`) - A map of the attributes requested in `attribute_names` parameter in `Consumer`.\n* `MD5OfMessageAttributes` (`str`) - An MD5 digest of the non-URL-encoded message attribute string.\n* `MessageAttributes` (`Dict[str, MessageAttributeValue]`) - Dictionary of user defined message attributes.\n\n**Example:**\n\n```python\ndef handle_message(self, message: Message):\n    print("MessageID: ", message.MessageId)\n    print("ReceiptHandle: ", message.ReceiptHandle)\n    print("MD5OfBody: ", message.MD5OfBody)\n    print("Body: ", message.Body)\n    print("Attributes: ", message.Attributes)\n```\n\n```\nMessageID:  29bab209-989d-41f3-85b4-c0e9f8d8b7a9\nReceiptHandle:  AQEBU2VaFVLF6eXzFVLwPIFCqrZC0twP+qzfy2mi...==\nMD5OfBody:  c72b9698fa1927e1dd12d3cf26ed84b3\nBody:  test message\nAttributes:  {\'ApproximateFirstReceiveTimestamp\': \'1640985674001\'}\n```\n\n### `MessageAttributeValue`\n\n`MessageAttributeValue` represents a user-defined SQS message attribute value. It is defined as a Python `dataclass` with the following attributes:\n\n* `StringValue` (`str`) - attribute value, if it is a `str`.\n* `BinaryValue` (`bytes`) - Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.\n* `DataType` (`str`) - SQS supports `String`, `Number`, and `Binary`. For the Number data type, you must use `StringValue`.\n\n**Example:**\n\n```python\ndef handle_message(self, message: Message):\n    msg_attributes = message.MessageAttributes\n    host = msg_attributes["host"].StringValue\n    age  = msg_attributes["age"].StringValue\n\n    print("Message body=", message.Body)\n    print("Message attribute host=", host)\n    print("Message attribute age=", age)\n```\n\n```\nMessage body=test message\nMessage attribute host=host001.example.com\nMessage attribute age=20\n```\n\n## FAQ\n\n### Does this support parallelization?\n\nNo. A message is fetched from the queue, processed, next message is fetched, and so on.\n\nHowever, you can run multiple copies of your consumer script on different instances. Make sure you set a sufficient visibility timeout while creating the SQS queue: \n* For example, consider you have set `5m` of visibility timeout and run two instances of the script. \n* If `Consumer 1` receives message `m1` at `11:00 AM`, it has to be processed and deleted before `11:05 AM`. Otherwise, `Consumer 2` can receive `m1` after `11:05 AM` resulting in duplication.\n\n### How do I configure AWS access to the queue?\n\nThe consumer needs permission to **receive** and **delete** messages from the queue. Here is a sample IAM Policy:\n\n```json\n{\n    "Version": "2012-10-17",\n    "Statement": [\n        {\n            "Effect": "Allow",\n            "Action": ["sqs:ReceiveMessage", "sqs:DeleteMessage"],\n            "Resource": [\n                "arn:aws:sqs:eu-west-1:12345678901:test_queue",\n            ]\n        }\n    ]\n}\n```\n',
    'author': 'Flyweight Group',
    'author_email': 'nobody@flyweightgroup.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/FlyweightGroup/aws_sqs_consumer_python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
