import os
import dask
import sys

import xarray as xr
import numpy as np

from casacore import tables as ctables

from prettytable import PrettyTable

from astrohack._utils import _system_message as console

from astrohack._utils._holog import _create_holog_meta_data 

from astrohack._utils._io import _load_pnt_dict 
from astrohack._utils._io import _extract_holog_chunk
from astrohack._utils._io import _open_no_dask_zarr
from astrohack._utils._io import _read_data_from_holog_json
from astrohack._utils._io import _read_meta_data
from astrohack._utils._io import _load_holog_file
from astrohack._utils._io import _load_image_file
from astrohack._utils._io import _load_panel_file

from memory_profiler import profile

class AstrohackDataFile:
    def __init__(self, file_stem, path='./'):
                        
        self._image_path = None
        self._holog_path = None

        self.holog = None
        self.image = None
            
        self._verify_holog_files(file_stem, path)
            

    def _verify_holog_files(self, file_stem, path):
        console.info("Verifying {stem}.* files in path={path} ...".format(stem=file_stem, path=path))

        file_path = "{path}/{stem}.holog.zarr".format(path=path, stem=file_stem)
            
        if os.path.isdir(file_path):
            console.info("Found {stem}.holog.zarr directory ...".format(stem=file_stem))
            self._holog_path = file_path
            self.holog = AstrohackHologFile(file_path)
                

        file_path = "{path}/{stem}.image.zarr".format(path=path, stem=file_stem)

        if os.path.isdir(file_path):
            console.info("Found {stem}.image.zarr directory ...".format(stem=file_stem))
            self._image_path = file_path
            self.image = AstrohackImageFile(file_path)

        file_path = "{path}/{stem}.panel.zarr".format(path=path, stem=file_stem)

        if os.path.isdir(file_path):
            console.info("Found {stem}.panel.zarr directory ...".format(stem=file_stem))
            self._image_path = file_path
            self.panel = AstrohackPanelFile(file_path)

class AstrohackImageFile(dict):
    """
        Data class for holography image data.
    """
    def __init__(self, file):
        super().__init__()

        self.file = file
        self._open = False

    def __getitem__(self, key):
        return super().__getitem__(key)
    
    def __setitem__(self, key, value):
        return super().__setitem__(key, value)
        
    def is_open(self):
        return self._open

    def open(self, file=None):
        """ Open hologgraphy file.

        Args:self =_
            file (str, optional): Path to holography file. Defaults to None.

        Returns:
            bool: bool describing whether the file was opened properly
        """

        if file is None:
            file = self.file
        
        try:
            _load_image_file(file, image_dict=self)

            self._open = True

        except Exception as e:
            console.error("[AstroHackImageFile.open()]: {}".format(e))
            self._open = False

        return self._open

    def summary(self):
        """
           Prints summary table of holog image file. 
        """

        table = PrettyTable()
        table.field_names = ["antenna", "ddi"]
        table.align = "l"
        
        for ant in self.keys():
            table.add_row([ant, list(self[int(ant)].keys())])
        
        print(table)


    def select(self, ant=None, ddi=None, polar=False):
        """Select data on the basis of ddi, scan, ant. This is a convenience function.

        Args:
            ddi (int, optional): Data description ID. Defaults to None.
            ant (int, optional): Antenna ID. Defaults to None.

        Returns:
            xarray.Dataset: xarray dataset of corresponding ddi, scan, antenna ID.
        """
        if ant is None and ddi is None:
            console.info("No selections made ...")
            return self
        else:
            if polar:
                return self[ant][ddi].apply(np.absolute), self[ant][ddi].apply(np.angle, deg=True)

            return self[ant][ddi]

class AstrohackHologFile(dict):
    """
        Data Class to interact ith holography imaging data.
    """
    def __init__(self, file):
        super().__init__()
        
        self.file = file
        self._meta_data = None
        self._open = False


    def __getitem__(self, key):
        return super().__getitem__(key)
    
    def __setitem__(self, key, value):
        return super().__setitem__(key, value)

    def is_open(self):
        return self._open

    def open(self, file=None, dask_load=False):
        """ Open hologgraphy file.

        Args:self =_
            file (str, optional): Path to holography file. Defaults to None.
            dask_load (bool, optional): If True the file is loaded with Dask. Defaults to False.

        Returns:
            bool: bool describing whether the file was opened properly
        """

        if file is None:
            file = self.file

        self._meta_data = _read_meta_data(holog_file=file)

        try:
            _load_holog_file(holog_file=file, dask_load=dask_load, load_pnt_dict=False, holog_dict=self)
            self._open = True

        except Exception as e:
            console.error("[AstrohackHologFile]: {}".format(e))
            self._open = False
        
        return self._open

    def summary(self):
        """
            Prints summary table of holog file.
        """

        table = PrettyTable()
        table.field_names = ["ddi", "scan", "antenna"]
        table.align = "l"
        
        for ddi in self.keys():
            for scan in self[ddi].keys():
                table.add_row([ddi, scan, list(self[ddi][scan].keys())])
        
        print(table)

    def select(self, ddi=None, scan=None, ant=None):
        """ Select data on the basis of ddi, scan, ant. This is a convenience function.

        Args:
            ddi (int, optional): Data description ID. Defaults to None.
            scan (int, optional): Scan number. Defaults to None.
            ant (int, optional): Antenna ID. Defaults to None.

        Returns:
            xarray.Dataset: xarray dataset of corresponding ddi, scan, antenna ID.
        """
        if ant is None or ddi is None or scan is None:
            console.info("No selections made ...")
            return self
        else:
            return self[ddi][scan][ant]

    @property
    def meta_data(self):
        """ Holog file meta data.

        Returns:
            JSON: JSON file of holography meta data.
        """

        return self._meta_data

class AstrohackPanelFile(dict):
    """
        Data class for holography panel data.
    """
    def __init__(self, file):
        super().__init__()

        self.file = file
        self._open = False

    def __getitem__(self, key):
        return super().__getitem__(key)
    
    def __setitem__(self, key, value):
        return super().__setitem__(key, value)
        
    def is_open(self):
        return self._open

    def open(self, file=None):
        """ Open panel file.

        Args:self =_
            file (str, optional): Path to holography file. Defaults to None.

        Returns:
            bool: bool describing whether the file was opened properly
        """

        if file is None:
            file = self.file
        
        try:
            _load_panel_file(file, panel_dict=self)

            self._open = True

        except Exception as e:
            console.error("[AstroHackPanelFile.open()]: {}".format(e))
            self._open = False

        return self._open

#    def summary(self):
#        """
#           Prints summary table of panel image file. 
#        """

#        table = PrettyTable()
#        table.field_names = ["antenna", "ddi"]
#        table.align = "l"
        
#        for ant in self.keys():
#            table.add_row([ant, list(self[int(ant)].keys())])
        
#        print(table)