# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['switchbot_client', 'switchbot_client.devices', 'switchbot_client.scenes']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=5.4.1,<7.0.0', 'requests>=2.0,<3.0', 'typing-extensions>=3.10,<5.0']

setup_kwargs = {
    'name': 'switchbot-client',
    'version': '0.3.1',
    'description': 'A Python client library for SwitchBot API.',
    'long_description': '# switchbot-client\n\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/switchbot-client.svg)](https://pypi.org/project/switchbot-client/)\n[![PyPI - Library Version](https://img.shields.io/pypi/v/switchbot-client.svg)](https://pypi.org/project/switchbot-client/)\n[![PyPI - Downloads](https://img.shields.io/pypi/dm/switchbot-client)](https://pypi.org/project/switchbot-client)\n[![License](https://img.shields.io/badge/license-MIT%2FApache--2.0-informational?style=flat-square)](README.md#License)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\nAn unofficial Python client implementation of the SwitchBot API.\n\n## Table of Contents\n\n- [Authentication](#authentication)\n- [Usage](#usage)\n- [License](#license)\n\n\n## Authentication\n\nBefore you start using this client, you need to get an open token.\nPlease follow the instructions in the official documentation below.\n\nhttps://github.com/OpenWonderLabs/SwitchBotAPI#authentication\n\nOnce you have the token, use one of the following methods to pass the information to the client.\n\n### Environment variables\n\nIf the environment variable `SWITCHBOT_OPEN_TOKEN` is present, \nthis client will automatically use this value.\n\n```shell\nexport SWITCHBOT_OPEN_TOKEN=your_switchbot_open_token\npython3 your_script.py\n```\n\n```python\n# your_script.py\nfrom switchbot_client import SwitchBotClient\n\nclient = SwitchBotClient()\nprint(client.devices())\n```\n\n### Constructor Arguments\n\nIt is also possible to initialize the client by passing a token directly as an argument.\n\n```python\nfrom switchbot_client import SwitchBotClient\n\nyour_token = "your_switchbot_open_token"\nclient = SwitchBotClient(token=your_token)\nprint(client.devices())\n```\n\n### Config file\n\nIf `~/.config/switchbot-client/config.yml` exists and has a `token` entry, \nthis client will automatically use the value.\n\n```shell\nmkdir -p ~/.config/switchbot-client\necho "token: your_switchbot_open_token" >>  ~/.config/switchbot-client/config.yml\npython3 your_script.py\n```\n\n```python\n# your_script.py\nfrom switchbot_client import SwitchBotClient\n\nclient = SwitchBotClient()\nprint(client.devices())\n```\n\n## Usage\n\n### Get Device List\n\n```python\nfrom switchbot_client import SwitchBotClient\n\nclient = SwitchBotClient()\nresult = client.devices()\nprint(result)\n```\n\n```\n[Meter({\'device_id\': \'ABCDEFG\', \'device_type\': \'Meter\', \'device_name\': \'Meter 0A\', \'hub_device_id\': \'ABCDE\', \'is_virtual_infrared\': False}), HubMini({\'device_id\': \'ABCDEFG\', \'device_type\': \'Hub Mini\', \'device_name\': \'Hub Mini 0\', \'hub_device_id\': None, \'is_virtual_infrared\': False}), Light({\'device_id\': \'12345\', \'device_type\': \'Light\', \'device_name\': \'My Light\', \'hub_device_id\': \'ABCDE\', \'is_virtual_infrared\': True}), AirConditioner({\'device_id\': \'12345\', \'device_type\': \'Air Conditioner\', \'device_name\': \'My Air Conditioner\', \'hub_device_id\': \'ABCDE\', \'is_virtual_infrared\': True})]\n```\n\nIf you run the above code, you will get a list of all the devices associated with your SwitchBot account. \nYou can perform operations on the acquired `device_id`, such as manipulating it or getting its status.\n\n### Get Device Status\n\n```python\nfrom switchbot_client import SwitchBotClient\n\nclient = SwitchBotClient()\ndevice_id = "YOUR_DEVICE_ID"\nprint(client.device(device_id).status())\n```\n\n```\nDeviceStatus(device_id=\'ABCDE\', device_type=\'Meter\', device_name=\'Meter 0\', hub_device_id=\'ABCDE\', data={\'deviceId\': \'ABCDE\', \'deviceType\': \'Meter\', \'hubDeviceId\': \'ABCDE\', \'humidity\': 50, \'temperature\': 25.0})\n```\n\nThis function allows you to get the status of a device.\nNote that only physical devices can be targeted, not virtual infrared remote devices.\n\nPlease refer to the following official document to know what kind of information can be obtained from each device.\n\nhttps://github.com/OpenWonderLabs/SwitchBotAPI#get-device-status\n\n### Control Device\n\n```python\nfrom switchbot_client import SwitchBotClient\nfrom switchbot_client.devices import Light\n\nclient = SwitchBotClient()\ndevice_id = "12345"  # My Light(virtual infrared remote devices)\ndevice = Light.create_by_id(client, device_id)\nprint(device.turn_on())\n```\n\n```\nSwitchBotCommandResult(status_code=100, message=\'success\', response_body={})\n```\n\nIt allows you to control the specified device.\nThe following documents define the commands that can be executed.\n\nhttps://github.com/OpenWonderLabs/SwitchBotAPI#send-device-control-commands\n\n### Get Scene List\n\n```python\nfrom switchbot_client import SwitchBotClient\n\nclient = SwitchBotClient()\nprint(client.scenes())\n```\n\n```\n[SwitchBotScene({\'scene_id\': \'12345\', \'scene_name\': \'My Scene1\'}), SwitchBotScene({\'scene_id\': \'23456\', \'scene_name\': \'My Scene2\'})]\n```\n\nYou can get a list of all the scenes associated with your SwitchBot account.\nNote that only manual scenes are returned from this api.\n\n### Execute Scene\n```python\nfrom switchbot_client import SwitchBotClient\n\nclient = SwitchBotClient()\nprint(client.scene("12345").execute())\n```\n\n```\nSwitchBotCommandResult(status_code=100, message=\'success\', response_body={})\n```\nThe specified scene can be executed immediately.\n\n### Raw API interface\n\nDevices and scenes also can be manipulated via the low-level raw API client.\nThe `SwitchBotAPIClient` class has methods for each endpoints of SwitchBot API.\n\nFor example the `/v1.0/devices` endpoint is implemented as `SwitchBotAPIClient.devices()`, \nthe `/v1.0/devices/{device_id}/status"` endpoint is implemented as `SwitchBotAPIClient.devices_status(device_id: str)`.\n\n\n### Examples\n\n```python\nfrom switchbot_client import devices\nfrom switchbot_client import SwitchBotClient\n\n\ndef call_this_function_when_i_go_out():\n    client = SwitchBotClient()\n    print("turn off all lights and air conditioners...")\n    for d in client.devices():\n        if isinstance(d, devices.Light):\n            d.turn_off()\n\n        if isinstance(d, devices.ColorBulb):\n            d.turn_off()\n\n        if isinstance(d, devices.AirConditioner):\n            d.turn_off()\n    print("done")\n\n\ndef control_devices_by_temperature():\n    client = SwitchBotClient()\n    all_devices = client.devices()\n\n    temperatures = [d.temperature() for d in all_devices if isinstance(d, devices.Meter)]\n    temperature = min(temperatures)\n\n    color_bulbs = [d for d in all_devices if isinstance(d, devices.ColorBulb)]\n    air_conditioners = [d for d in all_devices if isinstance(d, devices.AirConditioner)]\n\n    if temperature > 25.0:\n        print("hot!")\n        for d in color_bulbs:\n            d.set_color("#FF0000")\n\n        for d in air_conditioners:\n            d.set_all(\n                temperature=20.0,\n                mode=devices.AirConditioner.Parameters.MODE_COOL,\n                fan_speed=devices.AirConditioner.Parameters.FAN_SPEED_HIGH,\n                power=devices.AirConditioner.Parameters.POWER_ON\n            )\n\n    elif temperature < 15.0:\n        print("cold!")\n        for d in color_bulbs:\n            d.set_color("#0000FF")\n\n        for d in air_conditioners:\n            d.set_all(\n                temperature=20.0,\n                mode=devices.AirConditioner.Parameters.MODE_HEAT,\n                fan_speed=devices.AirConditioner.Parameters.FAN_SPEED_HIGH,\n                power=devices.AirConditioner.Parameters.POWER_ON\n            )\n```\n\n## License\n\nLicensed under either of\n\n- Apache License, Version 2.0\n   ([LICENSE-APACHE](LICENSE-APACHE) or http://www.apache.org/licenses/LICENSE-2.0)\n- MIT license\n   ([LICENSE-MIT](LICENSE-MIT) or http://opensource.org/licenses/MIT)\n\nat your option.\n\n## Contribution\n\nUnless you explicitly state otherwise, any contribution intentionally submitted\nfor inclusion in the work by you, as defined in the Apache-2.0 license, shall be\ndual licensed as above, without any additional terms or conditions.\n',
    'author': 'Kazuhito Osabe',
    'author_email': 'kzosabe@wip.place',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kzosabe/switchbot-client',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
