import morecantile
import pytest
from morecantile import Tile

# WebMercatorQuad TMS for creating tiles
WEBMERC_TMS = morecantile.tms.get("WebMercatorQuad")
# WorldMercatorWGS84Quad TMS for WGS84 Mercator tiles
WGS84_TMS = morecantile.tms.get("WorldMercatorWGS84Quad")
# EuropeanETRS89_LAEAQuad TMS for ETRS89 LAEA CRS
ETRS89_TMS = morecantile.tms.get("EuropeanETRS89_LAEAQuad")

# WebMercator tiles - regular cases
WEBMERC_TILES_REGULAR = [
    # WebMercatorQuad tiles - European region focus to avoid anti-meridian issues
    pytest.param(Tile(x=2, y=1, z=2), WEBMERC_TMS, id="webmerc_europe_center(2/2/1)"),
    pytest.param(Tile(x=1, y=1, z=2), WEBMERC_TMS, id="webmerc_europe_west(2/1/1)"),
    pytest.param(Tile(x=0, y=0, z=5), WEBMERC_TMS, id="webmerc_europe_south(5/0/0)"),
    # Note: webmerc_europe_east(2/3/1) removed - causes anti-meridian crossing when projected to ETRS89 LAEA
    pytest.param(Tile(x=2, y=0, z=2), WEBMERC_TMS, id="webmerc_europe_north(2/2/0)"),
    pytest.param(Tile(x=2, y=2, z=2), WEBMERC_TMS, id="webmerc_europe_south(2/2/2)"),
    # Higher zoom European region
    pytest.param(Tile(x=8, y=5, z=4), WEBMERC_TMS, id="webmerc_europe_zoom4(4/8/5)"),
    pytest.param(Tile(x=16, y=10, z=5), WEBMERC_TMS, id="webmerc_europe_zoom5(5/16/10)"),
    # Small bbox test
    pytest.param(Tile(x=8, y=8, z=5), WEBMERC_TMS, id="webmerc_small_bbox(5/8/8)"),
    # Additional anti-meridian tiles
    pytest.param(
        Tile(x=0, y=2, z=3), WEBMERC_TMS, id="webmerc_antimeridian_z3_west(3/0/2)"
    ),
    pytest.param(
        Tile(x=7, y=2, z=3), WEBMERC_TMS, id="webmerc_antimeridian_z3_east(3/7/2)"
    ),
    pytest.param(
        Tile(x=31, y=10, z=5), WEBMERC_TMS, id="webmerc_antimeridian_z5_east(5/31/10)"
    ),
    pytest.param(
        Tile(x=0, y=10, z=5), WEBMERC_TMS, id="webmerc_antimeridian_z5_west(5/0/10)"
    ),
    # Additional prime meridian tiles
    pytest.param(Tile(x=4, y=2, z=3), WEBMERC_TMS, id="webmerc_prime_meridian_z3(3/4/2)"),
    pytest.param(
        Tile(x=16, y=10, z=5), WEBMERC_TMS, id="webmerc_prime_meridian_z5(5/16/10)"
    ),
    pytest.param(Tile(x=15, y=10, z=5), WEBMERC_TMS, id="webmerc_prime_west_z5(5/15/10)"),
    pytest.param(Tile(x=17, y=10, z=5), WEBMERC_TMS, id="webmerc_prime_east_z5(5/17/10)"),
    # Additional equator tiles
    pytest.param(Tile(x=1, y=1, z=2), WEBMERC_TMS, id="webmerc_equator_north(2/1/1)"),
    pytest.param(Tile(x=0, y=2, z=2), WEBMERC_TMS, id="webmerc_equator_west(2/0/2)"),
    pytest.param(Tile(x=3, y=2, z=2), WEBMERC_TMS, id="webmerc_equator_east(2/3/2)"),
    pytest.param(Tile(x=4, y=4, z=3), WEBMERC_TMS, id="webmerc_equator_z3_south(3/4/4)"),
    pytest.param(Tile(x=4, y=3, z=3), WEBMERC_TMS, id="webmerc_equator_z3_north(3/4/3)"),
    pytest.param(
        Tile(x=16, y=16, z=5), WEBMERC_TMS, id="webmerc_equator_z5_south(5/16/16)"
    ),
    pytest.param(
        Tile(x=16, y=15, z=5), WEBMERC_TMS, id="webmerc_equator_z5_north(5/16/15)"
    ),
    pytest.param(
        Tile(x=31, y=16, z=5),
        WEBMERC_TMS,
        id="webmerc_equator_antimeridian_east(5/31/16)",
    ),
]

# WebMercator tiles - edge cases for integration tests (max 5)
WEBMERC_TILES_EDGE_CASES = [
    # Anti-meridian (180/-180 degrees) problematic tiles
    pytest.param(Tile(x=0, y=1, z=2), WEBMERC_TMS, id="webmerc_antimeridian_west(2/0/1)"),
    pytest.param(Tile(x=3, y=1, z=2), WEBMERC_TMS, id="webmerc_antimeridian_east(2/3/1)"),
    # Prime meridian (0 degrees) problematic tiles
    pytest.param(Tile(x=2, y=1, z=2), WEBMERC_TMS, id="webmerc_prime_meridian(2/2/1)"),
    # Equator (0 degrees latitude) tiles
    pytest.param(Tile(x=1, y=2, z=2), WEBMERC_TMS, id="webmerc_equator_south(2/1/2)"),
    # Equator at anti-meridian - most complex edge case
    pytest.param(
        Tile(x=0, y=16, z=5), WEBMERC_TMS, id="webmerc_equator_antimeridian_west(5/0/16)"
    ),
]

# WebMercator tiles (supported TMS) - combined
WEBMERC_TILES = WEBMERC_TILES_REGULAR + WEBMERC_TILES_EDGE_CASES

# ETRS89 tiles - regular cases
ETRS89_TILES_REGULAR = [
    # ETRS89 LAEA tiles - European region specific
    # Center of Europe tiles
    pytest.param(Tile(x=1, y=1, z=2), ETRS89_TMS, id="etrs89_center_europe(2/1/1)"),
    pytest.param(Tile(x=0, y=1, z=2), ETRS89_TMS, id="etrs89_west_europe(2/0/1)"),
    pytest.param(Tile(x=2, y=1, z=2), ETRS89_TMS, id="etrs89_east_europe(2/2/1)"),
    # Northern Europe (Scandinavia region)
    pytest.param(Tile(x=1, y=0, z=2), ETRS89_TMS, id="etrs89_north_europe(2/1/0)"),
    # Southern Europe (Mediterranean region)
    pytest.param(Tile(x=1, y=2, z=2), ETRS89_TMS, id="etrs89_south_europe(2/1/2)"),
    # Higher zoom cases within Europe
    pytest.param(Tile(x=2, y=2, z=3), ETRS89_TMS, id="etrs89_central_zoom3(3/2/2)"),
]

# ETRS89 tiles - edge cases for integration tests (max 5)
ETRS89_TILES_EDGE_CASES = [
    # Higher zoom edge cases within Europe
    pytest.param(Tile(x=4, y=4, z=4), ETRS89_TMS, id="etrs89_central_zoom4(4/4/4)"),
    pytest.param(Tile(x=6, y=6, z=4), ETRS89_TMS, id="etrs89_southeast_zoom4(4/6/6)"),
    # Small bbox test for ETRS89
    pytest.param(Tile(x=8, y=8, z=5), ETRS89_TMS, id="etrs89_small_bbox(5/8/8)"),
]

# ETRS89 tiles (some may not be supported) - combined
ETRS89_TILES = ETRS89_TILES_REGULAR + ETRS89_TILES_EDGE_CASES

# WGS84 tiles - regular cases
WGS84_TILES_REGULAR = [
    # Equator tiles for testing WGS84 coordinate handling
    # At z=2, equator is between y=1 and y=2 in WGS84 projection
    pytest.param(Tile(x=1, y=1, z=2), WGS84_TMS, id="wgs84_equator_north(2/1/1)"),
    pytest.param(Tile(x=1, y=2, z=2), WGS84_TMS, id="wgs84_equator_south(2/1/2)"),
    # Prime meridian (0 degrees longitude) tiles
    pytest.param(Tile(x=2, y=1, z=2), WGS84_TMS, id="wgs84_prime_meridian(2/2/1)"),
    pytest.param(Tile(x=1, y=1, z=2), WGS84_TMS, id="wgs84_prime_west(2/1/1)"),
    pytest.param(Tile(x=3, y=1, z=2), WGS84_TMS, id="wgs84_prime_east(2/3/1)"),
    # Anti-meridian tiles (180/-180 degrees longitude)
    # At z=2, x=0 covers western edge near anti-meridian
    pytest.param(Tile(x=0, y=1, z=2), WGS84_TMS, id="wgs84_antimeridian_west(2/0/1)"),
]

# WGS84 tiles - edge cases for integration tests (max 5)
WGS84_TILES_EDGE_CASES = [
    pytest.param(
        Tile(x=0, y=2, z=2), WGS84_TMS, id="wgs84_antimeridian_west_equator(2/0/2)"
    ),
    # Equator at anti-meridian - key test case for coordinate transformation
    pytest.param(
        Tile(x=0, y=1, z=3), WGS84_TMS, id="wgs84_equator_antimeridian_z3(3/0/1)"
    ),
    pytest.param(
        Tile(x=0, y=2, z=3), WGS84_TMS, id="wgs84_equator_antimeridian_south_z3(3/0/2)"
    ),
    # Higher zoom equator anti-meridian tiles
    pytest.param(
        Tile(x=0, y=15, z=5), WGS84_TMS, id="wgs84_equator_antimeridian_north_z5(5/0/15)"
    ),
    pytest.param(
        Tile(x=0, y=16, z=5), WGS84_TMS, id="wgs84_equator_antimeridian_south_z5(5/0/16)"
    ),
]

# WGS84 tiles for equator anti-meridian and 0 longitude testing - combined
WGS84_TILES = WGS84_TILES_REGULAR + WGS84_TILES_EDGE_CASES

# HRRR tiles - regular cases
# Actual HRRR domain: Lat(21.14, 47.84), Lon(-134.10, -60.92)
HRRR_TILES_REGULAR = [
    # Low zoom - full domain coverage
    pytest.param(Tile(x=0, y=1, z=2), WEBMERC_TMS, id="hrrr_west_z2(2/0/1)"),
    pytest.param(Tile(x=1, y=1, z=2), WEBMERC_TMS, id="hrrr_east_z2(2/1/1)"),
    # Medium zoom - domain corners and edges
    pytest.param(Tile(x=1, y=2, z=3), WEBMERC_TMS, id="hrrr_sw_corner_z3(3/1/2)"),
    pytest.param(Tile(x=2, y=2, z=3), WEBMERC_TMS, id="hrrr_se_corner_z3(3/2/2)"),
    pytest.param(Tile(x=1, y=2, z=3), WEBMERC_TMS, id="hrrr_nw_corner_z3(3/1/2)"),
    pytest.param(Tile(x=2, y=2, z=3), WEBMERC_TMS, id="hrrr_ne_corner_z3(3/2/2)"),
    pytest.param(Tile(x=1, y=3, z=3), WEBMERC_TMS, id="hrrr_south_z3(3/1/3)"),
    # Higher zoom - precise domain coverage
    pytest.param(Tile(x=4, y=11, z=5), WEBMERC_TMS, id="hrrr_west_edge_z5(5/4/11)"),
    pytest.param(Tile(x=10, y=11, z=5), WEBMERC_TMS, id="hrrr_east_edge_z5(5/10/11)"),
    pytest.param(Tile(x=4, y=13, z=5), WEBMERC_TMS, id="hrrr_south_edge_z5(5/4/13)"),
    pytest.param(Tile(x=7, y=11, z=5), WEBMERC_TMS, id="hrrr_center_z5(5/7/11)"),
    # Very high zoom cases
    pytest.param(Tile(x=16, y=44, z=7), WEBMERC_TMS, id="hrrr_sw_precise_z7(7/16/44)"),
    pytest.param(Tile(x=42, y=44, z=7), WEBMERC_TMS, id="hrrr_se_precise_z7(7/42/44)"),
    pytest.param(Tile(x=29, y=50, z=7), WEBMERC_TMS, id="hrrr_center_z7(7/29/50)"),
]

# HRRR tiles - edge cases for integration tests (max 5)
HRRR_TILES_EDGE_CASES = [
    # Ultra high zoom - precise boundaries (edge cases)
    pytest.param(
        Tile(x=130, y=356, z=10), WEBMERC_TMS, id="hrrr_sw_extreme_z10(10/130/356)"
    ),
    pytest.param(
        Tile(x=338, y=356, z=10), WEBMERC_TMS, id="hrrr_se_extreme_z10(10/338/356)"
    ),
    pytest.param(Tile(x=234, y=403, z=10), WEBMERC_TMS, id="hrrr_center_z10(10/234/403)"),
]

# HRRR tiles for testing Lambert Conformal Conic projection data - combined
HRRR_TILES = HRRR_TILES_REGULAR + HRRR_TILES_EDGE_CASES

# Para tiles - regular cases
# Para is approximately between 2.72°N to 9.93°S and 45.97°W to 58.99°W
PARA_TILES_REGULAR = [
    # Zoom level 4 - broader coverage
    pytest.param(Tile(x=5, y=7, z=4), WEBMERC_TMS, id="para_north_z4(4/5/7)"),
    pytest.param(Tile(x=5, y=8, z=4), WEBMERC_TMS, id="para_south_z4(4/5/8)"),
    # Zoom level 5 - more detailed coverage
    pytest.param(Tile(x=10, y=15, z=5), WEBMERC_TMS, id="para_northwest_z5(5/10/15)"),
    pytest.param(Tile(x=11, y=15, z=5), WEBMERC_TMS, id="para_northeast_z5(5/11/15)"),
    pytest.param(Tile(x=10, y=16, z=5), WEBMERC_TMS, id="para_southwest_z5(5/10/16)"),
    pytest.param(Tile(x=11, y=16, z=5), WEBMERC_TMS, id="para_southeast_z5(5/11/16)"),
    # Zoom level 6 - covering Belém (capital) area at ~1.5°S, 48.5°W
    pytest.param(Tile(x=22, y=31, z=6), WEBMERC_TMS, id="para_belem_z6(6/22/31)"),
    # Zoom level 7 - detailed view
    pytest.param(Tile(x=44, y=63, z=7), WEBMERC_TMS, id="para_north_z7(7/44/63)"),
    pytest.param(Tile(x=45, y=64, z=7), WEBMERC_TMS, id="para_central_z7(7/45/64)"),
    # Zoom level 8 - high detail for southern Para
    pytest.param(Tile(x=88, y=128, z=8), WEBMERC_TMS, id="para_south_z8(8/88/128)"),
]

# Para tiles - edge cases for integration tests (max 5)
PARA_TILES_EDGE_CASES = [
    # test upsampling at very high zoom levels - true edge cases
    pytest.param(
        Tile(x=1480, y=2064, z=12), WEBMERC_TMS, id="para_south_z8(12/1480/2064)"
    ),
    pytest.param(
        Tile(x=2964, y=4129, z=13), WEBMERC_TMS, id="para_south_z8(13/2964/4129)"
    ),
    pytest.param(
        Tile(x=5971, y=8252, z=14), WEBMERC_TMS, id="para_south_z8(14/5971/8252)"
    ),
]

# Para (Brazilian state) tiles for testing South American region - combined
PARA_TILES = PARA_TILES_REGULAR + PARA_TILES_EDGE_CASES

TILES = WEBMERC_TILES + WGS84_TILES + ETRS89_TILES
