#!/usr/bin/env python3
"""
Simple chat example using LlamaForge.
"""

import sys
import os
from pathlib import Path

# Add parent directory to path to allow imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from llamaforge import LlamaForge

def main():
    """Run a simple chat example."""
    print("LlamaForge Simple Chat Example")
    print("==============================")
    
    # Initialize LlamaForge
    llamaforge = LlamaForge()
    
    # Get available models
    models = llamaforge.model_manager.list_models()
    
    if not models:
        print("No models available. Please add a model first.")
        print("Example: llamaforge model add --name TheBloke/Llama-2-7B-Chat-GGUF")
        return
    
    # Select the first model
    model_name = models[0]["name"]
    print(f"Using model: {model_name}")
    
    # Start chat loop
    print("\nType 'exit' or 'quit' to end the conversation.")
    print("Type 'model <name>' to switch models.")
    print("Type 'help' for more commands.")
    print("-------------------------------")
    
    while True:
        # Get user input
        user_input = input("\nYou: ")
        
        # Check for exit command
        if user_input.lower() in ["exit", "quit"]:
            print("Goodbye!")
            break
        
        # Check for model switch command
        if user_input.lower().startswith("model "):
            new_model = user_input[6:].strip()
            model_names = [m["name"] for m in models]
            if new_model in model_names:
                model_name = new_model
                print(f"Switched to model: {model_name}")
            else:
                print(f"Model '{new_model}' not found. Available models: {', '.join(model_names)}")
            continue
        
        # Check for help command
        if user_input.lower() == "help":
            print("Available commands:")
            print("  exit, quit - End the conversation")
            print("  model <name> - Switch to a different model")
            print("  help - Show this help message")
            continue
        
        # Generate response
        print("\nAssistant: ", end="", flush=True)
        
        # In a real implementation, this would call the model
        # Here we just simulate a response
        response = f"This is a simulated response from {model_name}. In a real implementation, this would be generated by the model."
        
        # Print response token by token to simulate streaming
        for char in response:
            print(char, end="", flush=True)
            import time
            time.sleep(0.01)
        print()

if __name__ == "__main__":
    main() 