# Plugins API Reference

LlamaForge's plugin system allows you to extend functionality with preprocessing, postprocessing, tools, and commands. This reference explains how to use and create custom plugins.

## Plugin Types

LlamaForge supports four main types of plugins:

1. **Preprocessing Plugins**: Modify input text before it's sent to the model
2. **Postprocessing Plugins**: Modify output text after it's generated by the model
3. **Tool Plugins**: Provide external tools that models can use
4. **Command Plugins**: Add new commands to the CLI

## BasePlugin Class

All plugins inherit from the `BasePlugin` abstract class, which defines the common interface:

```python
from llamaforge.plugins.base import BasePlugin

class BasePlugin:
    """Base class for all plugins."""
    
    @property
    def name(self):
        """Return the name of the plugin."""
        raise NotImplementedError
    
    @property
    def description(self):
        """Return a description of the plugin."""
        raise NotImplementedError
    
    def process(self, *args, **kwargs):
        """Process inputs or perform actions."""
        raise NotImplementedError
```

## Using Plugins

Plugins are automatically loaded and managed by the `LlamaForge` class:

```python
from llamaforge import LlamaForge

forge = LlamaForge()

# Check available plugins
print(forge.list_plugins())

# Enable a specific plugin
forge.enable_plugin("text_formatter")

# Disable a plugin
forge.disable_plugin("calculator")

# Check if a plugin is enabled
print(forge.is_plugin_enabled("text_formatter"))  # True or False
```

## Preprocessing Plugins

Preprocessing plugins modify input text before it's sent to the model.

### TextFormatterPlugin

A built-in plugin for formatting input text.

```python
from llamaforge import LlamaForge

forge = LlamaForge()
forge.enable_plugin("text_formatter")

# Configure plugin parameters
forge.set_plugin_config("text_formatter", {
    "trim_whitespace": True,
    "add_system_instruction": True,
    "system_instruction": "You are a helpful assistant.",
    "format_as_chat": True,
    "chat_format": "llama2",
    "template": None
})

# Generate text (text_formatter will preprocess the input)
response = forge.generate("What is quantum computing?")
```

### Creating a Custom Preprocessing Plugin

```python
from llamaforge.plugins.base import BasePlugin

class MyPreprocessingPlugin(BasePlugin):
    """A custom preprocessing plugin."""
    
    @property
    def name(self):
        return "my_preprocessor"
    
    @property
    def description(self):
        return "My custom preprocessing plugin"
    
    def process(self, text, **kwargs):
        """Process input text before it's sent to the model."""
        # Do something with the text
        text = text.replace("[DATE]", "2023-07-01")
        text = text.replace("[NAME]", "User")
        
        return text
```

### Registering a Custom Preprocessing Plugin

```python
from llamaforge.plugins.preprocessing import register_preprocessor

# Register the plugin
register_preprocessor("my_preprocessor", MyPreprocessingPlugin)

# Then use it
forge = LlamaForge()
forge.enable_plugin("my_preprocessor")
```

## Postprocessing Plugins

Postprocessing plugins modify the output text after it's generated by the model.

### TextCleanerPlugin

A built-in plugin for cleaning model output.

```python
from llamaforge import LlamaForge

forge = LlamaForge()
forge.enable_plugin("text_cleaner")

# Configure plugin parameters
forge.set_plugin_config("text_cleaner", {
    "trim_whitespace": True,
    "remove_special_tokens": True,
    "remove_incomplete_sentences": True,
    "special_token_patterns": [
        "<\\|im_end\\|>.*",
        "<s>",
        "</s>"
    ]
})

# Generate text (text_cleaner will postprocess the output)
response = forge.generate("Write a short story about a robot.")
```

### Creating a Custom Postprocessing Plugin

```python
from llamaforge.plugins.base import BasePlugin

class MyPostprocessingPlugin(BasePlugin):
    """A custom postprocessing plugin."""
    
    @property
    def name(self):
        return "my_postprocessor"
    
    @property
    def description(self):
        return "My custom postprocessing plugin"
    
    @property
    def supports_streaming(self):
        """Whether this plugin supports streaming."""
        return False
    
    def process(self, text, **kwargs):
        """Process output text after it's generated by the model."""
        # Do something with the text
        text = text.replace("robot", "android")
        text = text.replace("computer", "electronic brain")
        
        return text
```

### Registering a Custom Postprocessing Plugin

```python
from llamaforge.plugins.postprocessing import register_postprocessor

# Register the plugin
register_postprocessor("my_postprocessor", MyPostprocessingPlugin)

# Then use it
forge = LlamaForge()
forge.enable_plugin("my_postprocessor")
```

## Tool Plugins

Tool plugins provide external functionality that models can use, similar to function calling.

### CalculatorPlugin

A built-in plugin that provides calculator functionality.

```python
from llamaforge import LlamaForge

forge = LlamaForge()
forge.enable_plugin("calculator")

# The model can now use the calculator plugin when needed
response = forge.generate(
    "Calculate the square root of 144 and then multiply it by 2."
)
```

### Creating a Custom Tool Plugin

```python
import requests
from llamaforge.plugins.base import BasePlugin

class WeatherPlugin(BasePlugin):
    """A custom tool plugin for weather data."""
    
    @property
    def name(self):
        return "weather"
    
    @property
    def description(self):
        return "Get current weather data for a location"
    
    @property
    def parameters(self):
        """Define parameters for the tool."""
        return {
            "type": "object",
            "properties": {
                "location": {
                    "type": "string",
                    "description": "The city and state/country"
                }
            },
            "required": ["location"]
        }
    
    def process(self, location, **kwargs):
        """Get weather data for the given location."""
        # Hypothetical weather API call
        api_key = kwargs.get("api_key", "default_key")
        response = requests.get(
            f"https://api.weather.example.com/current",
            params={"q": location, "key": api_key}
        )
        
        if response.status_code == 200:
            data = response.json()
            return {
                "temperature": data["current"]["temp_c"],
                "condition": data["current"]["condition"]["text"],
                "humidity": data["current"]["humidity"],
                "wind_speed": data["current"]["wind_kph"]
            }
        else:
            return {"error": f"Failed to get weather data: {response.status_code}"}
```

### Registering a Custom Tool Plugin

```python
from llamaforge.plugins.tools import register_tool

# Register the plugin
register_tool("weather", WeatherPlugin)

# Then use it
forge = LlamaForge()
forge.enable_plugin("weather")
forge.set_plugin_config("weather", {"api_key": "your_weather_api_key"})
```

## Command Plugins

Command plugins add new commands to the LlamaForge CLI.

### BenchmarkPlugin

A built-in plugin for benchmarking model performance.

```python
# Using the benchmark command from the CLI
$ llamaforge benchmark --model llama-2-7b --dataset alpaca_eval --num_samples 100
```

```python
# Using the benchmark command programmatically
from llamaforge import LlamaForge

forge = LlamaForge()
forge.enable_plugin("benchmark")

# Run a benchmark
results = forge.run_command("benchmark", {
    "model": "llama-2-7b",
    "dataset": "alpaca_eval",
    "num_samples": 100
})

print(f"Average latency: {results['avg_latency']}s")
print(f"Tokens per second: {results['tokens_per_second']}")
```

### Creating a Custom Command Plugin

```python
import json
from llamaforge.plugins.base import BasePlugin

class ExportPlugin(BasePlugin):
    """A custom command plugin for exporting model responses."""
    
    @property
    def name(self):
        return "export"
    
    @property
    def description(self):
        return "Export model responses to a file"
    
    @property
    def command_args(self):
        """Define command-line arguments for the command."""
        return [
            {
                "name": "--model",
                "help": "Model to use for generation",
                "required": True
            },
            {
                "name": "--input-file",
                "help": "File containing prompts to process",
                "required": True
            },
            {
                "name": "--output-file",
                "help": "File to save responses",
                "required": True
            },
            {
                "name": "--format",
                "help": "Output format (json or text)",
                "default": "json"
            }
        ]
    
    def process(self, model, input_file, output_file, format="json", **kwargs):
        """Export model responses to a file."""
        # Get LlamaForge instance (typically passed in kwargs)
        forge = kwargs.get("llamaforge")
        if forge is None:
            return {"error": "LlamaForge instance not provided"}
        
        # Load the model
        if not forge.is_model_loaded(model):
            forge.load_model(model)
        
        # Read input prompts
        with open(input_file, "r") as f:
            prompts = [line.strip() for line in f if line.strip()]
        
        # Generate responses
        responses = []
        for prompt in prompts:
            response = forge.generate(prompt)
            responses.append({"prompt": prompt, "response": response})
        
        # Save responses
        if format == "json":
            with open(output_file, "w") as f:
                json.dump(responses, f, indent=2)
        else:
            with open(output_file, "w") as f:
                for item in responses:
                    f.write(f"PROMPT: {item['prompt']}\n")
                    f.write(f"RESPONSE: {item['response']}\n\n")
        
        return {
            "success": True,
            "count": len(responses),
            "output_file": output_file
        }
```

### Registering a Custom Command Plugin

```python
from llamaforge.plugins.commands import register_command

# Register the plugin
register_command("export", ExportPlugin)

# Then use it
forge = LlamaForge()
forge.enable_plugin("export")

# Run the command
result = forge.run_command("export", {
    "model": "llama-2-7b",
    "input_file": "prompts.txt",
    "output_file": "responses.json"
})

print(f"Exported {result['count']} responses to {result['output_file']}")
```

## Plugin Configuration

Plugins can be configured via the `LlamaForge` class:

```python
from llamaforge import LlamaForge

forge = LlamaForge()

# Set plugin configuration
forge.set_plugin_config("text_formatter", {
    "trim_whitespace": True,
    "add_system_instruction": True,
    "system_instruction": "You are a helpful AI assistant."
})

# Get plugin configuration
config = forge.get_plugin_config("text_formatter")
print(config["system_instruction"])

# Get all plugin configurations
all_configs = forge.get_all_plugin_configs()
```

## Plugin Hooks and Lifecycle

LlamaForge plugins can implement additional lifecycle hooks:

```python
from llamaforge.plugins.base import BasePlugin

class AdvancedPlugin(BasePlugin):
    """A plugin with lifecycle hooks."""
    
    @property
    def name(self):
        return "advanced_plugin"
    
    @property
    def description(self):
        return "An advanced plugin with lifecycle hooks"
    
    def initialize(self, **kwargs):
        """Called when the plugin is first enabled."""
        print(f"Plugin {self.name} initialized")
        # Perform setup actions
        return True
    
    def shutdown(self):
        """Called when the plugin is disabled."""
        print(f"Plugin {self.name} shutting down")
        # Perform cleanup actions
        return True
    
    def process(self, *args, **kwargs):
        """Standard processing method."""
        # Process inputs
        return "Processed by advanced plugin"
```

## Plugin Dependencies

Plugins can specify dependencies on other plugins:

```python
from llamaforge.plugins.base import BasePlugin

class DependentPlugin(BasePlugin):
    """A plugin that depends on other plugins."""
    
    @property
    def name(self):
        return "dependent_plugin"
    
    @property
    def description(self):
        return "A plugin that depends on other plugins"
    
    @property
    def dependencies(self):
        """Specify plugin dependencies."""
        return ["text_formatter", "calculator"]
    
    def process(self, *args, **kwargs):
        """Process with dependencies."""
        # This plugin can assume that text_formatter and calculator are available
        return "Processed with dependencies"
```

## Plugin API Summary

LlamaForge plugin registries:

```python
# Preprocessing plugins
from llamaforge.plugins.preprocessing import register_preprocessor, get_preprocessor, list_preprocessors

# Postprocessing plugins
from llamaforge.plugins.postprocessing import register_postprocessor, get_postprocessor, list_postprocessors

# Tool plugins
from llamaforge.plugins.tools import register_tool, get_tool, list_tools

# Command plugins
from llamaforge.plugins.commands import register_command, get_command, list_commands
```

Plugin management in LlamaForge:

```python
forge = LlamaForge()

# Plugin listing and status
plugins = forge.list_plugins()
enabled = forge.list_enabled_plugins()

# Enable/disable plugins
forge.enable_plugin("text_formatter")
forge.disable_plugin("calculator")
status = forge.is_plugin_enabled("text_formatter")

# Plugin configuration
forge.set_plugin_config("text_formatter", {"trim_whitespace": True})
config = forge.get_plugin_config("text_formatter")
all_configs = forge.get_all_plugin_configs()

# Plugin execution (mainly for command plugins)
result = forge.run_command("benchmark", {"model": "llama-2-7b"})
``` 