// Copyright (c) Mito
import React, { useEffect, useState } from 'react';
import TextButton from '../../elements/TextButton';
import DefaultTaskpane from '../DefaultTaskpane/DefaultTaskpane';
import DefaultTaskpaneBody from '../DefaultTaskpane/DefaultTaskpaneBody';
import DefaultTaskpaneFooter from '../DefaultTaskpane/DefaultTaskpaneFooter';
import DefaultTaskpaneHeader from '../DefaultTaskpane/DefaultTaskpaneHeader';
import FileBrowser from './FileBrowser';
import { getElementsToDisplay, getFileEnding, getImportButtonStatus } from './importUtils';
import XLSXImport from './XLSXImport';
function ImportTaskpane(props) {
    const [importState, setImportState] = useState({
        pathContents: {
            path_parts: props.currPathParts,
            elements: []
        },
        sort: 'last_modified_descending',
        searchString: '',
        selectedElementIndex: -1,
        loadingFolder: false,
        loadingImport: false
    });
    // If the file being imported is an XLSX, we need additional configuration
    // and so we use an import wizard for help
    const [fileForImportWizard, setFileForImportWizard] = useState(undefined);
    // It is very convenient to have the full joined path for the file, so this state and the 
    // effect below it make it possible to access this easily
    const [fullFileNameForImportWizard, setFullFileNameForImportWizard] = useState(undefined);
    useEffect(() => {
        const getFullFileNameForImportWizard = async (fileForImportWizard) => {
            const finalPath = [...props.currPathParts];
            finalPath.push(fileForImportWizard);
            const fullFileName = await props.mitoAPI.getPathJoined(finalPath);
            setFullFileNameForImportWizard(fullFileName);
        };
        if (fileForImportWizard !== undefined) {
            void getFullFileNameForImportWizard(fileForImportWizard);
        }
        else {
            setFullFileNameForImportWizard(undefined);
        }
    }, [fileForImportWizard]);
    // We make sure to get the elements that are displayed and use the index on that to get the correct element
    const selectedElement = getElementsToDisplay(importState)[importState.selectedElementIndex];
    /*
        Any time the current path changes, we update
        the files that are displayed
    */
    useEffect(() => {
        // When the current path changes, we reload the path contents
        void loadPathContents(props.currPathParts);
        // We also unselect anything that might be selected
        setImportState(prevImportState => {
            return Object.assign(Object.assign({}, prevImportState), { selectedElementIndex: -1 });
        });
        // Log how long the path is
        void props.mitoAPI.log('curr_path_changed', { 'path_parts_length': props.currPathParts.length });
    }, [props.currPathParts]);
    /*
        Any time the selected element changes we log the file
        ending (or none, if it has none).
    */
    useEffect(() => {
        let selectedElementName = '';
        if (selectedElement === undefined) {
            selectedElementName = 'undefined';
        }
        else if (selectedElement.isDirectory) {
            selectedElementName = 'directory';
        }
        else {
            const fileEnding = getFileEnding(selectedElement.name);
            if (fileEnding !== undefined) {
                selectedElementName = fileEnding;
            }
            else {
                selectedElementName = 'No File Ending';
            }
        }
        void props.mitoAPI.log('selected_element_changed', { 'selected_element': selectedElementName });
    }, [selectedElement]);
    // Loads the path data from the API and sets it for the file browser
    async function loadPathContents(currPathParts) {
        setImportState(prevImportState => {
            return Object.assign(Object.assign({}, prevImportState), { loadingFolder: true });
        });
        const _pathContents = await props.mitoAPI.getPathContents(currPathParts);
        if (_pathContents) {
            setImportState(prevImportState => {
                return Object.assign(Object.assign({}, prevImportState), { pathContents: _pathContents, loadingFolder: false });
            });
        }
        else {
            setImportState(prevImportState => {
                return Object.assign(Object.assign({}, prevImportState), { loadingFolder: false });
            });
        }
    }
    async function importElement(element) {
        const importButtonStatus = getImportButtonStatus(element, props.userProfile.excelImportEnabled, importState.loadingImport);
        // Quit early if the selected thing is not importable, or if there
        // is nothing even selected
        if (importButtonStatus.disabled || element === undefined) {
            return;
        }
        if (!(element === null || element === void 0 ? void 0 : element.isDirectory) && (element === null || element === void 0 ? void 0 : element.name.toLowerCase().endsWith('.xlsx'))) {
            setFileForImportWizard(element.name);
            return;
        }
        // Do the actual import
        const finalPath = [...props.currPathParts];
        finalPath.push(element.name);
        const joinedPath = await props.mitoAPI.getPathJoined(finalPath);
        if (joinedPath === undefined) {
            return;
        }
        // And then actually import it
        setImportState(prevImportState => {
            return Object.assign(Object.assign({}, prevImportState), { loadingImport: true });
        });
        await props.mitoAPI.editSimpleImport([joinedPath]);
        setImportState(prevImportState => {
            return Object.assign(Object.assign({}, prevImportState), { loadingImport: false });
        });
    }
    const importButtonStatus = getImportButtonStatus(selectedElement, props.userProfile.excelImportEnabled, importState.loadingImport);
    if (fullFileNameForImportWizard !== undefined) {
        return (React.createElement(XLSXImport, { mitoAPI: props.mitoAPI, analysisData: props.analysisData, fileName: fullFileNameForImportWizard, fileForImportWizard: fileForImportWizard, setFileForImportWizard: setFileForImportWizard, setImportState: setImportState, setUIState: props.setUIState, importState: importState }));
    }
    return (React.createElement(DefaultTaskpane, null,
        React.createElement(DefaultTaskpaneHeader, { header: 'Import Files', setUIState: props.setUIState }),
        React.createElement(DefaultTaskpaneBody, { noScroll: true },
            React.createElement(FileBrowser, { mitoAPI: props.mitoAPI, setCurrPathParts: props.setCurrPathParts, importState: importState, setImportState: setImportState, importElement: importElement, userProfile: props.userProfile })),
        React.createElement(DefaultTaskpaneFooter, null,
            React.createElement(TextButton, { variant: 'dark', width: 'block', onClick: () => {
                    void importElement(selectedElement);
                }, disabled: importButtonStatus.disabled }, importButtonStatus.buttonText))));
}
export default ImportTaskpane;
//# sourceMappingURL=ImportTaskpane.js.map